# -*- coding: utf-8 -*-

import logging

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Debug, GitWithoutLfsProcessing
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.managers.config import ConfigManager
from sandbox.projects.sandbox_ci.managers.errors.merge_queue import MergeQueueError, MergeQueueErrorNames, MergeQueueErrorParser
from sandbox.sandboxsdk.errors import SandboxSubprocessError


class MergeQueueManager(object):
    def __init__(self, task):
        self._task = task

    def _get_context_prop(self, prop):
        current_value = getattr(self._task.Context, prop, ctm.NotExists)

        if current_value is not ctm.NotExists:
            return current_value

        return getattr(getattr(self._task, 'parent', {}).Context, prop, ctm.NotExists)

    def get_project_config(self, project_name=None):
        if not project_name:
            project_name = self._task.Parameters.project_github_repo

        properties = ConfigManager.get_properties()
        options = dict(project_name=project_name, build_context='merge-queue')

        return ConfigManager.get_project_conf(properties, options)

    def acquire_semaphore(self, semaphore_name):
        exec_limit = 1
        weight = 1

        self._task.Requirements.semaphores = ctt.Semaphores(
            [(semaphore_name, weight, exec_limit)],
            ctt.Status.Group.BREAK | ctt.Status.Group.FINISH,
        )

    def is_merge_conflict(self, error_name):
        return error_name == MergeQueueErrorNames.MERGE_CONFLICT or \
               error_name == MergeQueueErrorNames.ARC_MERGE_CONFLICT

    def merge_error_handler(self, e):
        """
        Обработчик ошибок. Выводит информацию об ошибке в описание задачи.

        :param e: Экземпляр ошибки
        :type e: Exception
        :raises: TaskFailure
        """
        logging.debug('Parsing known errors from "{}"'.format(e))

        if isinstance(e, MergeQueueError):
            error = e
        elif hasattr(e, 'stderr_full_path') and e.stderr_full_path is not None:
            with open(e.stderr_full_path, 'r') as f:
                stderr = f.read()

            error = MergeQueueErrorParser.parse_from_stderr(stderr)
        else:
            error = MergeQueueErrorParser.parse_from_exception(e)

        if error:
            logging.debug('Found error: {}'.format(error))

            self._task.set_info(error.description, do_escape=False)
            self._task.Context.merge_queue_error_name = error.name
            self._task.Context.merge_queue_error_code = getattr(error, "code", None)

            raise TaskFailure(error.feedback)

    def clone_project(self):
        git_url = self.git_parameters['url']
        git_base_ref = self.git_parameters['base_ref']

        logging.debug('Cloning repository from {git_url}#{ref} to {dir}'.format(
            git_url=git_url,
            ref=git_base_ref,
            dir=self._task.project_git_cache_dir,
        ))

        workcopy_params = {
            'dump-hashes': True,
            'cache-dir': self._task.project_git_cache_dir,
            'base-ref': git_base_ref,
        }

        with Debug('*'), self._task.vault.ssh_key(self._task.Parameters.ssh_key_vault_name), GitRetryWrapper(), GitWithoutLfsProcessing(git_url):
            try:
                hashes = self._task.scripts.run_js(
                    'script/prepare-working-copy.js',
                    git_url,
                    self._task.project_dir,
                    workcopy_params,
                )

                commit = self._task.project_git_exec('rev-parse', git_base_ref)

                self._task.project_git_exec('checkout', '-f', commit)
            except SandboxSubprocessError as e:
                self.merge_error_handler(e)

                raise

        profile = hashes.get('profile', {})
        prepare_wc_metrics = profile.get('metrics', {})
        git_cache_exist = profile.get('gitCacheExist', False)

        if prepare_wc_metrics:
            for tag, duration in prepare_wc_metrics.items():
                self._task.profiler.register_action(tag, tag, duration)

        self._task.cache_profiler.register_action('git_cache_exist', 'git_repo', reused=git_cache_exist)

    def arc_clone_project(self):
        try:
            project_config = self.get_project_config()

            if project_config.get('verbose_apply_tool', False):
                env.export({'VERBOSE_APPLY_TOOL': '1'})
        except Exception as error:
            logging.exception('Cannot get project config: {}'.format(error))

        git_url = self.git_parameters['url']
        commit = self._task.Context.project_git_base_commit
        workcopy_params = {
            'dump-hashes': True,
            'base-ref': self.git_parameters['base_ref'],
            'use-arc': True,
            'commit': commit,
            'merge-ref': self._task.Context.project_git_head_ref
        }
        # ключ нужен, потому что sha коммита в arc берется из git (по гитовому sha)
        with Debug('*'), self._task.vault.ssh_key(self._task.Parameters.ssh_key_vault_name):
            hashes = self._task.scripts.run_js(
                'script/prepare-working-copy.js',
                git_url,
                self._task.working_path('mirror'),
                workcopy_params,
                log_prefix='prepare-working-copy-arc',
            )
            logging.debug('Arc clone info: {}'.format(hashes))
            self._task.Context.arc_tree_hash = hashes.get('merged', hashes['base'])['treeHash']

    @property
    def git_parameters(self):
        return dict(
            url=self._get_context_prop('project_git_url'),
            base_ref=self._get_context_prop('project_git_base_ref')
        )
