# -*- coding: utf-8 -*-

import logging
import uuid

from sandbox import sdk2

from sandbox.common.utils import get_task_link
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.common.types import misc as ctm
from sandbox.common.types.misc import NotExists

from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.utils import github
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.utils import list_utils


class TaskDeclaration(object):
    def __init__(self, task_type, wait_tasks, parameters):
        self.id = int(str(uuid.uuid4().int)[:4] + str(uuid.uuid4().int)[:4])
        self.task = task_type
        self.wait_tasks = wait_tasks  # Идентификаторы тасок, а не их инстансы
        self.parameters = parameters


class MetaTaskManager(object):
    def __init__(self, task):
        """
        Менеджер задач

        :param task: инстанс задачи
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        self.task = task

    DONT_WAIT_TAG = 'DONT_WAIT'

    @property
    def subtasks(self):
        tasks = list(self.task.find().limit(0))

        if self.task.Context.subtasks is not NotExists:
            return filter(lambda task: task.id in self.task.Context.subtasks, tasks)

        return tasks

    @property
    def failed_subtasks(self):
        return self.filter_failed(self.subtasks)

    @property
    def not_finished_subtasks(self):
        return self.filter_not_finished(self.subtasks)

    @property
    def failed_skipped_steps(self):
        """
        Возвращает пропущенные из-за ошибок проверки.

        :rtype: list of dict
        """
        return self.filter_failed_skipped_steps(self.task.Context.skipped_steps)

    def report_github_statuses(self, commit_sha, force):
        """
        Выставляет github-статусы текущей задачи для указанного коммита

        :param commit_sha: хеш коммита
        :type commit_sha: str
        :param force: отправить статус независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        """
        github_statuses = self.get_github_statuses()

        logging.debug('Reporting github statuses of {}:{}'.format(self.task, github_statuses))

        for github_status in github_statuses:
            self.task.github_statuses.safe_report_status(
                owner=self.task.Parameters.project_github_owner,
                repo=self.task.Parameters.project_github_repo,
                sha=commit_sha,
                context=github_status.get('context'),
                state=github_status.get('state'),
                description=github_status.get('description'),
                url=github_status.get('url'),
                force=True,
            )

    def report_arcanum_checks(self, review_request_id, diff_set_id, force):
        """
        Выставляет чеки в ревью-реквест текущей задачи и подзадачи для указанного дифф-сета

        :param review_request_id: номер ревью-реквеста
        :type review_request_id: int
        :param diff_set_id: id дифф-сета
        :type diff_set_id: int
        :param force: отправить статус независимо от параметров задачи и конфигурации в genisys
        :type force: bool
        """
        checks = self.get_github_statuses()

        logging.debug('Reporting review request {rr_id} diff set {diff_set_id} checks of {task}:{checks}'.format(
            rr_id=review_request_id,
            diff_set_id=diff_set_id,
            task=self.task,
            checks=checks,
        ))

        for check in checks:
            logging.debug('Reporting review request {rr_id} diff set {diff_set_id} check: {check}'.format(
                rr_id=review_request_id,
                diff_set_id=diff_set_id,
                check=check,
            ))

            self.task.arcanum_checks.safe_report_status(
                review_request_id=review_request_id,
                diff_set_id=diff_set_id,
                context=check.get('context'),
                state=check.get('state'),
                url=check.get('url'),
                description=check.get('description'),
                force=force,
            )

    def get_github_statuses(self):
        """
        Возвращает информацию о github-статусах подзадач и пропущенных проверок

        :rtype: list of dict
        """

        github_statuses = self.get_github_statuses_for_tasks(self.subtasks)
        github_statuses += self.get_github_statuses_for_skipped_steps(self.task.Context.skipped_steps)

        return github_statuses

    def get_github_statuses_for_tasks(self, tasks=[]):
        """
        Возвращает информацию о github-статусах задач

        :param tasks: инстансы SANDBOX_CI задач
        :type tasks: list of sandbox.projects.sandbox_ci.task.BaseTask.BaseTask
        :rtype: list of dict
        """
        github_statuses = []

        if tasks:
            for task in tasks:
                if hasattr(task, 'custom_github_statuses'):
                    github_statuses += task.custom_github_statuses
                else:
                    github_statuses.append(dict(
                        context=task.github_context,
                        url=get_task_link(task.id),
                        state=self.task.scp_feedback.convert_task_status_to_scp_state(task.status),
                    ))
            logging.debug('Got github statuses of {}: {}'.format(tasks, github_statuses))

        return github_statuses

    def get_github_statuses_for_skipped_steps(self, skipped_steps=[]):
        """
        Возвращает информацию о github-статусах пропущенных проверок

        :param skipped_steps: информация о пропущенных проверках
        :type skipped_steps: list of dict
        :rtype: list of dict
        """
        github_statuses = []

        if skipped_steps:
            for step in skipped_steps:
                github_statuses.append(dict(
                    context=step['github_context'],
                    state=step['state'],
                    description=step['description'],
                ))
            step_labels = ', '.join([step['label'] for step in skipped_steps])
            logging.debug('Got github statuses of {}: {}'.format(step_labels, github_statuses))

        return github_statuses

    @staticmethod
    def filter_failed(tasks):
        failure_statuses = (ctt.Status.FAILURE,) + tuple(ctt.Status.Group.BREAK)
        return filter(lambda task: task.status in failure_statuses, tasks)

    @staticmethod
    def filter_not_finished(tasks):
        finished_statuses = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        return filter(lambda task: task.status not in finished_statuses, tasks)

    def is_any_failed(self, tasks):
        return bool(self.filter_failed(tasks))

    def store_failed_tasks(self, tasks):
        self.task.Context.failed_tasks.extend(t.id for t in tasks)

    def store_not_finished_tasks(self, tasks):
        self.task.Context.not_finished_tasks.extend(t.id for t in tasks)

    def get_subtasks_artifacts_ids(self, limit=15):
        subtask_artifacts_ids = []
        subtask_resources = self.get_subtask_resources(
            subtask_ids=map(lambda subtask: subtask.id, self.subtasks),
            subtask_type=SANDBOX_CI_ARTIFACT,
            limit=limit * len(self.subtasks),
        )

        for resource in subtask_resources:
            logging.info('Resource #{} proxy url: {}'.format(resource['id'], resource['http']['proxy']))
            if resource['state'] != ctr.State.READY:
                msg = 'Resource #{} ({}) is not ready'.format(resource['id'], resource['type'])
                logging.info(msg)
            if resource.get('attributes', {}).get('public', False):
                subtask_artifacts_ids.append(resource['id'])

        return subtask_artifacts_ids

    def get_subtask_resources(self, subtask_ids, subtask_type, limit=15):
        """
        Возвращает список ресурсов, найденных для указанных параметров.

        :param subtask_ids: идентификаторы тасок, для которых необходимо получить ресурсы
        :type subtask_ids: list of int
        :param subtask_type: тип искомого ресурса
        :type subtask_type: sandbox.projects.sandbox_ci.SANDBOX_CI_ARTIFACT
        :param limit: количество искомых ресурсов
        :type limit: int
        :rtype: list of sandbox.sdk2.resource.Resource
        """
        return self.task.server.resource.read(
            task_id=subtask_ids,
            type=str(subtask_type),
            limit=limit,
        ).get('items', [])

    def create_subtask(self, task_type, wait_tasks=None, reusable=False, waitable=True, **parameters):
        """
        :param task_type: тип создаваемой таски
        :type task_type: () -> sdk2.Task
        :param wait_tasks: список тасок для ожидания таской
        :type wait_tasks: list of sdk2.Task
        :param reusable:
        :type reusable: bool
        :param waitable: опция для добавления тега `DONT_WAIT` в таск
        :type waitable: bool
        :param parameters: параметры для таски
        :type parameters: dict
        :return:
        """
        subtask_parameters = self.prepare_subtask_parameters(wait_tasks, reusable, waitable, **parameters)

        logging.debug('Creating {task_type} with parameters: {parameters}'.format(
            task_type=task_type,
            parameters=subtask_parameters,
        ))

        return self.create_task_instance(task_type, self.task, subtask_parameters)

    def create_task_instance(self, task_type, parent_task, task_parameters):
        tasks_resource = task_parameters.pop("tasks_archive_resource", None) or self.task.Requirements.tasks_resource
        task = task_type(parent_task, **task_parameters)

        if tasks_resource:
            task.Requirements.tasks_resource = tasks_resource
            task.save()

        return task

    def declare_subtask(self, task_type, wait_tasks=None, reusable=False, waitable=True, **parameters):
        """
        Возвращает представление таски, не создавая её инстанс.

        :param task_type: тип создаваемой таски
        :type task_type: () -> sandbox.sdk2.Task
        :param wait_tasks: список тасок для ожидания таской
        :type wait_tasks: list of sandbox.sdk2.Task
        :param reusable:
        :type reusable: bool
        :param waitable: опция для добавление тега `DONT_WAIT` в таск
        :type waitable: bool
        :param parameters: параметры таски
        :type parameters: dict
        :rtype: TaskDeclaration
        """
        logging.debug('Preparing task declaration for "{task}" with awaited tasks ({wait_tasks}) and parameters ({parameters})'.format(
            task=task_type,
            wait_tasks=wait_tasks,
            parameters=parameters,
        ))

        subtask_parameters = self.prepare_subtask_parameters(wait_tasks, reusable, waitable, **parameters)

        task = TaskDeclaration(
            task_type=task_type,
            wait_tasks=MetaTaskManager.get_tasks_ids(subtask_parameters['wait_tasks']),
            parameters=subtask_parameters,
        )

        logging.debug('Task declaration for "{task}" has id ({task_id}) and parameters: {parameters}'.format(
            task_id=task.id,
            task=task.task,
            parameters=task.parameters,
        ))

        return task

    def create_declared_subtask(self, task_declaration):
        """
        :param task_declaration: представление таски
        :type task_declaration: TaskDeclaration
        :rtype: dict
        """
        task_id = task_declaration.id
        task_type = str(task_declaration.task)

        logging.debug('Creating draft {task_type} with virtual id ({task_id}) from task representation'.format(
            task_type=task_type,
            task_id=task_id,
        ))

        try:
            task = self.create_task_instance(task_declaration.task, self.task, task_declaration.parameters)

            return dict(virtual_id=task_id, task=task)
        except Exception as e:
            logging.error('Cannot create draft for {task_type} with virtual id ({task_id}): {error}'.format(
                task_type=task_type,
                task_id=task_id,
                error=e,
            ))

            raise

    def update_subtasks_notifications(self, subtask_parameters):
        """
        Обновление настроек нотификаций для подзадач.
        :param parameters: параметры, которые будут использоваться при создании сабтаски
        :type parameters: dict
        """
        should_notify = self.task.should_notify_subtask_statuses(subtask_parameters) \
            if hasattr(self.task, 'should_notify_subtask_statuses') else True

        if not should_notify:
            logging.debug('Removing notification settings for subtask')
            subtask_parameters.update({'notifications': []})

    def prepare_subtask_parameters(self, wait_tasks=None, reusable=False, waitable=True, **parameters):
        """
        Возвращает параметры, которые будут использоваться при создании сабтаски.

        :param wait_tasks: список тасок, которые должны выполниться перед создаваемой таской
        :type wait_tasks: list of sdk2.Task
        :param reusable: управление возможность переиспользования таски из кэша
        :type reusable: bool
        :param waitable: отключение ожидания сабтаски её родительской задачей
        :type waitable: bool
        :param parameters: параметры, которые будут использоваться при создании сабтаски
        :type parameters: dict
        :return: dict
        """
        if isinstance(wait_tasks, list):
            wait_tasks = list_utils.flatten(wait_tasks)
            wait_tasks = filter(lambda task: task is not None, wait_tasks)

        reuse_task = reusable and self.task.project_conf.get('reuse_subtasks_cache', False)
        reuse_task = reuse_task and self.task.Parameters.reuse_subtasks_cache

        subtask_parameters = self.get_subtask_default_parameters()
        subtask_parameters.update(
            wait_tasks=self.get_tasks_ids(wait_tasks),
            reuse_task_cache=reuse_task,
            tags=subtask_parameters.get('tags', []) + parameters.get('additional_tags', []),
        )

        if hasattr(self.task.Parameters, 'build_id'):
            subtask_parameters.update(
                build_id=self.task.Parameters.build_id,
            )

        # FIXME(white): remove this when patch for empty client tags in SANDBOX-6008 is deployed
        if not parameters.get("overwrite_client_tags_flag"):
            parameters.pop("overwritten_client_tags", None)
        # В объекте parameters могут содержаться такие же ключи, как в параметрах выше, поэтому делаем update отдельно
        subtask_parameters.update(parameters)

        if not waitable:
            subtask_parameters['tags'] = subtask_parameters.get('tags', []) + [self.DONT_WAIT_TAG]

        self.update_subtasks_notifications(subtask_parameters)
        return subtask_parameters

    def start_subtasks(self, subtasks):
        """
        :param subtasks: список сабтасок
        :type subtasks: list of sandbox.sdk2.Task
        :rtype: list of sandbox.sdk2.Task
        """
        subtasks = filter(bool, subtasks)  # subtasks может содержать None для задач, которые нужно пропустить.
        subtasks_ids = map(int, subtasks)

        started_subtasks = flow.parallel(self.start_subtask, subtasks_ids)

        logging.info('Subtasks start result {}'.format(started_subtasks))

        for task in started_subtasks:
            if task['status'] != ctm.BatchResultStatus.SUCCESS:
                msg = 'Task #{} is not enqueued: {}'.format(task['id'], task['message'])
                self.task.Context.noncritical_errors.append(msg)
                logging.error(msg)

        return subtasks

    def start_subtask(self, subtask_id):
        """
        :param subtask: идентификатор сабтаски
        :type subtask: int
        :rtype: sandbox.sdk2.Task
        """
        assert type(subtask_id) is int, 'Subtask id must be integer'

        return self.task.server.batch.tasks.start.update(subtask_id)[0]

    def get_waitable_tasks_ids(self, tasks):
        """
        Возвращает таски у которых нет тега `DONT_WAIT`

        :param tasks: список тасок
        :type tasks: list of sdk2.Task
        :rtype: list of sdk2.Task
        """
        return [t.id for t in tasks if self.DONT_WAIT_TAG not in t.Parameters.tags]

    def get_non_waitable_tasks_ids(self, tasks):
        """
        Возвращает таски у которых есть тег `DONT_WAIT`

        :param tasks: список тасок
        :type tasks: list of sdk2.Task
        :rtype: list of sdk2.Task
        """
        return [t.id for t in tasks if self.DONT_WAIT_TAG in t.Parameters.tags]

    def create_independent_task(self, task_type, **parameters):
        """
        Создает независимую (не сабтаску) sdk2 таску с заданными параметрами

        :type task_type: () -> sdk2.Task
        :param parameters: параметры для задачи
        :return:
        """
        subtask_parameters = self.get_subtask_default_parameters()
        subtask_parameters.update(parameters)

        logging.debug('Creating {} with params: {}'.format(task_type, parameters))

        task = self.create_task_instance(task_type, None, subtask_parameters)

        return task.enqueue()

    def run_task(self, task_type_name, parameters, children=True):
        """
        Создаёт драфт задачи, обновляет указанными параметрами и запускает задачу.

        Для sdk2 задач используй `create_independent_task`
        :param task_type_name: название типа задачи
        :type task_type_name: str
        :param parameters: параметры как в REST API (должны быть сереализуемыми)
        :type parameters: dict
        :param children: создавать задачу как сабтаск или автономно
        :return: созданную задачу
        """
        logging.debug('Creating {task_type} draft with parameters: {parameters}'.format(
            task_type=task_type_name,
            parameters=parameters,
        ))
        task = self.task.server.task(dict(
            type=task_type_name,
            children=children,
            **parameters
        ))

        task_id = task['id']
        logging.debug('Created {task_type}#{task_id} draft: {task_obj}'.format(
            task_type=task_type_name,
            task_id=task_id,
            task_obj=task,
        ))

        logging.debug('Starting {task_type}#{task_id} draft'.format(
            task_type=task_type_name,
            task_id=task_id,
        ))
        self.task.server.batch.tasks.start.update([task_id])
        logging.debug('Started {task_type}#{task_id}'.format(task_type=task_type_name, task_id=task_id))

        return task

    def stop_tasks(self, tasks):
        stop_tasks_ids = [task.id for task in tasks]

        if len(stop_tasks_ids):
            logging.debug('Trying to stop tasks: {}'.format(stop_tasks_ids))
            self.task.server.batch.tasks.stop = map(int, stop_tasks_ids)  # PUT request

    def skip_step(self, github_context, description, label, reason='disabled'):
        """
        Отмечает указанный шаг, как пропущенный.

        Записывает информацию о пропущенном шаге в контекст задачи и в статистику реиспользования,
        выставляет github-статус.

        :param github_context: Контекст github статуса
        :type github_context: str
        :param description: Описание github статуса
        :type description: str
        :param label: Идентификатор шага, используется для записи статистики
        :type label: str
        :param reason: Идентификатор причины пропуска шага (disabled, not modified, error)
        :type reason: str
        """
        if reason != 'error':
            self.task.cache_profiler.hit(label, 'task', reason)

        # Можно оставить GH статусы, потому что у Арканума они аналогичные (error и success)
        state = github.GitHubStatus.ERROR if reason == 'error' else github.GitHubStatus.SUCCESS

        self.task.Context.skipped_steps.append(dict(
            label=label,
            reason=reason,
            github_context=github_context,
            description=description,
            state=state,
        ))

        self.task.notify_scp(
            context=github_context,
            description=description,
            state=state,
        )

    @staticmethod
    def filter_failed_skipped_steps(steps):
        """
        Возвращает только те шаги, которые были пропущены из-за ошибок.

        :param steps: пропущенные шаги
        :type steps: list of dict
        :rtype: list of dict
        """
        if not steps:
            return []

        return filter(lambda step: step['reason'] == 'error', steps)

    def get_subtask_default_parameters(self):
        param_names = (
            # system params
            '_container',
            'environ',
            'owner',
            'priority',
            'notifications',
            'tags',
            'tasks_archive_resource',

            'statface_host',
            'send_statistic',

            # build params
            'project_build_context',
            'external_config',

            # params for GitHub status
            'project_github_owner',
            'project_github_repo',
            'project_github_commit',
            'report_github_statuses',

            # params for git checkout in overlayfs mode
            'git_checkout_params',

            # params for caching using tree_hash
            'project_tree_hash',

            # see FEI-9390
            'is_release',
            'send_comment_to_searel',
            'send_comment_to_issue',

            # params for Arcanum status
            'report_arcanum_checks',
            'arcanum_review_request_id',
            'arcanum_diff_set_id',

            # see FEI-15592
            'html_reporter_use_sqlite',

            # node_js params
            'node_js_version',

            # arc params
            'use_arc',
            'path_in_arcadia',
            'arc_ref',
        )

        return {
            p: getattr(self.task.Parameters, p)
            for p in param_names
            if hasattr(self.task.Parameters, p)
        }

    @staticmethod
    def get_tasks_ids(tasks):
        if not tasks:
            return []

        return map(MetaTaskManager.get_task_id, tasks if type(tasks) is list else [tasks])

    @staticmethod
    def get_task_id(task):
        return sandboxapi.Sandbox._to_id(task)

    @staticmethod
    def is_sandbox_task(task):
        return isinstance(task, sdk2.Task)

    @staticmethod
    def is_task_declaration(task):
        return not MetaTaskManager.is_sandbox_task(task)

    @staticmethod
    def flatten_task_ids(task_ids, recurse=True):
        """
        Костыль, который уйдет в рамках эпика https://st.yandex-team.ru/FEI-6887
        По входящему списку из id тасок возвращает плоский список, состоящий из входящих id тасок и id всех их сабтасок

        :param task_ids: список из id тасок
        :type steps: list of str
        :rtype: list of str
        """
        if not isinstance(task_ids, list):
            task_ids = [task_ids]

        flattened = []

        for task_id in task_ids:
            flattened.append(task_id)
            task = sdk2.Task[task_id]
            if hasattr(task.Context, 'subtasks') and task.Context.subtasks:
                flattened += MetaTaskManager.flatten_task_ids(task.Context.subtasks) if recurse else task.Context.subtasks

        return flattened
