# -*- coding: utf-8 -*-

import logging
import subprocess

from functools import partial
from sandbox import sdk2

from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.dict_utils import defaults


class PluginsProfilerReportManager(object):
    def __init__(self, task):
        self._task = task
        self._report_type_to_find = 'hermione-plugins-profiler'
        self._output_report_type = self._report_type_to_find
        self._output_dirname = self._report_type_to_find

    def publish_merged_reports(self, task_ids, status, attrs={}):
        try:
            reports_data_paths = self.__get_reports_data_paths(task_ids)

            if not len(reports_data_paths):
                return

            self.__merge_reports(reports_data_paths)
            self.__publish_report(attrs, status)

            return True
        except Exception as e:
            logging.error('Cannot merge reports of plugins-profiler: %s', str(e))
            return False

    def __get_reports_data_paths(self, task_ids):
        return filter(bool, flow.parallel(partial(self.__find_report_data), task_ids))

    def __find_report_data(self, task_id):
        report_resource = self.__get_last_report(task_id)

        if not report_resource:
            return

        return self.__format_report_url(report_resource);

    def __merge_reports(self, paths):
        with Node(self._task.Parameters.node_js_version):
            command_args = [
                './node_modules/.bin/hermione-profiler-ui',
                'generate',
                '--output', self._output_dirname,
                '--plugin-profiles',
            ] + paths

            run_process(
                command_args,
                log_prefix='merge_plugins_profiler_reports',
                shell=False,
                work_dir=self._task.project_dir
            )

    def __publish_report(self, common_attrs, status):
        attrs = defaults(common_attrs, dict(
            status=status,
            resource_path=self._output_dirname,
            type=self._output_report_type,
            add_to_context=True,
            public=True,
            root_path='index.html',
            build_id=getattr(self._task.Parameters, 'build_id'),
            ttl=60
        ))

        self._task.artifacts.create_reports([attrs], parallel=True)

    def __get_last_report(self, task_id):
        return sdk2.Resource.find(task_id=task_id, attrs={'type': self._report_type_to_find}).first()

    def __format_report_url(self, resource):
        return "{}/plugins.json".format(resource.http_proxy);
