# -*- coding: utf-8 -*-

import datetime as dt
import itertools
import json
import logging
import operator

from sandbox.common.utils import singleton_property
from sandbox.common.types import task as ctt
from sandbox.common.types import misc as ctm

from sandbox.projects.sandbox_ci.managers.profiled_action_creator import ProfiledActionCreator

from sandbox.projects.sandbox_ci.utils.task_history import TaskHistory

STATFACE_REPORT_PREFIX = 'Yandex.Productivity/sandbox-ci-actions-profile'

FINISH_STATUSES = [
    ctt.Status.SUCCESS, ctt.Status.RELEASING, ctt.Status.NOT_RELEASED, ctt.Status.RELEASED,
    ctt.Status.FAILURE, ctt.Status.DELETED, ctt.Status.STOPPED
]
ACTION_PROFILE_HISTORY_STATUS_FILTER = [
    ctt.Status.DRAFT, ctt.Status.ENQUEUING, ctt.Status.EXECUTING, ctt.Status.ENQUEUED, ctt.Status.ASSIGNED,
    ctt.Status.WAIT_TASK, ctt.Status.WAIT_RES, ctt.Status.PREPARING, ctt.Status.STOPPING, ctt.Status.FINISHING
] + FINISH_STATUSES


def get_history_tracker(task):
    history = task.server.task[task.id].audit.read()
    # leave only interested statuses in history
    history_filtered = filter(lambda state: state.get('status') in ACTION_PROFILE_HISTORY_STATUS_FILTER, history)
    return TaskHistory(history_filtered)


class ProfilerManager(object):
    """Manager that helps to profile actions."""

    def __init__(self, task):
        """
        :param task: Task instance.
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        self.task = task

        if not task.Context.profiler_actions:
            task.Context.profiler_actions = []

    @singleton_property
    def actions(self):
        """
        Allows to calculate the working time of current action in executing task.

        Examples:
            .. code-block:: python
                with self.actions.profiled_action('Action description'):
                    # <code will be profiled>

            Where `profied_action` is stage name and can be any.

        :rtype: ProfiledActionCreator
        """
        return ProfiledActionCreator(self.task, ProfilerManager)

    def register_action(self, action_tag, description, duration, start=None, finish=None):
        """
        Saves action info into task context.

        :param action_tag: string with action tag
        :param description: string with action description
        :param start: date when action started
        :param finish: date when action finished
        :param duration: int action duration in seconds
        :rtype: None
        """
        self.register_task_step(self.task, action_tag, description, duration, start, finish)

    @classmethod
    def register_task_step(cls, task, action_tag, description, duration, start=None, finish=None):
        task_actions = cls._get_task_actions(task)
        task_actions.append({
            'tag': action_tag,
            'description': description,
            'start_utc_datetime': str(start),
            'finish_utc_datetime': str(finish),
            'duration_seconds': duration,
        })

    @classmethod
    def _get_task_actions(cls, task):
        return task.Context.profiler_actions

    def register_wait_actions(self):
        """
        Saves queue and waiting total time into profiler

        :rtype: None
        """
        self._register_queue_wait_action()
        self._register_wait_action('assigned_total_time', 'Wait in assigned', ctt.Status.ASSIGNED)
        self._register_wait_action('wait_res_total_time', 'Wait resources', ctt.Status.WAIT_RES)
        self._register_wait_action('wait_subtasks_total_time', 'Wait tasks', ctt.Status.WAIT_TASK)

    def _register_wait_action(self, action_tag, description, status):
        history_tracker = get_history_tracker(self.task)
        self.register_action(
            action_tag,
            description=description,
            duration=history_tracker.get_transition_times_from(status)
        )

    def _register_queue_wait_action(self):
        history_tracker = get_history_tracker(self.task)

        enqueuing_total_time = history_tracker.get_transition_times_from(ctt.Status.ENQUEUING)
        enqueued_total_time = history_tracker.get_transition_times_from(ctt.Status.ENQUEUED)

        queue_total_time = enqueuing_total_time + enqueued_total_time

        self.register_action(
            'queue_total_time',
            description='Wait in queue',
            duration=queue_total_time,
        )

    def send_report(self, report_path=None, report_name=None, custom_data={}):
        report = self.get_report(report_path)
        report_data = self.__build_report_data(custom_data)
        report_config = self.__build_report_config(report, report_data, report_name)

        report.upload_config(
            config=report_config,
            request_kwargs=dict(timeout=60),
            _request_timeout=60,
        )

        report.upload_data(
            scale='s',
            data=[report_data],
            request_kwargs=dict(timeout=60),
            _request_timeout=60,
        )

    def get_report(self, report_path=None):
        """
        Returns statface reporter

        :param report_path: path to report
        :type report_path: str
        :rtype: () -> statface_client.StatfaceReport
        """
        if not report_path:
            report_path = '{prefix}/{task_type}'.format(
                prefix=self.statface_report_prefix,
                task_type=self.task.type.name,
            )

        return self.task.statface.get_report(report_path)

    @property
    def statface_report_prefix(self):
        return STATFACE_REPORT_PREFIX

    def __build_report_data(self, custom_data):
        """
        Builds data for report

        :rtype: dict
        """
        groups = self.__get_action_groups()

        current_status = self.task.Context.current_status
        if current_status is ctm.NotExists:
            current_status = None

        get_duration = operator.itemgetter('duration_seconds')

        data = dict(
            task_id=self.task.id,
            status=current_status,
            fielddate=dt.datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            **custom_data
        )

        for measure, values in groups:
            measure_sum = 0

            for value in list(values):
                duration = get_duration(value)

                if type(duration) is int:
                    measure_sum += duration

            data[measure] = measure_sum

        # Information about cache reuse to builded report data
        cache_actions = self.task.cache_profiler.get_actions()

        # Statface supports values with the following types: boolean, number and string
        for action_key, action_values in cache_actions.items():
            data[action_key] = json.dumps(action_values)

        return data

    def __get_action_groups(self):
        """Builds data for report"""
        get_tag = operator.itemgetter('tag')
        task_actions = self._get_task_actions(self.task)

        return itertools.groupby(sorted(task_actions, key=get_tag), key=get_tag)

    def __build_report_config(self, report, data, report_name):
        """
        Builds report config by report data

        :param report: StatfaceReport instance
        :type report: () -> statface_client.StatfaceReport
        :param data: data for report
        :type data: dict
        :rtype: () -> statface_client.report.StatfaceReportConfig
        """
        from statface_client.report import StatfaceReportConfig

        title = report_name if report_name else '{} actions profile'.format(self.task.type.name)

        dimensions = [('fielddate', 'date')]

        return StatfaceReportConfig(
            title=title,
            dimensions=dimensions,
            measures=self.__build_report_measures(report, data, dimensions),
        )

    def __build_report_measures(self, report, data, dimensions):
        """
        Builds measures for report config

        :param report: StatfaceReport instance
        :type report: () -> statface_client.StatfaceReport
        :param data: data for report
        :type data: dict
        :param dimensions: list of dimensions
        :type dimensions: list of tuple
        :rtype: list
        """
        current_measures = []

        # Add to measures only fileds – not dimensions
        for key, value in data.items():
            dimensions_for_current_key = [dimension for dimension in dimensions if dimension[0] == key]
            if len(dimensions_for_current_key) == 0:
                current_measures.append((key, self.__get_measure_type(value)))

        current_measures.append(('task_id', 'number'))
        logging.debug('Build current measures for report: {}'.format(current_measures))

        if not report.is_config_uploaded:
            return current_measures

        report_measures = self.__get_report_measures(report)
        measures = dict(set(report_measures).union(set(current_measures))).items()
        logging.debug('Merged loaded measures with current measures: {}'.format(measures))

        return measures

    def __get_measure_type(self, measure_data):
        """
        Statface supports measures with the following types: boolean, number and string

        :param measure_data: data of measure
        :rtype: str
        """
        measure_data_type = type(measure_data)

        if measure_data_type is int:
            return 'number'
        elif measure_data is bool:
            return 'boolean'

        return 'string'

    def __get_report_measures(self, report):
        """
        Returns measures from report

        :param report: StatfaceReport instance
        :type report: () -> statface_client.StatfaceReport
        :rtype: list
        """
        measures = [m.items()[0] for m in report.config.to_dict()['user_config']['measures']]

        logging.debug('Loaded report measures: {}'.format(measures))

        return measures
