# -*- coding: utf-8 -*-
import json
import logging

from sandbox.common.utils import get_task_link, singleton_property
from sandbox.projects.sandbox_ci.utils.request import RETRY_METHODS, send_request

SANDBOX_CI_SERVICE = 'https://sandbox-ci.si.yandex-team.ru'


class StartrekClient(object):
    def __init__(self, oauth_token=None, protocol='https', host='st-api.yandex-team.ru', port=443, base_path='/v2',
                 web_interface='st.yandex-team.ru'):
        self._url = '{}://{}:{}{}'.format(protocol, host, port, base_path)
        self._token = oauth_token
        self._web_interface = '{}://{}'.format(protocol, web_interface)
        self._headers = {
            'Authorization': 'OAuth {}'.format(self._token),
            'Content-Type': 'application/json',
        }

    @staticmethod
    def get_issue_key(issue):
        """
        Получить ключ тикета

        :param issue: Тикет
        :type issue: dict
        :rtype str
        """
        return issue.get('key')

    def post(self, url, **kwargs):
        """
        :param url: ссылка на ручку
        :type url: str
        :param kwargs: передаваемые в requests параметры
        :type kwargs: dict
        :return:
        """
        methods = frozenset(list(RETRY_METHODS) + ['POST'])

        return send_request('post', url, headers=self._headers, retry_methods=methods, **kwargs)

    def search(self, query):
        """
        Поиск в Стартреке по запросу

        :param query: запрос
        :type query: str
        :rtype dict
        """
        url = '{}/issues/_search'.format(self._url)

        data = {'query': query}

        res = self.post(url, data=json.dumps(data))

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return res.json()

    def search_searel_by_summary(self, summary):
        """
        Поиск в Стартреке открытого SEAREL-тикета по заголовку

        :param summary: заголовок тикета
        :type summary: str
        :rtype str
        """
        logging.debug('looking for SEAREL\'s with summary "{}"'.format(summary))

        query = 'queue: searel status: !closed summary: "{}"'.format(summary)

        issues = self.search(query)
        logging.debug('found issues: {}'.format(issues))

        # У трекера нечёткий поиск, из-за чего через поиск могут вернуться нерелевантные тикеты, например:
        # при поиске 'summary: "serp/web4@exp/v1.22.0"' вернутся тикеты:
        #  - report-templates/serp/web4@exp/v1.22.0
        #  - report-templates/serp/web4@exp2/v1.22.0
        # Второй тикет от другого релиза, но он есть в выдаче. По этой причине проверяем, что summary
        # найденых тикетов содержит строку, по которой происходил поиск.
        # Подробнее тут: FEI-16440.
        current_release_issues = filter(lambda issue: summary in issue.get('summary'), issues)

        issue_keys = map(self.get_issue_key, current_release_issues)

        logging.debug('issues keys {issue_keys} with summary "{summary}"'.format(
            issue_keys=issue_keys,
            summary=summary,
        ))

        size = len(issue_keys)
        if size == 1:
            return issue_keys[0]

        if size == 0:
            raise Exception('No open issue with summary {}'.format(summary))

        raise Exception('Were found more than one issues with summary {}'.format(summary))

    def add_comment(self, issue_key, text):
        """
        Добавить комментарий к тикету

        :param issue_key: ключ тикета
        :type issue_key: str
        :param text: комментарий
        :type text: str
        :rtype: dict
        """
        url = '{base_url}/issues/{issue_key}/comments'.format(
            base_url=self._url,
            issue_key=issue_key,
        )

        data = {'text': text}

        res = self.post(url, data=json.dumps(data))

        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}

        return res.json()

    def add_remotelinks(self, issue_key, origin_key, origin, relationship='relates', notify_author=False, backlink=False):  # noqa
        """
        Создает связь с Sandbox таской в Стартреке. Подробнее в https://nda.ya.ru/3TcFN3

        :param issue_key: ключ тикета
        :type issue_key: str
        :param origin_key: номер таски
        :type origin_key: str
        :param origin: тип сервиса
        :type origin: str
        :param relationship: тип связи, @see https://nda.ya.ru/3TcFMn
        :type relationship: str
        :param notify_author: отправить почтовое уведомление автору изменения
        :type notify_author: bool
        :param backlink: бэклинковать в удаленном сервисе
        :type backlink: bool
        :rtype: dict
        """
        url = '{base_url}/issues/{issue_key}/remotelinks?notifyAuthor={notify_author}&backlink={backlink}'.format(
            base_url=self._url,
            issue_key=issue_key,
            notify_author=notify_author,
            backlink=backlink,
        )

        data = {
            'relationship': relationship,
            'key': origin_key,
            'origin': origin,
        }

        res = self.post(url, data=json.dumps(data))

        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}

        return res.json()

    def get_issue_url(self, issue_key):
        return '{web_interface}/{issue_key}/'.format(
            web_interface=self._web_interface,
            issue_key=issue_key,
        )


class ReleaseManager(object):
    def __init__(self, task):
        """
        :param task: инстанс таски
        :type task: sandbox.projects.sandbox_ci.task.BaseTask
        """
        self.task = task

    @singleton_property
    def startrek_client(self):
        startrek_token = self.task.vault.read('robot-serp-bot_startrek_token')

        return StartrekClient(oauth_token=startrek_token)

    def get_searel_issue_key_by_summary_safe(self, summary):
        """
        :param summary: заголовок SEAREL тикета
        :type repo: summary
        :return: ключ SEAREL-тикета
        :rtype: str or None
        """
        try:
            return self.startrek_client.search_searel_by_summary(summary)
        except Exception as e:
            logging.exception('SEAREL issue search failed with: {}'.format(e.message))

    def add_status_comment(self, issue_key, status, **kwargs):
        """
        Пишет в релизный тикет в Стартреке комментарий о статусе текущей задачи.

        :param issue_key: ключ тикета
        :type issue_key: str
        :param status: статус задачи
        :type status: str
        :param kwargs: опции для кастомизации статуса
        :type kwargs: dict
        """
        try:
            is_success = status == 'SUCCESS'

            task_id = self.task.id
            task_url = get_task_link(task_id)

            comment = [
                u'{icon} Сборка (({task_url} {task_name})) завершилась {task_status}.',
                '{custom_text}',
                '(({task_report_url} {task_report_name})).',
                '' if is_success else u'(({task_restart_url} Перезапустить)).',
            ]

            placeholders = dict(
                icon=u'✅' if is_success else u'❗',
                task_url=task_url,
                task_name=self.task.github_context,
                task_status=u'!!(green)успешно!!' if is_success else u'с !!ошибкой!!',
                task_report_name=u'Отчёт',
                task_report_url='{}#checks-reports'.format(task_url),
                task_restart_url='{service}/clone-task?sandboxTaskId={task_id}'.format(
                    service=SANDBOX_CI_SERVICE,
                    task_id=task_id,
                ),
                custom_text='',
            )

            placeholders.update(kwargs)

            text = '\n'.join(comment).format(**placeholders)

            self.startrek_client.add_comment(issue_key=issue_key, text=text)
        except Exception as e:
            logging.exception('error while adding status comment: {}'.format(e.message))

            self.task.set_info(
                'Something went wrong while commenting current task status to Startrek.'
                ' See logs for more details.',
            )

    def add_task_commencement_comment(self, issue_key, **kwargs):
        """
        Пишет в релизный тикет в Стартреке комментарий о начале выполнения задачи.

        :param issue_key: ключ тикета
        :type issue_key: str
        :param kwargs: опции для кастомизации статуса
        :type kwargs: dict
        """
        try:
            task_id = self.task.id
            task_url = get_task_link(task_id)

            comment = [
                u'{icon} Сборка (({task_url} {task_name})) запущена.',
                '{custom_text}',
            ]

            placeholders = dict(
                icon=u'⌛',
                task_url=task_url,
                task_name=self.task.github_context,
                custom_text='',
            )

            placeholders.update(kwargs)

            text = '\n'.join(comment).format(**placeholders)
            self.startrek_client.add_comment(issue_key=issue_key, text=text)
        except Exception as e:
            logging.exception('Error while adding executing comment: {}'.format(e.message))
            self.task.set_info(
                'Something went wrong while commenting current task to Startrek.'
                ' See logs for more details.',
            )

    def add_issue_link_to_info(self, issue_key):
        """
        Записывает в инфо задачи ссылку на тикет

        :param issue_key: ключ тикета
        :type issue_key: str
        """
        try:
            issue_url = self.startrek_client.get_issue_url(issue_key=issue_key)
            issue_info = 'Release ticket: <a href="{issue_url}">{issue_key}</a>'
            self.task.set_info(issue_info.format(issue_url=issue_url, issue_key=issue_key), do_escape=False)
        except Exception as e:
            logging.exception('Error while adding release issue link to info: {}'.format(e.message))
            self.task.set_info(
                'Something went wrong while adding startrek release ticket link to info.'
                ' See logs for more details.',
            )

    def add_task_remotelink(self, issue_key):
        """
        Добавляет в связях тикета ссылку на таск

        :param issue_key: ключ тикета
        :type issue_key: str
        """
        try:
            self.startrek_client.add_remotelinks(
                issue_key=issue_key,
                origin_key=self.task.id,
                origin='ru.yandex.sandbox',
            )
        except Exception as e:
            logging.exception('error while adding remote link: {}'.format(e.message))

            self.task.set_info(
                'Something went wrong while linking current task to SEAREL ticket.'
                ' See logs for more details.',
            )
