# -*- coding: utf-8 -*-

import datetime as dt
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr


class Reports(object):
    """
    Reports collection.
    """

    def __init__(self, task):
        """
        :param task: Sandbox task instance.
        """
        self.task = task

    @staticmethod
    def status_badge(status):
        """
        Return HTML to display specified status.
        :param status: Status to display.
        :return: HTML string.
        """
        status_badge = {
            ctt.Status.SUCCESS: '<span class="status status_success">success</span>',
            ctt.Status.FAILURE: '<span class="status status_failure">failure</span>',
            ctr.State.BROKEN: '<span class="status status_broken">broken</span>',
        }

        return status_badge[status] if status in status_badge else '<span class="status">wait</span>'

    @staticmethod
    def description(url, report_type, root_path, report_description):
        url = '{}/{}'.format(url, root_path) if root_path else url

        description = '<a href="{url}" target="_blank">{text}</a>'.format(url=url, text=report_type)
        return '{} {}'.format(description, report_description) if report_description else description

    @staticmethod
    def wiki_description(url, report_type, root_path, report_description):
        url = "{}/{}".format(url, root_path) if root_path else url

        description = "(({url} {text}))".format(url=url, text=report_type)
        return "{} {}".format(description, report_description) if report_description else description

    def build_audit_link(self):
        if not hasattr(self.task.Parameters, 'build_id'):
            return {}

        url = 'https://charts.yandex-team.ru/preview/5tds2kjj5cf42?buildId={}'.format(self.task.Parameters.build_id)
        link = '<a href="{}" target="_blank">Build audit</a>'.format(url)
        return {'<h3>Task flow</h3>': link}

    def reports_artifacts(self, report_ids):
        """
        Table with links to the reports.
        :param report_ids: Iterable of resource identifiers with reports. Resources should have `type`, `status`
            and `root_path` (could be empty) attributes.
        :return: Report object.
        """
        return {'<h3 id="checks-reports">Checks reports</h3>': self.reports(report_ids)}

    def reports(self, report_ids):
        reports_data = self._get_reports_data(report_ids)

        return [self._report_data_to_report(**data) for data in reports_data]

    def wiki_report_artifacts(self, report_ids):
        """
        Text with wiki links to the reports and report statuses.
        :param report_ids: Iterable of resource identifiers with reports. Resources should have `type`, `status`
                and `root_path` (could be empty) attributes.
        :return: String.
        """
        reports_data = self._get_reports_data(report_ids)
        reports = [self._report_data_to_wiki_report(**data) for data in reports_data]
        return "Checks reports\n{}".format("".join(reports))

    def _get_reports_data(self, report_ids):
        reports_data = []
        for id in report_ids:
            resource = self.task.server.resource[id].read()

            status = resource['attributes']['status']
            # see https://st.yandex-team.ru/FEI-4998
            if resource['state'] == ctr.State.BROKEN:
                status = ctr.State.BROKEN

            reports_data.append({
                'url': resource['http']['proxy'],
                'report_type': resource['attributes']['type'],
                'status': status,
                'root_path': resource['attributes'].get('root_path', None),
                'description': resource['attributes'].get('report_description', None),
            })

        reports_data.sort(key=lambda d: d['report_type'])
        return reports_data

    def _report_data_to_report(self, status, url, report_type, root_path, description):
        return {
            'Status': self.status_badge(status),
            'Report': self.description(url, report_type, root_path, description),
        }

    def _report_data_to_wiki_report(self, status, url, report_type, root_path, description):
        report_line = "**!!({status_color}){status}!!** {report_description}\n"
        status_color = "green" if status == ctt.Status.SUCCESS else "red"
        return report_line.format(
            status_color=status_color,
            status=status,
            report_description=self.wiki_description(url, report_type, root_path, description),
        )

    @staticmethod
    def actions_profile(profile_info):
        """
        Table with task actions and their durations.
        :param profile_info: Iterable of profile info data. Each item should be a dict with
            `description` and `duration_seconds` keys.
        :return: Report object.
        """
        actions = [
            {
                'Action': data['description'],
                'Duration': str(time_duration(seconds=data['duration_seconds']))
            }
            for data in profile_info
        ]
        return {'<h3>Actions profile</h3>': actions}


class time_duration(dt.timedelta):
    """
    Helper type based on `datetime.timedelta` to output time duration in hours:mins:seconds format.
    """
    @property
    def hours(self):
        return self.seconds // 3600

    @property
    def mins(self):
        return (self.seconds % 3600) // 60

    @property
    def secs(self):
        return self.seconds % 60

    def __str__(self):
        return '{:0>2}:{:0>2}:{:0>2}'.format(self.hours, self.mins, self.secs)
