# -*- coding: utf-8 -*-

import os
import logging

from sandbox.common.errors import TaskError

import sandbox.projects.sandbox_ci.utils.flow as flow
import sandbox.projects.sandbox_ci.utils.static as static

# Документация по boto3:
# https://boto3.readthedocs.io/en/latest/reference/services/s3.html
# Что реализовано из S3 API можно посмотреть в таблице совместимости с Amazon S3 API:
# https://wiki.yandex-team.ru/mds/s3-api/s3-compatibility-list

ALLOWED_MIMETYPES = {
    '.js': 'application/javascript',
    '.svg': 'image/svg+xml',
    '.ttf': 'application/x-font-ttf',
    '.woff': 'application/font-woff',
    '.woff2': 'application/font-woff2',
    '.eot': 'application/vnd.ms-fontobject',
    '.gif': 'image/gif',
    '.png': 'image/png',
    '.jpg': 'image/jpeg',
    '.jpeg': 'image/jpeg',
    '.ico': 'image/x-icon',
    '.css': 'text/css',
    '.html': 'text/html',
    '.xml': 'application/xml',
    '.crx': 'application/x-chrome-extension',
    '.map': 'application/json',
    '.mp3': 'audio/mpeg',
    '.mp4': 'video/mp4',
    '.json': 'application/json',
    '.safariextz': 'application/octet-stream',
    '.vtt': 'text/vtt',
    '.swf': 'application/x-shockwave-flash',
}


def detect_content_type(path):
    """
    :param path:
    :type path: str
    :return: str
    """
    original_path = path

    # see FEI-11099
    if original_path.endswith('.br') or original_path.endswith('.gz'):
        original_path = original_path[:-3]

    # see INFRADUTY-8124
    (filename, ext) = os.path.splitext(original_path)

    if ext == '':
        raise TaskError('Can not detect extension for file {}'.format(path))

    mime_type = ALLOWED_MIMETYPES.get(ext)
    if mime_type is None:
        raise TaskError('Can not detect mime type for file {}'.format(path))

    return mime_type


class S3Manager(object):
    """Менеджер для работы с S3 API MDS"""

    def __init__(self, access_key_id, secret_access_key, bucket, endpoint='https://s3.mds.yandex.net'):
        """
        :param access_key_id:
        :type access_key_id: str
        :param secret_access_key:
        :type secret_access_key: str
        :param bucket: имя бакета в S3
        :type bucket: str
        :param endpoint: точка входа в S3 MDS, по умолчанию продакшен
        :type endpoint: str
        """
        import boto3
        from botocore.config import Config

        logging.getLogger('botocore').setLevel(logging.INFO)
        logging.getLogger('s3transfer').setLevel(logging.INFO)

        self._bucket = bucket
        session = boto3.session.Session(
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key,
        )
        self.client = session.client(
            service_name='s3',
            endpoint_url=endpoint,
            config=Config(
                parameter_validation=False,
                # https://botocore.readthedocs.io/en/latest/reference/config.html
                # speed up (default 10)
                max_pool_connections=512
            )
        )
        logging.info('S3 connected to endpoint {}'.format(endpoint))

    @property
    def bucket(self):
        return self._bucket

    @bucket.setter
    def bucket(self, name):
        self._bucket = name

    def create_bucket(self, name):
        """
        Создаёт новый бакет
        :param name: имя бакета
        :return: dict
        """
        return self.client.create_bucket(Bucket=name)

    def upload_dir(self, dir_path, key_prefix='', recursive_update=True):
        """
        Загружает содержимое директории в бакет
        :param dir_path: str путь до директории
        :param key_prefix: str префикc для всех объектов. Используется как namespace.
        :param recursive_update: аплоадить вложенные объекты, не проверяя существуют ли объекты по таким ключам.
        """
        upload_file = self.upload_file

        def _upload_file(name):
            path = os.path.join(dir_path, name)
            key = os.path.join(key_prefix, name)
            upload_file(path, key, update=recursive_update)

        def _get_file_names(dir_name):
            prefix_len = len(dir_name) + 1
            ret = list()
            for r, d, f in os.walk(dir_name):
                for name in f:
                    ret.append(os.path.join(r, name)[prefix_len:])

            logging.debug('Files listing: {}'.format(ret))

            return ret

        names = _get_file_names(dir_path)

        flow.parallel(_upload_file, names, 256)

    def upload_file(self, path, key, update=False):
        """
        Загружает файл в бакет и размещает его по указанному ключу
        :param path: str путь к файлу, который загружаем
        :param key: str ключ, по которому размещаем файл в бакете
        :param update: аплоадить, не проверяя есть ли уже объект по такому ключу
        :return:
        """
        if not update and self.has_object(key):
            logging.debug('Key {} is already used'.format(key))
            return

        content_type = detect_content_type(path)

        logging.info('Uploading path: {}, bucket: {}, key: {}, content-type: {}'.format(
            path,
            self.bucket,
            key,
            content_type,
        ))

        self.client.upload_file(
            path,
            self.bucket,
            key,
            ExtraArgs={'ContentType': content_type}
        )

    def has_object(self, key):
        """
        Проверяет есть ли объект по ключу
        :param key:
        :return: boolean
        """

        from botocore.errorfactory import ClientError

        ret = False
        try:
            res = self.client.head_object(Bucket=self.bucket, Key=key)
            ret = res['ResponseMetadata']['HTTPStatusCode'] == 200
        except ClientError:
            pass

        return ret

    def delete_object(self, key):
        """
        Удаляет объект и освобождает ключ
        :param key: str
        :return: dict
        """
        return self.client.delete_object(Bucket=self.bucket, Key=key)

    def get_object(self, key):
        """
        Достаёт объект по ключу
        :param key:
        :return: dict
        """
        return self.client.get_object(Bucket=self.bucket, Key=key)


def upload_dir(freeze_path, bucket_name, key_prefix, key_id, access_key, should_compress=False):
    try:
        if should_compress:
            logging.info('Compressing static: {}'.format(freeze_path))

            static.compress_static(freeze_path)

        logging.info('Uploading static to bucket "{}", key prefix is "{}"'.format(bucket_name, key_prefix))

        S3Manager(key_id, access_key, bucket_name).upload_dir(freeze_path, key_prefix)

        return True
    except Exception as e:
        logging.exception('Error while uploading static: {}'.format(e))

        return False
