# -*- coding: utf-8 -*-

import os
import logging
import json
import subprocess
import tarfile
import hashlib

from io import StringIO

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import check_process_return_code

from sandbox.projects.sandbox_ci.resources import SandboxCiScripts
from sandbox.projects.sandbox_ci.utils.git import git_clone
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.sandbox_ci.utils.process import run_process, format_args_for_run_process
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node

SCRIPTS_BRANCH_NAME = 'trunk'


def file_stem(file_path):
    """
    Returns basename without extension

    :param file_path: str
    """
    basename = os.path.basename(file_path)
    return os.path.splitext(basename)[0]


class ScriptsManager(object):
    """
    Manager that helps to work with scripts from github repository.

    Supports Node.js scripts.
    """

    def __init__(self, task, scripts_dir):
        """
        :param task: Task instance.
        :param scripts_dir: Scripts checkout directory
        """
        self.task = task
        self.scripts_dir = scripts_dir

    @property
    def dir(self):
        """
        Path to directory with scripts.
        """
        return self.task.working_path(self.scripts_dir)

    def path(self, *args):
        """
        Builds path to files in directory with scripts.

        If not pass arguments returns path to directory with scripts.
        """

        return self.dir.joinpath(*args)

    def fetch_archive(self, url=None):
        """
        Getting scripts as archive by URL.

        :param url: Archive url, default value is task property scripts_archive_url
        :type url: str
        """
        if url is None:
            url = self.task.Parameters.scripts_archive_url

        scripts_dir = str(self.dir)

        # Temporary directory + basename of archive URL
        tmp_dir = os.path.join(os.environ['TMP'], hashlib.md5(url).hexdigest(), os.path.basename(url))

        logging.debug('Save scripts from {source} to {dest}'.format(source=url, dest=tmp_dir))
        response = send_request('get', url, stream=True)

        with tarfile.open(mode='r|gz', fileobj=StringIO(response.content)) as tar:
            tar.extractall(tmp_dir)

        # The first directory after unpacking
        first_level_directory = os.path.join(tmp_dir, os.listdir(tmp_dir)[0])

        logging.debug('Create symlink from {source} to {dest}'.format(source=first_level_directory, dest=scripts_dir))

        os.symlink(first_level_directory, scripts_dir)

    def sync_resource(self):
        task = self.task
        scripts_dir = str(self.dir)

        resource = self.get_task_scripts_resource(task)

        logging.debug('Scripts resource: {}'.format(resource))

        self.unpack_scripts_resource(resource, scripts_dir)

    @staticmethod
    def get_task_scripts_resource(task):
        """
        :param task: Task instance
        :type task: sdk2.Task
        :rtype: sdk2.Resource
        """
        if task.Parameters.scripts_last_resource:
            params = dict(
                type=SandboxCiScripts,
                attrs=dict(
                    branch=SCRIPTS_BRANCH_NAME,
                    released=ctt.ReleaseStatus.STABLE,
                ),
                state=ctr.State.READY,
            )
            logging.debug('Looking for "{type}" resource in "{state}" with attrs: {attrs}'.format(
                type=params['type'],
                state=params['state'],
                attrs=params['attrs'],
            ))

            return sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()

        resource = task.Parameters.scripts_resource
        logging.debug('Resource is specified: {}'.format(resource))

        return resource

    @staticmethod
    def unpack_scripts_resource(resource, directory):
        """
        :param resource: Resource with CI scripts archive
        :type resource: sdk2.Resource
        :param directory: Scripts directory
        :type directory: str
        :rtype: None
        """
        resource_path = sdk2.ResourceData(resource).path

        logging.debug('Unpacking {source} to {destination}'.format(source=resource_path, destination=directory))

        with tarfile.open(name=str(resource_path), mode='r:*') as tar:
            tar.extractall(directory)

    def clone(self, url=None, ref=None):
        """
        Clones scripts from github repository.

        :param url: Repository url, default value is task property scripts_git_url
        :param ref: Checkout reference, default value is task property scripts_git_ref
        """

        if url is None:
            url = self.task.Parameters.scripts_git_url

        if ref is None:
            ref = self.task.Parameters.scripts_git_ref

        with GitRetryWrapper(), self.task.vault.ssh_key():
            logging.debug('Cloning scripts from {url}#{ref} to {dir}'.format(url=url, ref=ref, dir=self.dir))
            git_clone(url, ref, self.dir)

    def run_bash_script(self, script_path, *args, **kwargs):
        script_name = file_stem(script_path)
        script_fullpath = self.dir.joinpath(script_path)

        logging.debug('Running bash script "{script_path}" with args: {script_args}'.format(
            script_path=script_path,
            script_args=args,
        ))

        return run_process(
            tuple([script_fullpath]) + args,
            work_dir=kwargs.get('work_dir', self.task.project_dir),
            shell=True,
            log_prefix=kwargs.get('log_prefix', script_name)
        )

    def run_js(self, script_path, *args, **kwargs):
        """
        Runs js script in shell with NVM and parses stdout as JSON.

        :type script_path: str
        :returns: dictionary with script result.
        :rtype: dict
        """
        script_name = file_stem(script_path)
        script_fullpath = self.dir.joinpath(script_path)

        logging.debug('Running script "{script_path}" with args: {script_args}'.format(
            script_path=script_path,
            script_args=format_args_for_run_process(args),
        ))

        with Node(self.task.Parameters.node_js_version):
            process = run_process(
                tuple([script_fullpath]) + args,
                work_dir=kwargs.get('work_dir', self.task.path()),
                shell=True,
                check=False,
                log_prefix=kwargs.get('log_prefix', script_name),
                outputs_to_one_file=False,
                stdout=subprocess.PIPE,
            )

        # read output in stdout, and wait for the subprocess to exit
        stdout, _ = process.communicate()

        try:
            check_process_return_code(process)
        except SandboxSubprocessError as exception:
            raise RunJsScriptError(exception, stdout)

        if stdout:
            # write only stdout, stderr was recorded by run_process
            self.__write_stdout(stdout, process.stdout_path)
            return self.__parse_stdout(stdout)

    @staticmethod
    def __write_stdout(stdout, stdout_path):
        logging.debug('Writing stdout to {}'.format(stdout_path))

        try:
            with open(stdout_path, 'w') as stdout_file:
                stdout_file.write(stdout)
        except IOError as error:
            logging.debug('Error writing the file {path}: {error}'.format(
                path=stdout_path,
                error=error,
            ))

    @staticmethod
    def __parse_stdout(stdout):
        try:
            res_json = json.loads(stdout.strip())
            logging.debug('Script returned JSON: {}'.format(res_json))
            return res_json
        except ValueError as error:
            logging.debug('JSON parse error: {}'.format(error))


class RunJsScriptError(SandboxSubprocessError):
    def __init__(self, error, stdout=None, stderr=None):
        """
        :param error: instance of SandboxSubprocessError
        :param stdout: subprocess PIPE stdout
        :param stderr: subprocess PIPE stderr
        """
        for key, val in vars(error).items():
            setattr(self, key, val)

        self.stdout = stdout
        self.stderr = stderr
