# -*- coding: utf-8 -*-

import logging
import json
import os

from sandbox import sdk2

from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.common.types import misc as ctm
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT

SUBTASK_INDEXES_LIMIT = 100


class SelectiveManager(object):
    index_file_name = 'selective.log'
    index_resource_type = 'selective-index'
    log_file_name = 'selective-errors.log'
    log_resource_type = 'selective-errors'

    def __init__(self, task):
        """
        :param task: task
        :type task: sandbox.sdk2.Task
        """
        self.task = task

    def get_reports_attrs(self, **attrs):
        """
        :return: list of reports attrs
        :rtype: list of dicts
        """
        return [
            dict(
                resource_path=self.index_file_name,
                type=self.index_resource_type,
                **attrs
            ),
            dict(
                resource_path=self.log_file_name,
                type=self.log_resource_type,
                **attrs
            )
        ]

    def publish_merged_index(self, index_path, task_ids, attrs):
        """
        :param index_path: path to index file
        :type index_path: str
        :param task_ids: task ids
        :type task_ids: list of int
        :param attrs: extra attrs
        :type attrs: dict
        :return: index resource
        :rtype: sandbox.sdk2.Resource
        """
        try:
            resources = self.__get_subtask_resources()
            merged_index_path = str(index_path)
            self.__merge_indexes(resources, merged_index_path)

            return self.task.artifacts.create_report(
                resource_path=merged_index_path,
                type=self.index_resource_type,
                task_id=self.task.id,
                status=ctt.Status.SUCCESS,
                **attrs
            )
        except Exception as e:
            logging.exception('Cannot merge selective indexes: %s', e)

    def get_index(self, target_dir):
        """
        :param target_dir: out dir
        :type target_dir: str
        """
        task = self.task

        if not hasattr(task.Parameters, 'selective_run') or not hasattr(task.Parameters, 'project_base_tree_hash') or not task.Parameters.selective_run:
            logging.info('Getting index was skipped')
            return

        tree_hash = task.Parameters.project_base_tree_hash

        logging.info('Trying to find "{}" selective index for {} with tree_hash {}'.format(
            self.index_resource_type,
            task.tool,
            tree_hash,
        ))

        index_resource = task.artifacts.get_last_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            type=self.index_resource_type,
            tool=task.tool,
            project=task.project_name,
            project_tree_hash=tree_hash,
            released=ctt.ReleaseStatus.STABLE,
        )

        if not index_resource:
            logging.warn('Selective index not found')
            return

        self.task.artifacts.save_resources([index_resource], target_dir)
        # Ресурсы по умолчанию создаются read-only.
        os.chmod(os.path.join(str(target_dir), self.index_file_name), 0o777)

    def __get_subtask_resources(self):
        """
        :return: index resources
        :rtype: list of sandbox.sdk2.Resource
        """
        task_ids = self.__get_subtask_with_index_ids()
        if not task_ids:
            return []

        return sdk2.Resource.find(
            type=SANDBOX_CI_ARTIFACT,
            task_id=task_ids,
            attrs=dict(type=self.index_resource_type),
            state=ctr.State.READY,
        ).limit(SUBTASK_INDEXES_LIMIT)

    def __get_subtask_with_index_ids(self):
        subtasks = self.task.find(status=(ctt.Status.FAILURE, ctt.Status.SUCCESS))
        subtask_ids = map(int, subtasks)
        reused_task_ids = map(lambda t: None if t.Context.same_task == ctm.NotExists else t.Context.same_task, subtasks)

        return filter(None, subtask_ids + reused_task_ids)

    def __merge_indexes(self, resources, index_path):
        """
        :param resources: indexes
        :type resources: list of sandbox.sdk2.Resource
        :param index_path: out path
        :type index_path: str
        """
        merged = {}

        for res in resources:
            with open(str(sdk2.ResourceData(res).path)) as f:
                for test, data in json.load(f).items():
                    if test in merged:
                        self.__merge_index_data(merged[test], data)
                    else:
                        merged[test] = data

        with open(index_path, 'w') as f:
            f.write(json.dumps(merged, ensure_ascii=False).encode('utf8'))

    @staticmethod
    def __merge_index_data(dst, src):
        dst['selectors'] = list(set(dst.get('selectors', []) + src.get('selectors', [])))
        dst['clarifyBlockNames'] = list(set(dst.get('clarifyBlockNames', []) + src.get('clarifyBlockNames', [])))

        prev_no_sel = dst.get('noSelectors', {})
        no_sel = src.get('noSelectors', {})
        dst['noSelectors'] = {
            k: bool(prev_no_sel.get(k) and no_sel.get(k)) for k in list(set(prev_no_sel.keys() + no_sel.keys()))
        }
