# -*- coding: utf-8 -*-

import logging
from sandbox.sandboxsdk import process

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper


class SelectiveChecks(object):
    """
    Manager which helps to get selective checks to run (gemini and hermione)
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        """
        self.task = task

    @property
    def use_overlayfs(self):
        return getattr(self.task, 'use_overlayfs')

    @property
    def use_arc(self):
        return getattr(self.task.Parameters, 'use_arc', False)

    @property
    @in_case_of('use_overlayfs', '_work_dir_in_overlayfs_mode')
    def _work_dir(self):
        return self.task.project_dir

    def _work_dir_in_overlayfs_mode(self):
        return self.task.project_sources_dir

    @in_case_of('use_arc', '_get_arc_prev_merge_commit')
    def _get_prev_merge_commit(self):
        res = ''

        with GitRetryWrapper():
            res = process.run_process(
                cmd='git log --pretty=format:"%H" --skip 1 --merges --max-count 1',
                work_dir=str(self._work_dir),
                outs_to_pipe=True,
                shell=True
            ).communicate()[0].strip()

        return res

    def _get_arc_prev_merge_commit(self):
        return getattr(self.task.Parameters, 'project_range_base_hash')

    def checks_to_skip(self):
        """
        Возвращает список проверок, которые нужно пропустить

        :rtype: list of str
        """
        is_enabled = getattr(self.task.Parameters, 'selective_checks', False) and self.task.project_conf.get('selective_checks', False)

        if not is_enabled:
            logging.debug('Selective checks are disabled')
            return []

        checks_to_skip = self.__get_selective_checks().get('toSkip', [])
        logging.info('Checks to skip: {}'.format(', '.join(checks_to_skip)))
        return checks_to_skip

    def __get_selective_checks(self):
        prev_merge_commit = ''
        try:
            prev_merge_commit = self._get_prev_merge_commit()
        except Exception as e:
            logging.info('Cannot get prev merge commit {}'.format(e))
            prev_merge_commit = 'origin/{}'.format(self.task.Parameters.project_git_base_ref)

        """
        :return:
        {
            "toRun": ["gemini.touch-pad", "gemini.touch-phone", "hermione.touch-pad", "hermione.touch-phone"],
            "toSkip": ["gemini.desktop", "hermione.desktop"]
        }
        """
        with GitRetryWrapper():
            return self.task.scripts.run_js(
                'script/selective-checks/get-checks-conditions.js',
                {
                    'base': prev_merge_commit,
                    'working-copy': str(self._work_dir),
                    'config': '.config/selective-checks.js',
                    'json': True,
                }
            )
