# -*- coding: utf-8 -*-

import logging
import json

from sandbox.sandboxsdk import process
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT

diff_lines_type = 'diff-lines'
selective_coverage_log_type = 'selective-coverage-log'
selective_coverage_selection_type = 'selective-coverage-selection'
selective_coverage_fails_type = 'selective-coverage-fails'


class SelectiveCoverageManager(object):
    def __init__(self, task):
        """
        :param task: задача
        """
        self.task = task

    def create_diff_lines(self, commit, log_attrs, path='diff-lines.json'):
        """
        Генерирует и сохраняет ресурс JSON представления диффа между
        базовым коммитом и `commit`.

        Базовый коммит определяется следующим образом: ищутся все ресурсы лога
        покрытия по `log_attrs`, из них выбирается созданный раньше остальных.
        Базовый коммит берется из атрибута `commit` найденного ресурса.

        :param commit: хэш коммита
        :type commit: str
        :param log_attrs: список словарей атрибутов для поиска ресурсов лога покрытия
        :type log_attrs: list of dict
        :param path: путь, куда будет сохранен JSON (относительно working_path)
        :type path: str
        """
        try:
            base_commit = self._find_base_commit_for_diff(log_attrs)
            if base_commit is None:
                logging.debug('Unable to build diff-lines: failed to determine base commit')
                return

            diff_lines_path = str(self.task.working_path(path))
            self._run_npx('@yandex-int/diff-to-lines@1.1.0 {base_commit} {commit} > {path}'.format(
                base_commit=base_commit,
                commit=commit,
                path=diff_lines_path,
            ))

            self.task.artifacts.create_report(
                resource_path=diff_lines_path,
                type=diff_lines_type,
                base_commit=base_commit,
                commit=commit,
            )
        except Exception as e:
            logging.debug('Unable to build diff-lines: {}'.format(e))

    def fetch_diff_lines(self, commit, **attrs):
        """
        Загружает ресурс JSON представления диффа.

        :param commit: хэш коммита
        :type commit: str
        :param attrs: дополнительные атрибуты для поиска ресурса
        :rtype: sandbox.sdk2.Resource
        """
        diff_lines_resource = self.task.artifacts.get_last_project_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            type=diff_lines_type,
            commit=commit,
            **attrs
        )

        if diff_lines_resource:
            self.task.artifacts.save_resources([diff_lines_resource], self.task.project_dir)

        return diff_lines_resource

    def build_selection(
        self,
        commit,
        log_path='selective-coverage-log',
        diff_lines_path='diff-lines.json',
        selection_path='selective-coverage-selection.json',
        **log_attrs
    ):
        """
        Генерирует выборку тестов для запуска на основе изменений и лога
        покрытия. Изменения определяются на основе ресурса diff-lines,
        который должен быть опубликован в задаче сборки.

        Лог селективности создается в тестовых задачах.

        :param commit: хэш коммита
        :type commit: str
        :param log_attrs: атрибуты для поиска ресурсов лога покрытия
        """
        try:
            diff_lines_resource = self.fetch_diff_lines(commit=commit)
            if not diff_lines_resource:
                raise Exception('diff-lines not found')
            self.task.Context.selection_base_commit = diff_lines_resource.base_commit

            log_resource = self.fetch_log(**log_attrs)
            if not log_resource:
                raise Exception('selective-coverage-log not found')

            self._run_npx('@yandex-int/selective-coverage-checker -s {log_path} -d {diff_lines_path} > {selection_path}'.format(
                log_path=log_path,
                diff_lines_path=diff_lines_path,
                selection_path=selection_path,
            ))
        except Exception as e:
            logging.debug('Unable to create selective-coverage-selection: {}'.format(e))

    def build_changed_files_from_diff_lines(self, diff_lines_path='diff-lines.json', path='selective-changed-files.log'):
        """
        Создает список измененных файлов на основе diff-lines (JSON представления диффа).

        :param diff_lines_path: путь diff-lines
        :type diff_lines_path: str
        :param path: путь, куда будет сохранен список файлов
        :type path: str
        """
        diff_lines = json.load(open(str(self.task.project_dir / diff_lines_path)))
        files = map(lambda x: x['file'], diff_lines)

        with open(str(self.task.project_dir / path), 'w') as out:
            out.write('\n'.join(files))

    def get_selection_reports_attrs(self, commit, path='selective-coverage-selection.json'):
        """
        Получает данные о ресурсах выборки тестов для запуска на основе изменений и лога покрытия.

        :param commit: хэш коммита
        :type commit: str
        :param path: путь ресурса
        :type path: str
        """
        return [
            dict(
                self.task.report_common_attributes,
                resource_path=path,
                type=selective_coverage_selection_type,
                base_commit=self.task.Context.selection_base_commit,
                commit=commit,
            ),
            # Если есть selective-coverage-fails.json (генерируется selective-coverage-checker'ом), сохраняем и его.
            dict(
                self.task.report_common_attributes,
                resource_path='selective-coverage-fails.json',
                type=selective_coverage_fails_type,
                base_commit=self.task.Context.selection_base_commit,
                commit=commit,
            )
        ]

    def get_log_report_attrs(self, commit, path='selective-coverage-log'):
        """
        Получает данные о ресурсе лога покрытия.

        :param commit: хэш коммита
        :type commit: str
        :param path: путь ресурса
        :type path: str
        """
        return dict(
            self.task.report_common_attributes,
            resource_path=path,
            type=selective_coverage_log_type,
            commit=commit,
        )

    def fetch_log(self, **attrs):
        """
        Загружает ресурс лога покрытия.

        :param attrs: атрибуты для поиска
        :rtype: sandbox.sdk2.Resource
        """
        log_resource = self.task.artifacts.get_last_project_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            type=selective_coverage_log_type,
            **attrs
        )

        if log_resource:
            self.task.artifacts.save_resources([log_resource], self.task.project_dir)

        return log_resource

    def _find_base_commit_for_diff(self, log_attrs):
        find_res = lambda attrs: self.task.artifacts.get_last_project_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            type=selective_coverage_log_type,
            **attrs
        )

        resources = map(find_res, log_attrs)
        if None in resources:
            return None

        earliest = min(resources, key=lambda r: r.created)

        return earliest.commit

    def _run_npx(self, cmd):
        npx_cmd = 'npm_config_registry=http://npm.yandex-team.ru npx {}'.format(cmd)
        logging.debug('Run npx command: {}'.format(npx_cmd))

        process.run_process(
            npx_cmd,
            work_dir=str(self.task.project_dir),
            shell=True,
            log_prefix='selective-coverage',
        )
