# -*- coding: utf-8 -*-

import logging
import json

from sandbox.sandboxsdk import process
from sandbox.projects.sandbox_ci.utils.context import Node

TESTCOP_CLI_VERSION = "1.4.0"


class SkipListManager(object):
    def __init__(self, task):
        """
        :param task: задача
        """
        self.task = task

    def get_id(self, project, tool, branch='dev', path='skip-list-id.json'):
        """
        Генерирует и сохраняет ресурс JSON с id скип листа.

        :param project: название проекта
        :type project: str
        :param tool: название инструмента
        :type tool: str
        :param branch: название ветки
        :type branch: str
        :param path: путь, куда будет сохранен JSON (относительно working_path)
        :type path: str
        :rtype: int
        """
        # allow npm-related errors to fail the task
        self._npm_install('@yandex-int/testcop-cli@{version}'.format(version=TESTCOP_CLI_VERSION))
        try:
            skip_list_id_path = str(self.task.working_path(path))

            self._run_npx('@yandex-int/testcop-cli@{version} get-skip-list-id --project {project} --tool {tool} --branch {branch} > {path}'.format(
                version=TESTCOP_CLI_VERSION,
                project=project,
                tool=tool,
                branch=branch,
                path=skip_list_id_path,
            ))

            return int(self._read_json(skip_list_id_path))
        except Exception as e:
            logging.debug('Unable to get id: {}'.format(e))

    def get_tests(self, project, tool, branch='dev', path='skip-list.json'):
        """
        Генерирует и сохраняет ресурс JSON с тестами из скип листа.

        :param project: название проекта
        :type project: str
        :param tool: название инструмента
        :type tool: str
        :param branch: название ветки
        :type branch: str
        :param path: путь, куда будет сохранен JSON (относительно working_path)
        :type path: str
        :rtype: list of dict
        """
        # allow npm-related errors to fail the task
        self._npm_install('@yandex-int/testcop-cli@{version}'.format(version=TESTCOP_CLI_VERSION))
        try:
            skip_list_path = str(self.task.working_path(path))

            self._run_npx('@yandex-int/testcop-cli@{version} get-skip-list --project {project} --tool {tool} --branch {branch} > {path}'.format(
                version=TESTCOP_CLI_VERSION,
                project=project,
                tool=tool,
                branch=branch,
                path=skip_list_path,
            ))

            return self._read_json(skip_list_path)
        except Exception as e:
            logging.debug('Unable to get skips: {}'.format(e))

    def _read_json(self, filepath):
        logging.debug('Try to read file: {}'.format(filepath))

        with open(filepath, 'r') as out_data:
            return json.load(out_data)

    def _run_npx(self, cmd):
        npx_cmd = 'npm_config_registry=http://npm.yandex-team.ru npx {}'.format(cmd)
        logging.debug('Run npx command: {}'.format(npx_cmd))

        with Node(self.task.Parameters.node_js_version):
            process.run_process(
                npx_cmd,
                work_dir=str(self.task.working_path()),
                shell=True,
                log_prefix='skip-list',
            )

    def _npm_install(self, package):
        npm_cmd = 'npm_config_registry=http://npm.yandex-team.ru npm i --no-package-lock {}'.format(package)
        logging.debug('Run npm command: {}'.format(npm_cmd))

        with Node(self.task.Parameters.node_js_version):
            process.run_process(
                npm_cmd,
                work_dir=str(self.task.working_path()),
                shell=True,
                log_prefix='npm-install',
            )
