# -*- coding: utf-8 -*-

import os
import logging

from sandbox import sdk2
from sandbox.common.types import task as ctt, resource as ctr

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT, HermioneStabilityIndex


class StabilityIndexManager(object):
    def __init__(self, task):
        """
        :param task: задача
        """
        self.task = task

    @property
    def __resource_type(self):
        return 'stability-index'

    @property
    def __output_path(self):
        return self.__build_index_path(self.__tool, 'output')

    @property
    def __input_path(self):
        return self.__build_index_path(self.__tool, 'input')

    @property
    def __input_path_without_tool(self):
        # TODO: оторвать после https://a.yandex-team.ru/review/806302/details, нужно для обратной совместимости
        return str(self.task.project_dir / 'input-stability-index.sqlite')

    @property
    def __tool(self):
        return self.task.tool if hasattr(self.task, 'tool') else 'hermione'

    def __build_index_path(self, tool, type):
        return str(self.task.project_dir / '{}.{}-stability-index.sqlite'.format(tool, type))

    def prepare_input(self, target_dir):
        try:
            latest_released_stability_index = self.__get_latest_released(tool=self.__tool)
            if latest_released_stability_index:
                self.task.artifacts.save_resources([latest_released_stability_index], target_dir)
        except Exception as e:
            logging.error('Cannot prepare an input stability index: {}'.format(e))

    def configure(self):
        os.environ['hermione_muted_tests_stability_index_output_path'] = self.__output_path
        os.environ['hermione_muted_tests_stability_index_output_run_id'] = str(self.task.id)
        os.environ['hermione_muted_tests_stability_index_output_context'] = self.task.Parameters.project_build_context

        if os.path.exists(self.__input_path):
            os.environ['hermione_muted_tests_stability_index_input_path'] = self.__input_path
        elif os.path.exists(self.__input_path_without_tool):
            # TODO: оторвать после https://a.yandex-team.ru/review/806302/details, нужно для обратной совместимости
            os.environ['hermione_muted_tests_stability_index_input_path'] = self.__input_path_without_tool
        else:
            os.environ['hermione_muted_tests_stability_index_input_enabled'] = 'false'

    def get_output_report_attrs(self):
        return self.__get_attrs(self.__output_path, tool=self.__tool)

    def publish_merged(self, tasks, tool=None):
        """
        :param tasks: задачи, индексы которых нужно объединить
        :type tasks: list of sandbox.sdk2.Task
        :param tool: tool задач
        :type tool: string
        """
        try:
            stability_indexes = self.__get_from_tasks(tasks)
            if tool is None:
                tool = self.__tool

            if len(stability_indexes):
                latest_released = self.__get_latest_released(tool=tool)
                if latest_released:
                    stability_indexes.append(latest_released)

                merged_index_path = self.__build_index_path(tool, 'input')

                self.__merge(input=map(lambda i: sdk2.ResourceData(i).path, stability_indexes), output=merged_index_path)
                self.__clean(merged_index_path)
                self.__publish(merged_index_path, tool=tool, released=ctt.ReleaseStatus.STABLE)
        except Exception as e:
            logging.exception('Cannot publish merged stability index: %s', e)

    def __publish(self, path, **kwargs):
        """
        :param path: путь до ресурса в текущей рабочей директории
        :type path: str
        :param **kwargs: дополнительные атрибуты для паблиша ресурса
        :type **kwargs: dict
        """
        self.task.artifacts.create_report(**self.__get_attrs(path, **kwargs))

    def __get_attrs(self, path, **kwargs):
        return dict(
            resource_type=HermioneStabilityIndex,
            resource_path=os.path.basename(path),
            type=self.__resource_type,
            **kwargs
        )

    def __get_from_tasks(self, tasks):
        """
        :param tasks: список тасок, в ресурсах которых будет осуществляться поиск индекса стабильности
        :type tasks: list of sandbox.sdk2.Task
        :rtype: list of sandbox.sdk2.Resource
        """
        return filter(bool, map(lambda task: sdk2.Resource.find(task_id=task.id, state=ctr.State.READY, attrs={'type': self.__resource_type}).first(), tasks))

    def __get_latest_released(self, **kwargs):
        """
        :param **kwargs: дополнительные атрибуты для поиска ресурса
        :type **kwargs: dict
        :rtype: sandbox.sdk2.Resource
        """
        def find(resource_type):
            return self.task.artifacts.get_last_artifact_resource(
                resource_type=resource_type,
                type=self.__resource_type,
                project=self.task.project_name,
                released=ctt.ReleaseStatus.STABLE,
                **kwargs
            )

        return find(HermioneStabilityIndex) or find(SANDBOX_CI_ARTIFACT)

    def __merge(self, input, output):
        """
        :param input: список путей до индексов стабильности, которые надо смёржить
        :type input: list of str
        :param output: путь, по которому будет сохранен объединённый индекс стабильности
        :type output: str
        """
        sdk2.Path(os.path.dirname(output)).mkdir(parents=True, exist_ok=True)

        self.task.scripts.run_js('script/sqlite-cli.js', 'merge', {
            'input': input,
            'output': output,
        })

    def __clean(self, input):
        self.task.scripts.run_js('script/stability-index-cleaner.js', {
            'input': input,
            'time-threshold': os.environ.get('hermione_muted_tests_stability_index_cleaner_time_threshold', 0),
        })
