# -*- coding: utf-8 -*-

import os

from sandbox.common.utils import singleton_property
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.sandbox_ci.utils.arcanum import ArcanumApi
from sandbox.projects.sandbox_ci.utils.startrek import parse_issue_keys
from sandbox.projects.sandbox_ci.utils.ref_formatter import is_arcanum_ref, is_github_ref, strip_ref_prefix

TRIVIAL_PREFIX = 'TRIVIAL'


class StartrekManager(object):

    def __init__(self, task):
        """
        :param task: Task instance
        :type task: sdk2.Task
        """
        self.task = task

    @property
    def refs_names(self):
        return getattr(self.task.Parameters, 'project_git_merge_ref', [])

    @singleton_property
    def arcanum_api(self):
        arcanum_api_token = os.environ.get('ARCANUM_API_OAUTH_TOKEN')

        if not arcanum_api_token:
            raise Exception('Should use Arcanum but ARCANUM_API_OAUTH_TOKEN env is missing')

        return ArcanumApi(token=arcanum_api_token)

    @singleton_property
    def github_api(self):
        return GitHubApi()

    def _get_pr_titles(self):
        return map(self._get_pr_title, self.refs_names)

    def _get_pr_title(self, ref):
        pr_num = strip_ref_prefix(ref)

        if is_arcanum_ref(ref):
            return self.arcanum_api.get_pr_title(pr_num)

        if is_github_ref(ref):
            owner = self.task.Parameters.project_github_owner
            repo = self.task.Parameters.project_github_repo

            return self.github_api.get_pr_title(owner, repo, pr_num)

    def get_issue_keys(self):
        parse_issue_keys(self._get_pr_titles())

    def get_issues_info(self):
        titles = self._get_pr_titles()

        return {
            'has_trivial': any(map(lambda title: title.startswith(TRIVIAL_PREFIX), titles)),
            'keys': parse_issue_keys(titles)
        }
