# -*- coding: utf-8 -*-

import os
import logging

from sandbox import sdk2

from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.dict_utils import defaults


class StatReportManager(object):
    def __init__(self, task):
        self.task = task

        self._html_output = 'stat-reporter/{}-html'.format(self.task.tool)
        self._json_output = 'stat-reporter/{}.json'.format(self.task.tool)
        self._html_type = '{}-stat-reporter-html'.format(self.task.tool)
        self._json_type = '{}-stat-reporter-json'.format(self.task.tool)

    def set_env(self):
        os.environ['stat_reporter_reporters_html_enabled'] = 'true'
        os.environ['stat_reporter_reporters_json_enabled'] = 'true'
        os.environ['stat_reporter_reporters_html_path'] = self._html_output
        os.environ['stat_reporter_reporters_json_path'] = self._json_output

    def get_reports_attrs(self, common_attributes):
        return [
            defaults(common_attributes, dict(
                resource_path=self._json_output,
                type=self._json_type
            )),
            defaults(common_attributes, dict(
                resource_path=self._html_output,
                type=self._html_type,
                add_to_context=True,
                root_path='index.html'
            ))
        ]

    def publish_merged_stat_reports(self, task_ids, attrs={}):
        try:
            reports_paths = map(lambda i: sdk2.ResourceData(i).path, self.__load_stat_reports(task_ids))

            if not len(reports_paths):
                return

            self.__merge_stat_reports(reports_paths)

            self.task.artifacts.create_reports(self.get_reports_attrs(attrs), parallel=True)
        except Exception as e:
            logging.error('Cannot merge stat reports: {}'.format(e))

    def __load_stat_reports(self, task_ids):
        return filter(bool, map(lambda task_id: sdk2.Resource.find(task_id=task_id, attrs={'type': self._json_type}).first(), task_ids))

    def __merge_stat_reports(self, reports_paths):
        with Node(self.task.Parameters.node_js_version):
            run_process(
                ['./node_modules/.bin/hermione', 'merge-stat-reports', '-h', self._html_output, '-j', self._json_output] + reports_paths,
                log_prefix='merge_stat_reports',
                shell=False,
                work_dir=self.task.project_dir
            )
