# -*- coding: utf-8 -*-

from sandbox import common
import logging


class DecoratedStatReport:
    def __init__(self, report, task):
        self._report = report
        self._task = task

    def __getattr__(self, attribute):
        return getattr(self._report, attribute)

    def _should_report_to_statface(self):
        send_statistic_config = self._task.project_conf.get('stat', {}).get('send_statistic', False)
        send_statistic = getattr(self._task.Parameters, 'send_statistic', True)

        return send_statistic_config and send_statistic

    def upload_data(self, scale, data, format='py', request_kwargs=None, **statface_params):
        if self._should_report_to_statface():
            logging.debug('Uploading data to Statface: {}'.format(data))
            self._report.upload_data(scale, data, format, request_kwargs, **statface_params)
        else:
            logging.warn('Skip uploading data to Statface')

    def upload_config(self, config, overwrite=True, request_kwargs=None, scale=None, **statface_params):
        if self._should_report_to_statface():
            logging.debug('Uploading config to Statface: {}'.format(config.to_dict()))
            self._report.upload_config(config, overwrite, request_kwargs, scale, **statface_params)
        else:
            logging.warn('Skip uploading config to Statface')


class StatfaceManager(object):
    """Manager that helps to send data to Statface."""

    def __init__(self, task):
        """
        :param task: Task instance
        :type task: sdk2.Task
        """
        self.task = task

    @property
    @common.utils.singleton
    def client(self):
        from statface_client import StatfaceClient

        logger = logging.getLogger('statface_client')
        logger.addHandler(logging.StreamHandler())
        logger.setLevel(logging.DEBUG)

        stat_oauth_token = self.task.vault.read('env.STAT_OAUTH_TOKEN')

        # не все таски наследуются от BaseTask, поэтому
        # определим дефолтный хост на случай если параметра нет
        host = 'upload.stat.yandex-team.ru'
        if hasattr(self.task.Parameters, 'statface_host'):
            host = str(self.task.Parameters.statface_host)

        return StatfaceClient(oauth_token=stat_oauth_token, host=host)

    def get_report(self, report_path):
        return DecoratedStatReport(self.client.get_report(report_path), self.task)

    def get_new_report(self, report_path):
        return DecoratedStatReport(self.client.get_new_report(report_path), self.task)

    def get_old_report(self, report_path):
        return DecoratedStatReport(self.client.get_old_report(report_path), self.task)

    def get_dict(self, name, **kwargs):
        params = dict(name=name, **kwargs)

        logging.debug('get dictionary with params: {}'.format(params))

        res = self.client._request('get', '_api/dictionary', params=params)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return res.text
