# -*- coding: utf-8 -*-

import re
import logging
from sandbox.sandboxsdk import process
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper

PROJECTS_REGEX = r'(?:services|projects)\/(.+?)\/'


class TagsManager(object):
    def __init__(self, task):
        self._task = task

    def get_project_tags(self, work_dir, owner, repo, base_hash, use_arc=False):
        tags = []
        base_tag = '{}/{}'.format(owner, repo)
        try:
            arc_command = ['arc', 'diff', '--name-only', base_hash, 'HEAD', work_dir, ' | cut -d "/" -f4-']
            git_command = ['git', 'diff', '--name-only', '--no-renames', base_hash]
            command = arc_command if use_arc else git_command

            logging.debug('Running command: {}'.format(command))

            with GitRetryWrapper():
                p = process.run_process(
                    cmd=command,
                    work_dir=work_dir,
                    log_prefix='project_tag',
                    shell=True,
                )

            with open(p.stdout_path, 'r') as out_file:
                file_paths = out_file.readlines()
                project_names = self.extract_project_names(file_paths)
                tags = self.format_project_tags(project_names, base_tag)

        except Exception as error:
            logging.exception('Cannot add tags with projects {}'.format(error))

        if not tags:
            tags.append(base_tag)

        return tags

    @staticmethod
    def extract_project_names(file_paths):
        names = set()

        for file_path in file_paths:
            matches = re.findall(PROJECTS_REGEX, file_path)
            names.update(matches)

        logging.debug('Project names from changed file names: {}'.format(names))

        return list(names)

    @staticmethod
    def format_project_tags(project_names, base_tag):
        tags = set()

        for project_name in project_names:
            tag = '{}/{}'.format(base_tag, project_name)
            tags.add(tag)

        logging.debug('Formatted project tags: {}'.format(tags))

        return list(tags)
