# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2

from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property
from sandbox.common.errors import TaskStop, TaskFailure


class TaskDependenciesManager(object):
    def __init__(self, task):
        self.task = task

    @singleton_property
    def output_parameters(self):
        """
        Возвращает зависимые output-параметры других задач.

        Пример: `{164174753: 'output_field1', 164174754: ('output_field1', 'output_field2')}`

        :return: output-параметры других задач
        :rtype: dict
        """
        wait_output_parameters = self.task.Parameters.wait_output_parameters
        dependency_output_parameters = {}

        if not wait_output_parameters:
            return dependency_output_parameters

        logging.debug('Try parse wait_output_parameters: {}'.format(wait_output_parameters))

        for task_id, params in wait_output_parameters.items():
            try:
                task_targets = [param.strip() for param in params.split(',')]
                dependency_output_parameters[int(task_id)] = filter(bool, task_targets)
            except ValueError:
                logging.warning('Invalid task wait_output_parameters: {}'.format(params))
                continue

        return dependency_output_parameters

    def wait_output_parameters(self, timeout):
        """
        Ожидает output-параметры из других задач.

        Набор output-параметров и id их задач задаётся в параметре `wait_output_parameters`.
        """
        if not self.output_parameters:
            logging.debug('No output parameters to wait')
            return

        with self.task.memoize_stage.wait_output_parameters:
            raise sdk2.WaitOutput(self.output_parameters, wait_all=True, timeout=timeout)

    def get_output_parameter(self, param_name):
        for [task_id, params] in self.output_parameters.items():
            for p in params:
                if p != param_name:
                    continue

                task = sdk2.Task.find(id=task_id).first()
                if not task:
                    raise TaskFailure('Could not find task dependency')

                return getattr(task.Parameters, param_name, None)

    def ensure_output_parameter(self, param_name, param_value, error_message='', safe=False):
        for [task_id, params] in self.output_parameters.items():
            for p in params:
                if p != param_name:
                    continue

                output_parameters = {param_name: param_value}
                task = sdk2.Task.find(id=task_id, output_parameters=output_parameters).first()

                if task:
                    return True

                error_message = error_message or 'Expected {} of task #{} to be {}'.format(param_name, task_id, param_value)

                if safe:
                    logging.info(error_message)
                    return False
                else:
                    raise TaskStop(error_message)

        return True

    def wait_tasks(self, fail_on_deps_fail):
        dependency_task_ids = self.task.Parameters.wait_tasks

        if not dependency_task_ids:
            logging.debug('No task dependencies to wait')
            return

        with self.task.memoize_stage.wait_tasks:
            logging.info('Will wait for task dependencies: {}'.format(dependency_task_ids))
            raise sdk2.WaitTask(dependency_task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        tasks = list(sdk2.Task.find(id=dependency_task_ids, children=True).limit(len(dependency_task_ids)))
        self.task.Context.wait_tasks_statuses = {str(t): t.status for t in tasks}

        if fail_on_deps_fail and self.task.meta.is_any_failed(tasks):
            self.task.meta.store_failed_tasks(tasks)

            raise TaskStop('Failed task dependencies, could not continue')
