# -*- coding: utf-8 -*-

import logging
import inspect
import os

from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common import gnupg
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import singleton


class VaultManager(object):
    def __init__(self, task, default_owner='SANDBOX_CI_SEARCH_INTERFACES'):
        self.task = task
        self.default_owner = default_owner

    def owner_fallback(self):
        """Считать овнера из задачи или использовать дефолтного"""
        return self.task.vault_owner if hasattr(self.task, 'vault_owner') else self.default_owner

    def is_available(self):
        """
        Доступны ли Vault в данном окружении.
        Если код выполняется в серверном хуке — не доступны.
        """
        return not _is_serverside()

    def read(self, name, owner=None):
        if not self.is_available():
            return ''

        owner = owner or self.owner_fallback()

        return self._read(name, owner)

    @singleton
    def read_all(self, owner=None):
        """Прочитать все записи из Vault"""
        if not self.is_available():
            return []

        owner = owner or self.owner_fallback()

        return self._read_all(owner)

    @singleton
    def read_all_env(self):
        """
        Получить из Vault все переменные, имена которых начинаются с `env.`
        """
        logging.debug('Reading env variables from vault')

        vault_items = self.read_all()
        env_vault_items = filter(_is_env, vault_items)

        with sdk2.Vault.batch:
            vault_item_futures = {_strip_env(item): sdk2.Vault.data(self.owner_fallback(), item['name']) for item in env_vault_items}

        return {name: future.result() for name, future in vault_item_futures.items()}

    @singleton
    def ssh_key(self, vault_name=None):
        """Прочитать приватный ключ для определенного пользователя"""
        if not vault_name:
            vault_name = os.environ.get('SSH_KEY_VAULT_NAME', 'robot-serp-bot.id_rsa') if not vault_name else vault_name

        return sdk2.ssh.Key(self.task, self.owner_fallback(), vault_name)

    @singleton
    def gpg_key(self, secret_name, public_name):
        return gnupg.GpgKey(channel.task, self.owner_fallback(), secret_name, public_name)

    @singleton
    def _read(self, name, owner):
        try:
            logging.debug('Fetching "{}" from vault for owner "{}"'.format(name, owner))
            return sdk2.Vault.data(owner, name)
        except Exception:
            raise TaskFailure('Could not fetch "{}" from vault, check out that you run task under "{}" owner and author belongs to the group'.format(name, owner))

    @singleton
    def _read_all(self, owner):
        logging.debug('Reading everything from %s vault', owner)

        shouldReadVaults = True
        offset = 0
        limit = 100
        items = []

        while shouldReadVaults:
            response = self.task.server.vault.read(owner=owner, offset=offset, limit=limit)
            total = response.get('total', 0)
            items += response.get('items', [])
            offset += limit

            shouldReadVaults = total > offset

        logging.debug('Read {items_count} records from {owner} vault'.format(owner=owner, items_count=len(items)))

        return items


def _is_env(vault_item):
    return vault_item['name'].startswith('env.')


def _strip_env(vault_item):
    """Обрезать 'env.' в начале имени"""
    return vault_item['name'][4:]


def _is_serverside():
    called_functions = set(map(lambda item: item[0].f_code.co_name, inspect.stack()))
    server_callbacks = set(["on_save", "on_create", "on_enqueue"])

    return len(called_functions.intersection(server_callbacks)) > 0
