# -*- coding: utf-8 -*-

import logging
import requests
import json


class CustomEncoder(json.JSONEncoder):
    """
    Конвертирует Object в JSON
    У task.Parameters и task.Context есть реализация метода __getstate__,
    который возвращает информацию в виде tuple в дальнейшем сеализируемый в JSON.
    Данный класс нужен для кастомного преобразования, если же мы будем использовать JSONDecoder (по умолчанию),
    то получим ошибки вида "... is not JSON serializable"
    https://docs.python.org/2/library/json.html
    """

    def default(self, o):
        if hasattr(o, "__getstate__"):
            return o.__getstate__()
        return super(CustomEncoder, self).default(o)


class WebHookManager(object):
    """
    Manager that helps to send data to services.
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        """
        self.task = task

    def payload(self, status):
        """
        :param status: статус задачи
        :type status: str
        :rtype: str
        """
        ctx = self.task.Context.__getstate__()
        params = self.task.Parameters.__getstate__()

        payload = {
            'id': self.task.id,
            'status': status,
            'type': self.task.type.name,
            'context': ctx
        }

        payload.update(params)

        return json.dumps(payload, cls=CustomEncoder)

    def send_post(self, url, data):
        """
        :param url: URL для отправки payload-а
        :type url: str
        :param data: payload
        :type data: str
        :rtype: dict
        """
        try:
            res = requests.post(
                url,
                headers={'Content-Type': 'application/json'},
                data=data,
                verify=False
            )

            logging.debug('response for {url} with code {code} is: {text}'.format(
                url=url,
                code=res.status_code,
                text=res.text
            ))

            return res
        except Exception as error:
            logging.exception('request to {url} failed with {error}'.format(
                url=url,
                error=error
            ))

    def send(self, status, urls):
        """
        :param status: статус задачи
        :type status: str
        :param urls: URL-ы для отправки payload-а
        :type urls: list of str
        :rtype: None
        """
        payload = self.payload(status)

        if payload:
            logging.debug('payload: {}'.format(payload))

            for url in urls:
                self.send_post(url, payload)
        else:
            logging.debug('payload is empty')
