# -*- coding: utf-8 -*-
import re
import sys
import unicodedata
import urllib
from collections import defaultdict

from sandbox.common.errors import TaskFailure

from sandbox.projects.sandbox_ci.pulse.const import HANDLERS_MAP, ROUTES_CONFIG

multiple_spaces_pattern = re.compile(r'\s\s+')
url_pattern = re.compile(r'^(https?:)?//[\w.-]+/', re.IGNORECASE)
text_pattern = re.compile(r'.*text=([^&]*)(?:&.*|$)', re.UNICODE)

# Построить параметры для unicode.translate с целью удаления из запросов служебных символов
# Взято из скриптов аналитиков, которыми они считают топ запросов
query_translation = {
    index: u' ' for index in xrange(sys.maxunicode)
    if unicodedata.category(unichr(index)).startswith('P')
}


def get_routes_json_id(project, platform):
    """
    Получить ID проекта в пространстве routes.json на основе пульсовых ID

    :type project: basestring
    :type platform: basestring

    :rtype: str
    """
    return ROUTES_CONFIG.get(project, {}).get(platform)


def normalize_query(query):
    """
    Нормализовать запрос из логов

    Если является URL – возвращается без изменений
    :type query: basestring
    :rtype: basestring|None
    """
    if isinstance(query, str):
        try:
            query = query.decode('utf8')
        except UnicodeDecodeError:
            return None

    if url_pattern.match(query):
        return query

    query = query.translate(query_translation)
    query = query.lower().strip()
    query = multiple_spaces_pattern.sub(' ', query)

    try:
        return query.encode('utf8')
    except UnicodeEncodeError:
        return None


def get_text_from_url(url_path):
    """
    Извлекает текст(GET параметр) из строки запроса
    :type url_path: basestring
    :rtype: basestring
    """
    matches = text_pattern.match(url_path)
    if not matches:
        return None

    try:
        query_text = matches.group(1).encode('utf8')
    except UnicodeError:
        return None

    raw_text = urllib.unquote_plus(query_text)

    return normalize_query(raw_text)


def build_url_path(project, platform, params):
    """
    Построить path для запроса к сервису по именам проекта и платформы
    :type project: basestring
    :type platform: basestring
    :type params: dict
    :rtype: basestring
    """
    url_path = HANDLERS_MAP.get(project, {}).get(platform)
    if not url_path:
        return None

    return url_path + '?' + urllib.urlencode(params, doseq=True)


def parse_query_params(query_params_list):
    res = defaultdict(list)
    for chunk in query_params_list:
        data = chunk.split('=', 1)
        if len(data) != 2:
            raise TaskFailure('Error in query param value: %s, `name=value` expected' % chunk)
        name, value = data
        res[name].append(value)
    return dict(res)
