# coding=utf-8
import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.sdk2.vcs import svn
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.projects.resource_types import BUILD_OUTPUT
from sandbox.projects.sandbox_ci.pulse.resources import PulseBinary
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement


class BuildPulseBinaries(TasksResourceRequirement, sdk2.Task):
    """
        Построить Pulse Shooter для нужных платформ
    """
    platforms = ['linux', 'darwin']

    class Requirements(sdk2.Requirements):
        disk_space = 5000

    class Context(sdk2.Context):
        task_data = []

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Platforms') as pl_group:
            build_linux = sdk2.parameters.Bool(
                'Build Linux',
                default=True,
            )

            build_darwin = sdk2.parameters.Bool(
                'Build macOS',
                default=True,
            )

        with sdk2.parameters.Group('Projects') as proj_group:
            build_static = sdk2.parameters.Bool(
                'Build Pulse Static',
                default=True,
            )

            build_shooter = sdk2.parameters.Bool(
                'Build Pulse Shooter',
                default=True,
            )

            build_aggregator = sdk2.parameters.Bool(
                'Build Pulse Aggregator',
                default=True,
            )

            build_memory_leaks_detector = sdk2.parameters.Bool(
                'Build Memory Leaks Detector',
                default=True,
            )

            build_report = sdk2.parameters.Bool(
                'Build Pulse Report',
                default=True,
            )

            build_html_differ = sdk2.parameters.Bool(
                'Build HTML Differ',
                default=True,
            )

            build_html_diff_viewer = sdk2.parameters.Bool(
                'Build Diff viewer',
                default=True,
            )

            build_blockstat_log_analyzer = sdk2.parameters.Bool(
                'Build Blockstat log analyzer',
                default=True,
            )

    def on_execute(self):
        if self.Parameters.build_html_diff_viewer:
            with self.memoize_stage.get_diff_viewer:
                viewer_dir = os.path.join(os.getcwd(), 'diff_viewer')
                viewer_archive = os.path.join(os.getcwd(), 'diff_viewer.tar')

                svn.Arcadia.checkout(
                    'arcadia:/arc/trunk/arcadia/velocity/pulse/diff_viewer',
                    path=viewer_dir,
                )

                shutil.rmtree(os.path.join(viewer_dir, '.svn'))
                shutil.rmtree(os.path.join(viewer_dir, 'test'))

                with tarfile.open(viewer_archive, 'w') as tar:
                    tar.add(viewer_dir, arcname='')

                viewer_resource = PulseBinary(
                    task=self,
                    description='Pulse diff viewer',
                    path=viewer_archive,
                    project='pulse-diff-viewer',
                    platform='all',
                )

                sdk2.ResourceData(viewer_resource).ready()

        with self.memoize_stage.build:
            ya_make_params = {
                'checkout_arcadia_from_url': 'arcadia:/arc/trunk/arcadia@HEAD',
                'result_rd': 'Pulse',
                'allure_report': False,
                'check_return_code': True,
            }

            task_data = []
            for platform in self.platforms:
                if (
                        platform == 'linux' and not self.Parameters.build_linux or
                        platform == 'darwin' and not self.Parameters.build_darwin
                ):
                    continue

                run_tests = platform == 'linux'

                if self.Parameters.build_static:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/static/bin',
                        'result_rd': params['result_rd'] + ' Static (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-static',
                        'task_id': self._create_ya_make('Build Pulse Static (%s)' % platform, params),
                    })

                if self.Parameters.build_shooter:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/shooter/bin',
                        'result_rd': params['result_rd'] + ' Shooter (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-shooter',
                        'task_id': self._create_ya_make('Build Pulse Shooter (%s)' % platform, params),
                    })

                if self.Parameters.build_aggregator:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/aggregator/bin',
                        'result_rd': params['result_rd'] + ' Aggregator (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-aggregator',
                        'task_id': self._create_ya_make('Build Pulse Aggregator (%s)' % platform, params),
                    })

                if self.Parameters.build_memory_leaks_detector:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/memory_leaks_detector/bin',
                        'result_rd': params['result_rd'] + ' Memory leaks detector (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-memory-leaks-detector',
                        'task_id': self._create_ya_make('Build Memory Leaks Detector (%s)' % platform, params),
                    })

                if self.Parameters.build_report:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/report/bin',
                        'result_rd': params['result_rd'] + ' Report (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-report',
                        'task_id': self._create_ya_make('Build Pulse Report (%s)' % platform, params),
                    })

                if self.Parameters.build_html_differ:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/html_differ/bin',
                        'result_rd': params['result_rd'] + ' Pulse HTML Differ (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-html-differ',
                        'task_id': self._create_ya_make('Build Pulse HTML Differ (%s)' % platform, params),
                    })

                if self.Parameters.build_blockstat_log_analyzer:
                    params = ya_make_params.copy()
                    params.update({
                        'targets': 'velocity/pulse/blockstat_log_analyzer/bin',
                        'result_rd': params['result_rd'] + ' Blockstat log analyzer (%s)' % platform,
                        'target_platform_flags': '--target-platform %s' % platform,
                        'test': run_tests,
                    })

                    task_data.append({
                        'platform': platform,
                        'project': 'pulse-blockstat-log-analyzer',
                        'task_id': self._create_ya_make('Build Blockstat log analyzer (%s)' % platform, params),
                    })

            self.Context.task_data = task_data

            if not task_data and not self.Parameters.build_html_diff_viewer:
                raise TaskFailure('You should choose at least one build target')

            raise sdk2.WaitTask(
                tasks=[x['task_id'] for x in task_data],
                statuses=tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK),
                wait_all=True,
            )

        with self.memoize_stage.publish_resources:
            for data in self.Context.task_data:
                task_id = data['task_id']
                platform = data['platform']
                project = data['project']

                status = sdk2.Task[task_id].status
                if status != ctt.Status.SUCCESS:
                    raise TaskFailure('Task %s ended with status %s' % (task_id, status))

                ya_resource = BUILD_OUTPUT.find(
                    task_id=task_id,
                ).first()

                if not ya_resource:
                    raise TaskFailure('There is no BUILD_OUTPUT resource')

                ya_resource_data = sdk2.ResourceData(ya_resource)

                target_dir = os.path.join(os.getcwd(), platform, project)
                os.makedirs(target_dir)

                target_file = os.path.join(target_dir, project)
                shutil.copy(
                    os.path.join(str(ya_resource_data.path), 'bin', project),
                    target_file,
                )

                build_resource = PulseBinary(
                    task=self,
                    description='%s (%s)' % (project, platform),
                    path=target_file,
                    platform=platform,
                    project=project,
                )

                sdk2.ResourceData(build_resource).ready()

    def _create_ya_make(self, description, params):
        """
            Это хак для sdk2, скопирован из
            https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/sandbox_acceptance/__init__.py?rev=2525039#L67-86
        """
        task = self.server.task({
            'type': 'YA_MAKE',
            'context': params,
            'children': True,
        })

        update = {
            'description': description,
            'notifications': self._get_notifications(),
            'owner': self.owner,
            'priority': {
                'class': self.Parameters.priority.cls,
                'subclass': self.Parameters.priority.scls,
            },
        }

        self.server.task[task['id']].update(update)
        self.server.batch.tasks.start.update([task['id']])

        return task['id']

    def _get_notifications(self):
        """
            Это хак для sdk2, скопировал из https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/sandbox_acceptance/__init__.py?rev=2525039#L67-86
        """
        return self.server.task[self.id].read()['notifications']
