# coding=utf-8
import logging
from collections import defaultdict


class LimitsNotSet(Exception):
    pass


def load_pulse_genisys_config(subsection, group_field=None):
    """
    Загружает конфиг для pulse из genisys

    :link: https://genisys.yandex-team.ru/sections/pulse
    :param subsection: str
    :param group_field: str
    :rtype: dict
    """
    from library.config import query

    logging.info('Trying to get config from Genisys. subsection={}, group={}'.format(subsection, group_field))

    config = query(section='pulse.{}'.format(subsection), subsection=None, evaluate=True, as_dict=True)

    logging.info('Config from Genisys. subsection={}, group={}, content={}'.format(subsection, group_field, config))

    if not config:
        return {}

    group = process_config(config).get(group_field, {})
    if not group:
        raise EnvironmentError(
            'Unable to get group "{}" for {}.\n Config: {}'.format(group_field, subsection, config)
        )

    return group


def process_config(config):
    """
    Согласует форматы конфига и внутреннее представление метрик

    TODO: оторвать в ходе SERP-63724

    :type config: dict
    :rtype: dict
    """
    if not config:
        return {}

    # Все поля за исключением лимитов берём без изменений
    new_config = {n: v for n, v in config.iteritems() if n != 'limits'}

    # Лимиты для статики так же оставляем без изменений
    limits_config = config.get('limits', {})
    new_config['limits'] = {
        'static_delta': limits_config.get('static', {}),
        'static_file_size': limits_config.get('static_file_size', {}),
        'shooting': defaultdict(lambda: defaultdict(dict)),
    }

    # Для лимитов динамики меняем формат записи лимитов для процентилей – добавляем префикс p
    for platform_name, platform_limits in limits_config.get('shooting', {}).iteritems():
        for metric_name, metric_data in platform_limits.iteritems():
            for aggr_name, aggr_limit in metric_data.iteritems():
                aggr_name = str(aggr_name)
                if aggr_name.isdigit():
                    aggr_name = 'p' + aggr_name
                new_config['limits']['shooting'][platform_name][metric_name][aggr_name] = aggr_limit

    return new_config
