#  -*- coding: utf-8 -*-
import gzip
import json
import logging
import os
import shutil
import tarfile
import tempfile

import sandbox.common.types.misc as ctm
from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.common.dolbilka import DolbilkaPlanner
from sandbox.projects.sandbox_ci.pulse import const as pulse_const
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_params
from sandbox.projects.sandbox_ci import pulse
from sandbox.projects.sandbox_ci.pulse.prepare_report_renderer_plan_source import PrepareReportRendererPlanSource
from sandbox.projects.sandbox_ci.pulse.resources import (
    SearchParsedAccessLog,
    ReportRendererPlan,
    ReportRendererPhantomData,
    ReportRendererPlanJson,
    ReportRendererPlanApphost,
    ReportRendererPhantomDataApphost,
    ReportRendererApphostFailedBodies,
)
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PHANTOM_PLAN_BASE_PATH = 'rr-phantom-ammo'
DOLBILKA_PLAN_BASE_PATH = 'rr-dolbilka-ammo'
DOLBILKA_PLAN_APPHOST_BASE_PATH = 'rr-dolbilka-apphost-ammo'
JSON_PLAN_BASE_PATH = 'rr-json-ammo.json'

PHANTOM_PLAN_COMPRESSED_BASE_PATH = '%s.gz' % PHANTOM_PLAN_BASE_PATH
PHANTOM_PLAN_APPHOST_COMPRESSED_BASE_PATH = '%s-apphost.gz' % PHANTOM_PLAN_BASE_PATH
JSON_PLAN_COMPRESSED_BASE_PATH = '%s.gz' % JSON_PLAN_BASE_PATH

PHANTOM_PLAN_ACTUAL_PATH = 'rr-phantom-ammo-2'
DOLBILKA_PLAN_ACTUAL_PATH = 'rr-dolbilka-ammo-2'
DOLBILKA_PLAN_APPHOST_ACTUAL_PATH = 'rr-dolbilka-apphost-ammo-2'
JSON_PLAN_ACTUAL_PATH = 'rr-json-ammo-2.json'

PHANTOM_PLAN_COMPRESSED_ACTUAL_PATH = '%s.gz' % PHANTOM_PLAN_ACTUAL_PATH
PHANTOM_PLAN_APPHOST_COMPRESSED_ACTUAL_PATH = '%s-apphost.gz' % PHANTOM_PLAN_ACTUAL_PATH
JSON_PLAN_COMPRESSED_ACTUAL_PATH = '%s.gz' % JSON_PLAN_ACTUAL_PATH

SHOTS_COUNT_PATH = 'shots_count.info'

REPORT_HANDLER_CONSTANTS_BASE = {
    'phantomPlanPath': PHANTOM_PLAN_BASE_PATH,
    'jsonPlanPath': JSON_PLAN_BASE_PATH,
    'shotsCountPath': SHOTS_COUNT_PATH,
}

REPORT_HANDLER_CONSTANTS_ACTUAL = {
    'phantomPlanPath': PHANTOM_PLAN_ACTUAL_PATH,
    'jsonPlanPath': JSON_PLAN_ACTUAL_PATH,
}

PHANTOM_PLAN_PATH_MAP = {
    # actual: value
    False: PHANTOM_PLAN_BASE_PATH,
    True: PHANTOM_PLAN_ACTUAL_PATH,
}

JSON_PLAN_PATH_MAP = {
    # actual: value
    False: JSON_PLAN_BASE_PATH,
    True: JSON_PLAN_ACTUAL_PATH,
}

JSON_PLAN_COMPRESSED_PATH_MAP = {
    # actual: value
    False: JSON_PLAN_COMPRESSED_BASE_PATH,
    True: JSON_PLAN_COMPRESSED_ACTUAL_PATH,
}

PHANTOM_PLAN_COMPRESSED_PATH_MAP = {
    # (apphost, actual): value
    (False, False): PHANTOM_PLAN_COMPRESSED_BASE_PATH,
    (False, True): PHANTOM_PLAN_COMPRESSED_ACTUAL_PATH,
    (True, False): PHANTOM_PLAN_APPHOST_COMPRESSED_BASE_PATH,
    (True, True): PHANTOM_PLAN_APPHOST_COMPRESSED_ACTUAL_PATH,
}

DOLBILKA_PLAN_PATH_MAP = {
    # (apphost, actual): value
    (False, False): DOLBILKA_PLAN_BASE_PATH,
    (False, True): DOLBILKA_PLAN_ACTUAL_PATH,
    (True, False): DOLBILKA_PLAN_APPHOST_BASE_PATH,
    (True, True): DOLBILKA_PLAN_APPHOST_ACTUAL_PATH,
}


class PrepareReportRendererPlan(TasksResourceRequirement, sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 30 * 1024
        cores = 2

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String(
            'Project',
            description='Project for which plan will be generated',
            default='web4',
            required=True,
        )

        platform = sdk2.parameters.String(
            'Platform',
            description='Platform for which plan will be generated',
            default='desktop',
            required=True,
        )

        use_soy = sdk2.parameters.Bool(
            'Use Scraper over YT',
            default=False,
        )

        with use_soy.value[True]:
            soy_operation_priority = pulse_params.soy_operation_priority()

        access_log = sdk2.parameters.Resource(
            'Access log resource',
            resource_type=SearchParsedAccessLog,
        )

        access_log_table = sdk2.parameters.String(
            'Access log YT table',
            required=True,
        )

        threshold = sdk2.parameters.Integer(
            'Threshold',
            description='Threshold of minimum possible results for creating plan (0 - no limit)',
            default=0,
        )

        report_host = sdk2.parameters.String(
            'Report host',
            description='Host for data request',
            default='hamster.yandex.ru'
        )

        request_limit = sdk2.parameters.Integer(
            'Request limit',
            default=0,
        )

        parallel_request_limit = sdk2.parameters.Integer(
            'Parallel request limit',
            default=50,
        )

        base_query_params = sdk2.parameters.List(
            'Base ammo query params',
            description='Additional flags for base data request',
            default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
        )

        build_actual_ammo = sdk2.parameters.Bool(
            'Build actual ammo',
            description='Build additional (actual) ammo resource',
            default=False,
            sub_fields={'true': ['actual_query_params']},
        )

        actual_query_params = sdk2.parameters.List(
            'Actual ammo query params',
            description='Additional flags for actual data request',
            default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
        )

        build_dolbilka_plan = sdk2.parameters.Bool(
            'Build plan versions in Dolbilka format',
            description='Gather report-renderer ammo in Dolbilka format (for classic and Apphost both)',
            default=True,
        )

        apphost_format = sdk2.parameters.Bool(
            'Build main plan in Apphost format',
            description='Gather report-renderer ammo in Apphost format',
            default=False,
        )

        build_json_plan = sdk2.parameters.Bool(
            'Build additional plan in JSON format',
            description='Gather report-renderer ammo in JSON format (Incompatible with Apphost format)',
            default=False,
        )

        report_handler_package = sdk2.parameters.String(
            'Report handler package',
            default='@yandex-int/pulse-report-handler@3.9.13',
            required=True,
        )

        ammo_resources_ttl = sdk2.parameters.Integer(
            'Ammo resources TTL (days)',
            default=7,
            required=True,
        )

        with sdk2.parameters.Group('Advanced'):
            _container = parameters.environment_container()

        node_js = parameters.NodeJsParameters

    class Context(sdk2.Context):
        source_table_task_id = None
        source_table_path = None

    @property
    def _failed_bodies_dir(self):
        return os.path.join(os.getcwd(), 'failed-bodies')

    def on_execute(self):
        with self.memoize_stage.validate_params():
            self._validate_params()

        if self.Parameters.use_soy:
            with self.memoize_stage.prepare_soy_table():
                self._prepare_soy_table()

            with self.memoize_stage.get_soy_table():
                self._handle_soy_table()

        with self.memoize_stage.prepare_fs():
            self._prepare_fs()

        with Node(self.Parameters.node_js_version):
            self._run_handler()

        self._save_failed_bodies_if_presented()

        with open(SHOTS_COUNT_PATH, 'r') as shots_count_file:
            count = int(shots_count_file.read())

        threshold = int(self.Parameters.threshold)

        if count < threshold:
            raise TaskFailure('Not enough results for creating plan. {count} instead {threshold}'.format(
                count=count,
                threshold=threshold,
            ))

        attributes = {
            'project': self.Parameters.project,
            'platform': self.Parameters.platform,
            'count': count,
            'ttl': self.Parameters.ammo_resources_ttl,
        }

        self._process_ammo(attributes, pulse_const.AMMO_TYPE_BASE)
        if self.Parameters.build_actual_ammo:
            self._process_ammo(attributes, pulse_const.AMMO_TYPE_ACTUAL)

    def _validate_params(self):
        if self.Parameters.apphost_format and self.Parameters.build_json_plan:
            raise SandboxTaskFailureError('Unable to build JSON plan in Apphost mode')

        if self.Parameters.use_soy and not self.Parameters.apphost_format:
            raise SandboxTaskFailureError('Only Apphost mode is supported with SoY')

    def _prepare_soy_table(self):
        task = PrepareReportRendererPlanSource(
            self,
            access_log_table=self.Parameters.access_log_table,
            report_host=self.Parameters.report_host,
            request_limit=self.Parameters.request_limit,
            base_query_params=self.Parameters.base_query_params,
            build_actual_ammo=self.Parameters.build_actual_ammo,
            actual_query_params=self.Parameters.actual_query_params,
            soy_operation_priority=self.Parameters.soy_operation_priority,
        ).enqueue()

        self.Context.source_table_task_id = task.id

        raise sdk2.WaitTask(
            task,
            (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
        )

    def _handle_soy_table(self):
        source_table_task = sdk2.Task[self.Context.source_table_task_id]
        if source_table_task.status != ctt.Status.SUCCESS:
            msg = 'Preparing SoY source table ended with error: {}\n{}'.format(
                source_table_task.status,
                source_table_task.info
            )
            raise TaskFailure(msg)

        self.Context.source_table_path = source_table_task.Parameters.soy_output_table_path

    def _prepare_fs(self):
        os.makedirs(self._failed_bodies_dir)

    def _access_log(self):
        access_log_id = self.Parameters.access_log

        if access_log_id:
            return str(sdk2.ResourceData(access_log_id).path)

        attributes = {
            'released': ctt.ReleaseStatus.STABLE,
            'platform': self.Parameters.platform,
            'project': self.Parameters.project,
        }

        resource = sdk2.Resource.find(
            resource_type=SearchParsedAccessLog,
            attrs=attributes,
        ).first()

        logging.info('ACCESS_LOG resource - {}'.format(resource))

        if not resource:
            raise TaskFailure('Unable to find "{resource_type}" with attributes "{attributes}"'.format(
                resource_type=SearchParsedAccessLog,
                attributes=attributes,
            ))

        return str(sdk2.ResourceData(resource).path)

    def _process_ammo(self, attributes, ammo_type):
        use_apphost_format = bool(self.Parameters.apphost_format)
        is_actual = ammo_type == pulse_const.AMMO_TYPE_ACTUAL

        self._process_phantom(attributes, use_apphost_format, ammo_type, is_actual)
        self._process_dolbilka(attributes, use_apphost_format, ammo_type, is_actual)
        self._process_json(attributes, ammo_type, is_actual)

    def _process_phantom(self, attributes, use_apphost_format, ammo_type, is_actual):
        res_type = ReportRendererPhantomData
        if use_apphost_format:
            res_type = ReportRendererPhantomDataApphost

        phantom_plan_raw = PHANTOM_PLAN_PATH_MAP[is_actual]
        phantom_plan_compressed = PHANTOM_PLAN_COMPRESSED_PATH_MAP[use_apphost_format, is_actual]

        with open(phantom_plan_raw) as fp_raw, gzip.open(phantom_plan_compressed, 'w') as fp_gz:
            shutil.copyfileobj(fp_raw, fp_gz)

        resource = res_type(
            task=self,
            description=phantom_plan_compressed,
            path=phantom_plan_compressed,
            ammo_type=ammo_type,
            **attributes
        )

        sdk2.ResourceData(resource).ready()

    def _process_dolbilka(self, attributes, use_apphost_format, ammo_type, is_actual):
        if not self.Parameters.build_dolbilka_plan:
            return

        phantom_plan_raw = PHANTOM_PLAN_PATH_MAP[is_actual]
        dolbilka_plan = DOLBILKA_PLAN_PATH_MAP[use_apphost_format, is_actual]

        res_type = ReportRendererPlan
        if use_apphost_format:
            res_type = ReportRendererPlanApphost

        DolbilkaPlanner().create_plan(
            phantom_plan_raw,
            result_path=dolbilka_plan,
            loader_type='phantom'
        )

        resource = res_type(
            task=self,
            description=dolbilka_plan,
            path=dolbilka_plan,
            ammo_type=ammo_type,
            **attributes
        )

        sdk2.ResourceData(resource).ready()

    def _process_json(self, attributes, ammo_type, is_actual):
        if not self.Parameters.build_json_plan:
            return

        json_plan_raw = JSON_PLAN_PATH_MAP[is_actual]
        json_plan_compressed = JSON_PLAN_COMPRESSED_PATH_MAP[is_actual]

        with open(json_plan_raw) as fp_raw, gzip.open(json_plan_compressed, 'w') as fp_gz:
            shutil.copyfileobj(fp_raw, fp_gz)

        resource = ReportRendererPlanJson(
            task=self,
            description=json_plan_compressed,
            path=json_plan_compressed,
            ammo_type=ammo_type,
            **attributes
        )

        sdk2.ResourceData(resource).ready()

    def _run_handler(self):
        run_process(
            (
                'npx %s' % self.Parameters.report_handler_package,
                self._create_config(),
            ),
            shell=True,
            work_dir=os.getcwd(),
            log_prefix='report_handler_executing',
            environment=dict(
                os.environ,
                npm_config_registry='https://npm.yandex-team.ru',
                YT_TOKEN=sdk2.Vault.data('SANDBOX_CI_SEARCH_INTERFACES', 'robot-drunken-flash-yt'),
            ),
        )

    def _create_config(self):
        _, config_file = tempfile.mkstemp()

        project = str(self.Parameters.project)
        platform = str(self.Parameters.platform)

        project_config = pulse_const.ROUTES_CONFIG.get(project, {})
        route_id = project_config.get(platform)
        if not route_id:
            raise SandboxTaskFailureError('There is no route ID for project %s and platform %s' % (project, platform))

        req_info = 'abc:velocityandstability:pulse:%s' % self.id

        base_query_params = pulse.parse_query_params(self.Parameters.base_query_params)
        base_query_params.update({'ammo-type': 'base', 'reqinfo': req_info})

        actual_query_params = None
        if self.Parameters.build_actual_ammo:
            actual_query_params = pulse.parse_query_params(self.Parameters.actual_query_params)
            actual_query_params.update({'ammo-type': 'actual', 'reqinfo': req_info})

        handler = '/' if self.Parameters.apphost_format else '/%s' % route_id
        endpoint_suffix = None

        if self.Parameters.apphost_format:
            apphost_params = pulse_const.APPHOST_PARAMS.get(self.Parameters.project)
            if apphost_params is None:
                raise SandboxTaskFailureError('There is no Apphost support for project %s' % project)

            endpoint_suffix = apphost_params.get('endpoint_suffix')
            apphost_query_params = apphost_params.get('query_params')

            if apphost_query_params:
                base_query_params.update(apphost_query_params)
                if actual_query_params:
                    actual_query_params.update(apphost_query_params)

        config = {
            'queryParams': base_query_params,
            'project': project,
            'platform': platform,
            'routeId': route_id,
            'handler': handler,
            'endpointSuffix': endpoint_suffix,
            'reportHost': 'https://' + str(self.Parameters.report_host),
            'parallelRequestLimit': int(self.Parameters.parallel_request_limit),
            'apphost': bool(self.Parameters.apphost_format),
            'hasPreSearch': project_config.get('_config', {}).get('has_pre_search', False),
            'buildJsonPlan': bool(self.Parameters.build_json_plan),
            'saveFailedBodies': True,
            'failedBodiesDir': self._failed_bodies_dir,
        }
        config.update(REPORT_HANDLER_CONSTANTS_BASE)

        expected_shots_count = project_config.get('_config', {}).get(platform, {}).get('expected_shots_count', 0)
        if expected_shots_count:
            config['expectedShotsCount'] = expected_shots_count

        if self.Parameters.request_limit:
            config['requestLimit'] = int(self.Parameters.request_limit)

        if self.Parameters.use_soy:
            config.update({
                'readerMode': True,
                'sourceTable': self.Context.source_table_path,
            })
        else:
            config.update({
                'sourceLog': self._access_log()
            })

        if actual_query_params:
            actual_config = {'queryParams': actual_query_params}
            actual_config.update(REPORT_HANDLER_CONSTANTS_ACTUAL)

            config.update({'additionalAmmo': {'enabled': True, 'configs': (actual_config,)}})

        logging.debug('Report handler config: %s', config)
        with open(config_file, 'w') as fp:
            json.dump(config, fp, indent=2)

        return config_file

    def _save_failed_bodies_if_presented(self):
        failed_bodies_count = len(os.listdir(self._failed_bodies_dir))

        if not failed_bodies_count:
            logging.info('No failed bodies, so do not save')
            return

        logging.info('Have %s failed bodies. Create resource', failed_bodies_count)

        failed_bodies_archive = os.path.join(os.getcwd(), 'failed-bodies.tar.gz')
        with tarfile.open(failed_bodies_archive, 'w:gz') as tar:
            tar.add(self._failed_bodies_dir)

        resource = ReportRendererApphostFailedBodies(
            self,
            description='Report Renderer Apphost failed bodies',
            path=failed_bodies_archive,
            count=failed_bodies_count,
        )

        sdk2.ResourceData(resource).ready()
