#  -*- coding: utf-8 -*-
import json
import logging

from uuid import uuid4

from sandbox import sdk2
import sandbox.projects.sandbox_ci.pulse.utils.yql as yql_utils

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property

from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.release_machine.helpers.soy_helper import SoyApi
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.pulse import const as pulse_const
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_params
from sandbox.projects.sandbox_ci.pulse.prepare_report_renderer_plan_source.sql import build_soy_table_query

SOY_OPERATION_PRIORITIES = {
    'low': -1000,
    'normal': 0,
    'high': 1000,
}


class PrepareReportRendererPlanSource(TasksResourceRequirement, sdk2.Task):
    class Parameters(sdk2.Parameters):
        access_log_table = sdk2.parameters.String(
            'Access log YT table path',
            required=True,
        )

        report_host = sdk2.parameters.String(
            'Report host',
            description='Host for data request',
            default='hamster.yandex.ru'
        )

        request_limit = sdk2.parameters.Integer(
            'Request limit',
            default=0,
        )

        base_query_params = sdk2.parameters.List(
            'Base ammo query params',
            description='Additional flags for base data request',
            default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
        )

        build_actual_ammo = sdk2.parameters.Bool(
            'Build actual ammo',
            description='Build additional (actual) ammo resource',
            default=False,
            sub_fields={'true': ['actual_query_params']},
        )

        actual_query_params = sdk2.parameters.List(
            'Actual ammo query params',
            description='Additional flags for actual data request',
            default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
        )

        soy_pool = sdk2.parameters.String(
            'SoY operation pool',
            default='velocity',
            required=True,
        )

        soy_operation_priority = pulse_params.soy_operation_priority()

        with sdk2.parameters.Output():
            soy_output_table_path = sdk2.parameters.String('SoY output table')

    class Context(sdk2.Context):
        soy_source_table_task_id = None
        soy_source_table_path = None
        soy_job_id = None
        soy_job_url = None
        soy_started = False

    @singleton_property
    def soy(self):
        return SoyApi(token=sdk2.Vault.data('SANDBOX_CI_SEARCH_INTERFACES', 'robot-drunken-flash-soy'))

    @property
    def soy_operation_priority(self):
        return SOY_OPERATION_PRIORITIES.get(self.Parameters.soy_operation_priority, 0)

    @sdk2.header()
    def header(self):
        if not self.Context.soy_started:
            return 'Waiting for SoY operation start...'

        return 'SoY operation: <a href={url}>{id}</a>'.format(
            id=self.Context.soy_job_id,
            url=self.Context.soy_job_url,
        )

    def on_execute(self):
        with self.memoize_stage.prepare_soy_table(commit_on_entrance=False):
            self._prepare_soy_table()

        with self.memoize_stage.run_soy(commit_on_entrance=False):
            self._check_subtask_status(self.Context.soy_source_table_task_id, 'RUN_YQL_2')
            self._run_soy()

        with self.memoize_stage.wait_soy(commit_on_entrance=False, commit_on_wait=False):
            self._wait_soy()

    def on_failure(self, prev_status):
        self._abort_soy()
        super(PrepareReportRendererPlanSource, self).on_failure(prev_status)

    def on_timeout(self, prev_status):
        self._abort_soy()
        super(PrepareReportRendererPlanSource, self).on_timeout(prev_status)

    def on_break(self, *args, **kwargs):
        self._abort_soy()
        super(PrepareReportRendererPlanSource, self).on_break(*args, **kwargs)

    def on_terminate(self):
        self._abort_soy()
        super(PrepareReportRendererPlanSource, self).on_terminate()

    def _prepare_soy_table(self):
        output_table_path = self.Context.soy_source_table_path = '//tmp/robot-drunken-flash/' + str(uuid4())

        query = build_soy_table_query(
            input_table=self.Parameters.access_log_table,
            output_table=output_table_path,
            use_batch=self.Parameters.build_actual_ammo,
            report_host=self.Parameters.report_host,
            base_query_params=self.Parameters.base_query_params,
            actual_query_params=self.Parameters.actual_query_params
        )

        subtask = RunYQL2(
            self,
            owner=yql_utils.YQL_TOKEN_OWNER,
            kill_timeout=self.Parameters.kill_timeout,
            description='Build SoY source table\n' + self.Parameters.description,
            query=query,
            yql_token_vault_name=yql_utils.YQL_TOKEN_NAME,
            use_v1_syntax=True,
            publish_query=True,
            publish_download_link=True,
            public_download_link=True,
            download_format='JSON',
            trace_query=True,
            retry_period=60,
        ).enqueue()

        self.Context.soy_source_table_task_id = subtask.id

        self._wait_task(subtask)

    def _run_soy(self):
        self.Context.soy_job_id = str(uuid4())

        description = 'Build report renderer plan source for task #{}. '.format(self.id)

        if self.Parameters.base_query_params:
            description += 'Base query params: {}. '.format(json.dumps(self.Parameters.base_query_params))

        if self.Parameters.build_actual_ammo and self.Parameters.actual_query_params:
            description += 'Actual query params: {}. '.format(json.dumps(self.Parameters.actual_query_params))

        res = self.soy.create({
            'id': self.Context.soy_job_id,
            'input_table': self.Context.soy_source_table_path,
            'pool': self.Parameters.soy_pool,
            'priority': self.soy_operation_priority,
            'batched_requests': self.Parameters.build_actual_ammo,
            'description': description,
        })

        if res.get('status') == 'ok':
            self.Context.soy_started = True

        if res.get('status') == 'error':
            raise TaskFailure('SoY job starting failed with error: {}'.format(res.get('error_msg')))

    def _abort_soy(self):
        if not self.Context.soy_job_url:
            return

        self.soy.abort(self.Context.soy_job_id)

    def _wait_task(self, task):
        raise sdk2.WaitTask(
            task,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            timeout=self.Parameters.kill_timeout,
            wait_all=True,
        )

    def _wait_soy(self):
        res = self.soy.status(self.Context.soy_job_id)

        self.Context.soy_job_url = res.get('UI')

        operation_status = res.get('simplified_operation_status')

        if operation_status not in ('completed', 'canceled', 'failed'):
            return self._sleep(300)
        elif operation_status in ('canceled', 'failed'):
            raise TaskFailure('SoY error')

        self.Parameters.soy_output_table_path = res.get('output_path')

    def _check_subtask_status(self, task_id, label):
        if not task_id:
            return logging.warn('Wrong task ID for check: %s', task_id)

        subtask = sdk2.Task[task_id]

        if subtask.status != ctt.Status.SUCCESS:
            raise TaskFailure(
                'Subtask %s ended with error: %s\n%s' % (
                    label,
                    subtask.status,
                    subtask.info,
                )
            )

    def _sleep(self, time=60):
        raise sdk2.WaitTime(time)
