#  -*- coding: utf-8 -*-
import json

import sandbox.projects.sandbox_ci.pulse.utils.yql as yql_utils

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.sandbox_ci.pulse import build_url_path, parse_query_params
from sandbox.projects.sandbox_ci.pulse.prepare_ticount_plan.sql import build_query
from sandbox.projects.sandbox_ci.pulse.resources import TicountPlan
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.yql.RunYQL2 import RunYQL2


class PrepareTicountPlan(TasksResourceRequirement, sdk2.Task):
    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String(
            'Project',
            default='web4',
            required=True,
        )

        platform = sdk2.parameters.String(
            'Platform',
            default='desktop',
            required=True,
        )

        requests_number = sdk2.parameters.Integer(
            'Requests number',
            default=100,
            required=True,
        )

        query_params = sdk2.parameters.List(
            'Query params',
            default=[
                'no-tests=1',
                'promo=nomooa',
            ]
        )

        release_plan = sdk2.parameters.Bool(
            'Release plan',
            default=False,
        )

    class Context(sdk2.Context):
        yql_query = None
        yql_task_id = None

    def on_prepare(self):
        self.Context.yql_query = build_query(
            project=self.Parameters.project,
            platform=self.Parameters.platform,
            limit=self.Parameters.requests_number,
        )

    def on_execute(self):
        with self.memoize_stage.fetch_requests(commit_on_entrance=False):
            self._fetch_requests()

        with self.memoize_stage.prepare_plan:
            self._prepare_plan()

        if self.Parameters.release_plan:
            with self.memoize_stage.release_plan:
                self.server.release(
                    params=dict(
                        task_id=self.id,
                        type='stable',
                        subject='Ticount plan release',
                    )
                )

    def _fetch_requests(self):
        subtask = RunYQL2(
            self,
            owner=yql_utils.YQL_TOKEN_OWNER,
            kill_timeout=self.Parameters.kill_timeout,
            description=self.Parameters.description,
            query=self.Context.yql_query,
            yql_token_vault_name=yql_utils.YQL_TOKEN_NAME,
            use_v1_syntax=True,
            publish_query=True,
            publish_download_link=True,
            public_download_link=True,
            download_format='JSON',
            trace_query=True,
            retry_period=60,
        ).enqueue()

        self.Context.yql_task_id = subtask.id

        raise sdk2.WaitTask(
            subtask,
            Status.Group.FINISH | Status.Group.BREAK,
            timeout=self.Parameters.kill_timeout,
            wait_all=True,
        )

    def _prepare_plan(self):
        task = sdk2.Task[self.Context.yql_task_id]

        if task.status != Status.SUCCESS:
            raise TaskFailure('YQL failed')

        yql_task_result_url = task.Parameters.full_results_download_url or task.Parameters.results_sample_download_url
        yql_task_result_file = 'yql_result'
        yql_utils.download_from_yql(yql_task_result_url, yql_task_result_file)

        plan_file = 'requests.plan'
        query_params_dict = parse_query_params(self.Parameters.query_params)
        with open(yql_task_result_file) as fd, open(plan_file, 'w') as wd:
            for item in fd:
                text = json.loads(item).get('text')
                url = build_url_path(
                    self.Parameters.project,
                    self.Parameters.platform,
                    dict(text=text.encode('utf8'), **query_params_dict),
                )
                wd.write(url + '\n')

        resource = TicountPlan(
            task=self,
            description=plan_file,
            path=plan_file,
            project=self.Parameters.project,
            platform=self.Parameters.platform,
            project_type=self.Parameters.platform,
        )

        sdk2.ResourceData(resource).ready()
