# coding=utf-8
import json
import logging
import os
import re
import shutil
import stat
import tarfile
from collections import namedtuple
from datetime import datetime
from subprocess import Popen
from threading import Timer

from sandbox import sdk2
from sandbox.agentr.errors import InvalidResource
from sandbox.common.utils import singleton_property, classproperty
from sandbox.common.types import client as ctc
from sandbox.projects.report_renderer.resource_types import REPORT_RENDERER_BUNDLE
from sandbox.projects.sandbox_ci.pulse import utils
from sandbox.projects.sandbox_ci.pulse.const import (
    PROJECTS_PLATFORMS_WITH_PULSE, ROUTES_CONFIG, PULSE_STAT_LINKS, PULSE_WIKI_LINKS,
    VELOCITY_EMAIL, FIJI_VELOCITY_EMAIL, PULSE_BINARY_SHOOTER, PULSE_BINARY_AGGREGATOR, PULSE_BINARY_DIFF_VIEWER,
    PULSE_BINARY_REPORT, PULSE_BINARY_HTML_DIFFER, PULSE_BINARY_MEMORY_LEAKS_DETECTOR,
    PULSE_BINARY_BLOCKSTAT_LOG_ANALYZER, PROJECTS_WITH_HTML_DIFFER_FILTRATION_SUPPORT,
)
from sandbox.projects.sandbox_ci.pulse.resource_finder import ResourceTemplatesFinder
from sandbox.projects.sandbox_ci.pulse.resources import (
    PulseBinary, PulseShooterLogs,
    PulseShooterResponses, ReportRendererPhantomData, ReportRendererPhantomDataApphost,
)
from sandbox.projects.sandbox_ci.pulse import config as pulse_config
from sandbox.projects.sandbox_ci.pulse.statface import PulseShooterStatfaceData
from sandbox.projects.sandbox_ci.task import BasePulseTask
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

APPHOST_TAG = 'APPHOST'
CUSTOM_ROUTE_ID_TAG = 'CUSTOM_ROUTES_ID'
COREDUMP_FILTER_PATTERN = '(pulse-|ynode).*'

N_PULSE_SHOOTER_BIN = PULSE_BINARY_SHOOTER
N_PULSE_AGGREGATOR_BIN = PULSE_BINARY_AGGREGATOR
N_PULSE_REPORT_BIN = PULSE_BINARY_REPORT
N_PULSE_DIFFER_BIN = PULSE_BINARY_HTML_DIFFER
N_PULSE_DIFF_VIEWER = PULSE_BINARY_DIFF_VIEWER
N_PULSE_MEMORY_LEAKS_DETECTOR_BIN = PULSE_BINARY_MEMORY_LEAKS_DETECTOR
N_PULSE_BLOCKSTAT_LOG_ANALYZER_BIN = PULSE_BINARY_BLOCKSTAT_LOG_ANALYZER

N_TEMPLATES_BASE = 'templates-base'
N_TEMPLATES_ACTUAL = 'templates-actual'
N_RR_BUNDLE_BASE = 'report-renderer-bundle-base'
N_RR_BUNDLE_ACTUAL = 'report-renderer-bundle-actual'
N_AMMO_BASE = 'ammo-base'
N_AMMO_ACTUAL = 'ammo-actual'

TEMPLATES_CTX_MAP = {
    N_TEMPLATES_BASE: ('base', 'base_route_path'),
    N_TEMPLATES_ACTUAL: ('actual', 'actual_route_path'),
}

RR_CTX_MAP = {
    N_RR_BUNDLE_BASE: ('rr_base', 'rr_base_id', 'ynode_base_path', 'rr_base_path'),
    N_RR_BUNDLE_ACTUAL: ('rr_actual', 'rr_actual_id', 'ynode_actual_path', 'rr_actual_path'),
}

PULSE_BINARIES_CTX_MAP = {
    N_PULSE_SHOOTER_BIN: ('final_pulse_shooter_id', 'pulse_shooter_path'),
    N_PULSE_AGGREGATOR_BIN: ('final_pulse_aggregator_id', 'pulse_aggregator_path'),
    N_PULSE_DIFFER_BIN: ('final_html_differ_id', 'html_differ_path'),
    N_PULSE_REPORT_BIN: ('final_pulse_report_id', 'pulse_report_path'),
    N_PULSE_DIFF_VIEWER: ('final_diff_viewer_id', 'diff_viewer_path'),
    N_PULSE_MEMORY_LEAKS_DETECTOR_BIN: ('final_memory_leaks_detector_id', 'memory_leaks_detector_path'),
    N_PULSE_BLOCKSTAT_LOG_ANALYZER_BIN: ('final_blockstat_log_analyzer_id', 'blockstat_log_analyzer_path'),
}

AMMO_CTX_MAP = {
    N_AMMO_BASE: ('ammo_base_id', 'ammo_base_path'),
    N_AMMO_ACTUAL: ('ammo_actual_id', 'ammo_actual_path'),
}

REQUIRED_CTX_FIELDS = (
    'base_route_id', 'actual_route_id',
    'rr_base_id', 'ynode_base_path', 'rr_base_path',
    'rr_actual_id', 'ynode_actual_path', 'rr_actual_path',
    'final_pulse_shooter_id', 'pulse_shooter_path',
    'final_pulse_aggregator_id', 'pulse_aggregator_path',
    'final_html_differ_id', 'html_differ_path',
    'final_pulse_report_id', 'pulse_report_path',
    'final_diff_viewer_id', 'diff_viewer_path',
    'final_memory_leaks_detector_id', 'memory_leaks_detector_path',
    'final_blockstat_log_analyzer_id', 'blockstat_log_analyzer_path',
    'ammo_base_id', 'ammo_base_path',
)

MEMORY_LEAKS_DETECTOR_MEASUREMENTS = (
    'HEAP_TOTAL',
    'HEAP_USED',
    'RESIDENT_SET',
    'EXTERNAL'
)

ResourceFetchByIdParams = namedtuple('ResourceFetchByIdParams', ('fetch_func', 'existing_value'))
ResourceFetchByAttrsParams = namedtuple('ResourceFetchParams', ('resource_type', 'attrs', 'existing_value'))
ResourceFetchResultRaw = namedtuple('ResourceFetchResultRaw', ('resource_name', 'resource', 'resource_data'))
ResourceFetchResult = namedtuple('ResourceFetchResult', ('resource', 'resource_data'))


PULSE_ERROR_CODE = 3
REPORT_RENDERER_ERROR_CODE = 4


class PulseError(SandboxTaskFailureError):
    pass


class ReportRendererError(SandboxTaskFailureError):
    pass


class PulseTemplateError(SandboxTaskFailureError):
    pass


class PulseBlockstatLogAnalyzerError(SandboxTaskFailureError):
    pass


class PulseShooter(BasePulseTask):
    """
    Measure templating time for Report Renderer
    """

    class Requirements(BasePulseTask.Requirements):
        disk_space = 40 * 1024
        client_tags = ctc.Tag.INTEL_E5_2650

    class Context(BasePulseTask.Context):
        report_html = 'Report is not ready yet'
        html_diff_report_link = ''
        html_filtered_diff_report_link = ''

        base_route_path = None
        actual_route_path = None

        final_pulse_shooter_id = None
        final_pulse_aggregator_id = None
        final_pulse_report_id = None
        final_html_differ_id = None
        final_diff_viewer_id = None
        final_memory_leaks_detector_id = None

        pulse_shooter_path = None
        pulse_aggregator_path = None
        pulse_report_path = None
        html_differ_path = None
        diff_viewer_path = None
        memory_leaks_detector_path = None

        rr_base_id = None
        rr_actual_id = None

        ynode_base_path = None
        rr_base_path = None
        ynode_actual_path = None
        rr_actual_path = None

        ammo_base_id = None
        ammo_base_path = None
        ammo_actual_id = None
        ammo_actual_path = None

        was_timeout = False

    class Parameters(BasePulseTask.Parameters):
        kill_timeout = 90 * 60

        with sdk2.parameters.Group('Templates') as templates_block:
            use_custom_routes_id = sdk2.parameters.Bool(
                'Use custom routes id',
                default=False,
            )

            with use_custom_routes_id.value[True]:
                custom_routes_id = sdk2.parameters.String(
                    'Routes id',
                    default="",
                    required=False,
                )

            with use_custom_routes_id.value[False]:
                service = sdk2.parameters.String('Service', required=False)

                with sdk2.parameters.String('Platform') as platform:
                    _unique_names = set()
                    for _platforms_list in PROJECTS_PLATFORMS_WITH_PULSE.itervalues():
                        _unique_names.update(_platforms_list)
                    for _i, _platform_name in enumerate(sorted(_unique_names)):
                        platform.values[_platform_name] = platform.Value(_platform_name, default=(_i == 0))

            base_templates_package = sdk2.parameters.Resource(
                'Base templates package',
            )

            actual_templates_package = sdk2.parameters.Resource(
                'Actual templates package',
                required=True,
            )

        with sdk2.parameters.Group('Shooting components') as shooting_components:
            rr_base = sdk2.parameters.Resource(
                'Report renderer base',
                required=False,
                resource_type=REPORT_RENDERER_BUNDLE,
                default=None,
            )

            rr_actual = sdk2.parameters.Resource(
                'Report renderer actual',
                required=False,
                resource_type=REPORT_RENDERER_BUNDLE,
                default=None,
            )

            pulse_shooter_ammo_base = sdk2.parameters.Resource(
                'Ammo base',
                required=False,
                resource_type=[
                    ReportRendererPhantomData,
                    ReportRendererPhantomDataApphost,
                ],
                default=None,
            )

            pulse_shooter_ammo_actual = sdk2.parameters.Resource(
                'Ammo actual',
                required=False,
                resource_type=[
                    ReportRendererPhantomData,
                    ReportRendererPhantomDataApphost,
                ],
                default=None,
            )

        with sdk2.parameters.Group('Template flags') as template_flags_block:
            flags_base = sdk2.parameters.Dict('Base template flags')
            flags_actual = sdk2.parameters.Dict('Actual template flags')
            additional_timing_on = sdk2.parameters.Bool(
                'Enable additional timing',
                description='Измерить время выполнения адаптеров',
                default=False,
            )

        with sdk2.parameters.Group('Shooting params') as shooting_params_block:
            pulse_shooter_binary = sdk2.parameters.Resource(
                'Pulse Shooter binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            pulse_aggregator_binary = sdk2.parameters.Resource(
                'Pulse Aggregator binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            pulse_report_binary = sdk2.parameters.Resource(
                'Pulse Report binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            html_differ_binary = sdk2.parameters.Resource(
                'HTML Differ binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            diff_viewer_binary = sdk2.parameters.Resource(
                'Diff Viewer binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            memory_leaks_detector_binary = sdk2.parameters.Resource(
                'Memory leaks detector binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            blockstat_log_analyzer_binary = sdk2.parameters.Resource(
                'Blockstat log analyzer binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

            detect_memory_leaks = sdk2.parameters.Bool(
                'Detect memory leaks (Slow)',
                default=False,
            )

            heap_snapshots_enabled = sdk2.parameters.Bool(
                'Take heap snapshots',
                default=False,
                description='Сохранять дампы памяти',
            )

            with heap_snapshots_enabled.value[True]:
                heap_snapshots_count = sdk2.parameters.Integer(
                    'Snapshots during main shooting',
                    default=3,
                )

            rr_workers = sdk2.parameters.Integer(
                'RR workers',
                default=10,
                required=True,
            )

            pulse_shooter_workers = sdk2.parameters.Integer(
                'Pulse Shooter workers',
                default=10,
                required=True,
            )

            threshold_percentile = sdk2.parameters.Float(
                'Threshold percentile',
                default=90,
                required=True,
            )

            threshold_constant = sdk2.parameters.Float(
                'Threshold constant',
                default=80,
                required=True,
            )

            threshold_significance = sdk2.parameters.Float(
                'Significance threshold (MW test)',
                default=99.9,
                required=True,
            )

            with detect_memory_leaks.value[True]:
                request_limit_with_memory_capture = sdk2.parameters.Integer(
                    'Requests to shoot',
                    default=50000,
                    required=True,
                )

            with detect_memory_leaks.value[False]:
                request_limit = sdk2.parameters.Integer(
                    'Requests to shoot',
                    default=5000,
                    required=True,
                )

            with detect_memory_leaks.value[True]:
                memory_capture_requests_per_measurement = sdk2.parameters.Integer(
                    'Requests per measurement',
                    default=10
                )

                memory_capture_gc_frequency = sdk2.parameters.Integer(
                    'Run GC on every Nth request',
                    default=0
                )

            generate_blockstat_report = sdk2.parameters.Bool(
                'Aggregate Blockstat logs and save blocks statistics',
                default=False,
            )

            with generate_blockstat_report.value[True]:
                blockstat_report_filter = sdk2.parameters.String(
                    'Counter for Blockstat report filtering',
                    default='',
                )

            apphost_mode = sdk2.parameters.Bool(
                'Apphost mode',
                default=False,
            )

            with apphost_mode.value[True]:
                ahproxy = sdk2.parameters.Bool(
                    'Enable ahproxy',
                    default=True,
                )

            profile_v8 = sdk2.parameters.Bool(
                'Enable v8 profiling',
                description='Generate v8.log with profile events',
                default=False,
            )

            profile_v8_txt = sdk2.parameters.Bool(
                'Generate v8 profile.txt',
                description='Generate human readable profile when v8.log presented',
                default=False,
            )

            allow_natives_syntax = sdk2.parameters.Bool(
                'Allow natives syntax',
                description='Allow natives V8 syntax in ynode',
                default=False,
            )

            no_opt_base = sdk2.parameters.Bool(
                'Disable TurboFan for base',
                description='Disable TurboFan optimizations for base ynode',
                default=False,
            )

            no_opt_actual = sdk2.parameters.Bool(
                'Disable TurboFan for actual',
                description='Disable TurboFan optimizations for actual ynode',
                default=False,
            )

            trace_opt = sdk2.parameters.Bool(
                'Trace TurboFan optimizations',
                description='Trace TurboFan optimizations to ynode-out.log',
                default=False,
            )

            trace_deopt = sdk2.parameters.Bool(
                'Trace TurboFan deoptimizations',
                description='Trace TurboFan deoptimizations to ynode-out.log',
                default=False,
            )

        with sdk2.parameters.Output():
            results = sdk2.parameters.JSON('Results')

    @singleton_property
    def cache_parameters(self):
        cache_params = super(PulseShooter, self).cache_parameters
        task_params = self.Parameters
        cache_params.update(
            platform=task_params.platform,
            request_limit=task_params.request_limit,
            rr_base=None,
            rr_actual=None,
            pulse_shooter_ammo_base=None,
            pulse_shooter_ammo_actual=None,
        )

        if task_params.use_custom_routes_id:
            cache_params.update(
                custom_routes_id=task_params.custom_routes_id,
            )
        else:
            cache_params.update(
                project=self.project_name,
            )

        if task_params.service:
            cache_params['service'] = task_params.service
        if task_params.pulse_shooter_ammo_base:
            cache_params['pulse_shooter_ammo_base'] = task_params.pulse_shooter_ammo_base.id
        if task_params.pulse_shooter_ammo_actual:
            cache_params['pulse_shooter_ammo_actual'] = task_params.pulse_shooter_ammo_actual.id
        if task_params.rr_base:
            cache_params['rr_base'] = task_params.rr_base.id
        if task_params.rr_actual:
            cache_params['rr_actual'] = task_params.rr_actual.id

        return cache_params

    @classproperty
    def task_label(self):
        return 'pulse.shooter'

    @property
    def project_name(self):
        return self.Parameters.project

    @property
    def service_platform(self):
        if self.Parameters.service and not self.Parameters.platform.startswith(self.Parameters.service):
            return self.format_service_platform(self.Parameters.service, self.Parameters.platform)

        return self.Parameters.platform

    @property
    def email_to_notify(self):
        if self.Parameters.use_custom_routes_id:
            return None
        if self.project_name == 'fiji':
            return FIJI_VELOCITY_EMAIL
        return VELOCITY_EMAIL

    @property
    def _report(self):
        return self.Context.report_html

    @singleton_property
    def _resource_templates_finder(self):
        base_task = self.parent

        if not self.parent or self.parent.type.name == 'PULSE_SHOOTER_CUSTOM':
            base_task = self.Parameters.actual_templates_package.task

        return ResourceTemplatesFinder(base_task)

    @staticmethod
    def format_service_platform(service, platform):
        return '{}-{}'.format(service, platform)

    @staticmethod
    def format_platform(platform):
        return platform.replace('-', '/', 1)

    @classmethod
    def format_github_context(cls, platform):
        return u'Pulse Shooter: {}'.format(platform)

    @property
    def github_context(self):
        platform = self.Parameters.platform
        if self.Parameters.service:
            platform = '{}/{}'.format(self.Parameters.service, self.Parameters.platform)
        return self.format_github_context(platform)

    @property
    def rr_workers(self):
        if self.Parameters.detect_memory_leaks or self.Parameters.heap_snapshots_enabled:
            return 1
        return self.Parameters.rr_workers

    @property
    def pulse_shooter_workers(self):
        if self.Parameters.heap_snapshots_enabled:
            return 1
        return self.Parameters.pulse_shooter_workers

    @property
    def is_html_differ_filtration_supported(self):
        return self.project_name in PROJECTS_WITH_HTML_DIFFER_FILTRATION_SUPPORT

    def on_enqueue(self):
        super(PulseShooter, self).on_enqueue()
        self._add_tags()

    def on_success(self, prev_status):
        super(PulseShooter, self).on_success(prev_status)

        tags = list(self.Parameters.tags)
        if 'PULSE_ERROR' in tags:
            tags.remove('PULSE_ERROR')

        if 'REPORT_RENDERER_ERROR' in tags:
            tags.remove('REPORT_RENDERER_ERROR')

        if 'TEMPLATE_ERROR' in tags:
            tags.remove('TEMPLATE_ERROR')

        self.Parameters.tags = tags

    def execute(self):
        logging.info('Enter Pulse Shooter')

        try:
            self._init()

            with self.profiler.actions.messages('Messages'):
                if self.Parameters.pulse_shooter_ammo_actual:
                    self.set_info(
                        (
                            'Using two shooting baskets can produce less stable results, ' +
                            'but necessary for "control vs experiment" comparisons (ABT, flags.json, data-dependent flags).\n' +
                            'Please check <a href="{}">documentation</a>.'
                        ).format(PULSE_WIKI_LINKS.get('two_shooting_baskets')),
                        do_escape=False,
                    )

            with self.profiler.actions.preparation('Resource fetching'):
                self._fetch_resources()

            with self.profiler.actions.shooting('Shooting'):
                self._run_pulse_shooter()

            with self.profiler.actions.check_template_errors('Check template errors'):
                self._check_exist_errors()

            with self.profiler.actions.aggregation('Aggregation'):
                self._run_pulse_aggregator()

            with self.profiler.actions.build_html_diff('Build HTML diff'):
                self._run_html_differ()

                if self.is_html_differ_filtration_supported and self.Parameters.pulse_shooter_ammo_actual:
                    self._run_html_differ(use_blockstat_filter=True)

            if self.Parameters.detect_memory_leaks:
                with self.profiler.actions.memory_leaks_detection('Memory leaks detection'):
                    self._run_memory_leaks_detector()

            if self.Parameters.generate_blockstat_report:
                with self.profiler.actions.analyze_blockstat_logs('Analyze Blockstat logs'):
                    self._run_blockstat_log_analyzer()

            with self.profiler.actions.report_building('Report building'):
                self._run_pulse_report()

        except ReportRendererError as ex:
            self.Parameters.tags = self.Parameters.tags + ['REPORT_RENDERER_ERROR']
            self.set_info(
                'Report renderer error occured, <a href="{}">rr-log</a>'.format(
                    self.Context.rr_errors_link), do_escape=False)
            self.set_info('<hr/>', do_escape=False)
            raise ex

        except PulseError as ex:
            self.Parameters.tags = self.Parameters.tags + ['PULSE_ERROR']
            self.set_info(
                'Pulse error occured, <a href="{}">pulse-shooter-log</a>'.format(
                    self.Context.pulse_shooter_errors_link), do_escape=False)
            raise ex

        except PulseTemplateError as ex:
            self.Parameters.tags = self.Parameters.tags + ['TEMPLATE_ERROR']
            self.set_info(
                'Template error occured, template-errors-{base,actual}.log in header',
                do_escape=False
            )
            raise ex

        finally:
            self._build_report()

        with self.profiler.actions.check_limits('Check limits'):
            self.check_exceeded_limits()

        if self.should_report_to_stat:
            with self.profiler.actions.report_to_stat('Report to Stat'):
                self._send_statface_report_safe()

        logging.info('Exit Pulse Shooter')

    def _init(self):
        self.base_pulse_shooter_results = None
        self.actual_pulse_shooter_results = None

        self.pulse_aggregator_results = None
        self.pulse_report_result = None
        self.memory_leaks_detector_result = (None, None)

    def _fetch_resources(self):
        resources_to_fetch = self._get_resources_to_fetch()

        fetch_results = {}
        for result in flow.parallel(lambda args: self._fetch_resource(*args), resources_to_fetch.items()):
            fetch_results[result.resource_name] = ResourceFetchResult(result.resource, result.resource_data)

        for resource_name, result in fetch_results.iteritems():
            self._process_if_template_resource(resource_name, result)
            self._process_if_rr_resource(resource_name, result)
            self._process_if_pulse_binary_resource(resource_name, result)
            self._process_if_ammo_resource(resource_name, result)

        self._adjust_rr_instances()
        self._validate_resource_ctx_fields()

    def _get_resources_to_fetch(self):
        params = self.Parameters

        platform = params.platform
        if params.service:
            platform = self.format_service_platform(params.service, platform)

        ammo_resource_type = ReportRendererPhantomData
        if self.Parameters.apphost_mode:
            ammo_resource_type = ReportRendererPhantomDataApphost

        resources_to_fetch = {
            N_TEMPLATES_BASE: ResourceFetchByIdParams(
                lambda: self._resource_templates_finder.base_dynamic_resource_id(),
                params.base_templates_package,
            ),
            N_TEMPLATES_ACTUAL: ResourceFetchByIdParams(
                lambda: self._resource_templates_finder.actual_dynamic_resource_id(),
                params.actual_templates_package,
            ),
            N_PULSE_SHOOTER_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_SHOOTER},
                params.pulse_shooter_binary,
            ),
            N_PULSE_AGGREGATOR_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_AGGREGATOR},
                params.pulse_aggregator_binary,
            ),
            N_PULSE_REPORT_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_REPORT},
                params.pulse_report_binary,
            ),
            N_PULSE_DIFFER_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_HTML_DIFFER},
                params.html_differ_binary,
            ),
            N_PULSE_DIFF_VIEWER: ResourceFetchByAttrsParams(
                PulseBinary,
                {'project': PULSE_BINARY_DIFF_VIEWER, 'platform': 'all', 'released': 'stable'},
                params.diff_viewer_binary
            ),
            N_RR_BUNDLE_BASE: ResourceFetchByAttrsParams(
                REPORT_RENDERER_BUNDLE,
                {'released': 'stable', 'platform': 'linux'},
                params.rr_base,
            ),
            N_AMMO_BASE: ResourceFetchByAttrsParams(
                ammo_resource_type,
                {'project': self.Parameters.project, 'platform': platform, 'released': 'stable'},
                self.Parameters.pulse_shooter_ammo_base,
            ),
            N_PULSE_MEMORY_LEAKS_DETECTOR_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_MEMORY_LEAKS_DETECTOR},
                params.memory_leaks_detector_binary,
            ),
            N_PULSE_BLOCKSTAT_LOG_ANALYZER_BIN: ResourceFetchByAttrsParams(
                PulseBinary,
                {'platform': 'linux', 'released': 'stable', 'project': PULSE_BINARY_BLOCKSTAT_LOG_ANALYZER},
                params.blockstat_log_analyzer_binary,
            ),
        }

        if (
            (params.rr_base or params.rr_actual) and
            getattr(params.rr_base, 'id', None) != getattr(params.rr_actual, 'id', None)
        ):
            resources_to_fetch[N_RR_BUNDLE_ACTUAL] = ResourceFetchByAttrsParams(
                REPORT_RENDERER_BUNDLE,
                {'released': 'stable', 'platform': 'linux'},
                params.rr_actual,
            )

        if params.pulse_shooter_ammo_actual:
            if params.pulse_shooter_ammo_actual.task.id != params.pulse_shooter_ammo_base.task.id:
                raise SandboxTaskFailureError(
                    'When you use double ammo scheme, '
                    'you should use ammo collected by same task'
                )

            resources_to_fetch[N_AMMO_ACTUAL] = ResourceFetchByIdParams(lambda: None, params.pulse_shooter_ammo_actual)

        return resources_to_fetch

    def _fetch_resource(self, res_name, fetch_params):
        logging.info('Start fetching resource with params %s', fetch_params)

        if isinstance(fetch_params, ResourceFetchByIdParams):
            logging.debug('Fetching by ID resource with params %s', fetch_params)
            result = self._fetch_resource_by_id(res_name, fetch_params)
        else:
            logging.debug('Fetching by attrs resource with params %s', fetch_params)
            result = self._fetch_resource_by_attrs(res_name, fetch_params)

        logging.info('Finish fetching resource with params %s', fetch_params)
        return result

    def _fetch_resource_by_id(self, res_name, fetch_params):
        existing_value = fetch_params.existing_value

        resource = None
        if isinstance(existing_value, int):
            resource = sdk2.Resource[existing_value]
        elif existing_value:
            resource = existing_value

        logging.debug('Resource by ID existing value: %s', resource)

        if not resource:
            resource_id = fetch_params.fetch_func()
            resource = sdk2.Resource[resource_id]

        if not resource:
            raise SandboxTaskFailureError('Could not find resource by ID with params %s' % fetch_params)

        resource_data = sdk2.ResourceData(resource)
        return ResourceFetchResultRaw(res_name, resource, resource_data)

    def _fetch_resource_by_attrs(self, res_name, fetch_params):
        res_type = fetch_params.resource_type
        attrs = fetch_params.attrs
        existing_value = fetch_params.existing_value

        resource = None
        if isinstance(existing_value, int):
            resource = sdk2.Resource[existing_value]
        elif existing_value:
            resource = existing_value

        logging.debug('Resource by attrs existing value: %s', resource)

        if not resource:
            resource = sdk2.Resource.find(
                type=res_type,
                attrs=attrs,
            ).order(-sdk2.Resource.id).first()

        if not resource:
            raise SandboxTaskFailureError('Could not find resource %s with attrs %s' % (res_type, attrs))

        resource_data = sdk2.ResourceData(resource)
        return ResourceFetchResultRaw(res_name, resource, resource_data)

    def _process_if_template_resource(self, resource_name, result):
        template_params = TEMPLATES_CTX_MAP.get(resource_name)
        if not template_params:
            return

        dir_prefix, route_id_field = template_params
        target_dir = os.path.join(os.getcwd(), dir_prefix + '-templates')

        resource_data = result.resource_data
        route_id_path = utils.unpack_templates(resource_data, target_dir)

        setattr(self.Context, route_id_field, route_id_path)

    def _process_if_rr_resource(self, resource_name, result):
        rr_params = RR_CTX_MAP.get(resource_name)
        if not rr_params:
            return

        rr_dir_name, rr_id_field, ynode_path_field, rr_path_field = rr_params

        rr_archive_path = str(result.resource_data.path)
        rr_dir = os.path.join(os.getcwd(), rr_dir_name, 'report-renderer')

        with tarfile.open(rr_archive_path) as tar:
            tar.extractall(rr_dir)

        setattr(self.Context, rr_id_field, result.resource.id)
        setattr(self.Context, ynode_path_field, os.path.join(rr_dir, 'ynode', 'bin', 'ynode'))
        setattr(self.Context, rr_path_field, os.path.join(rr_dir, 'report-renderer'))

    def _process_if_pulse_binary_resource(self, resource_name, result):
        pulse_binary_fields = PULSE_BINARIES_CTX_MAP.get(resource_name)
        if not pulse_binary_fields:
            return

        ctx_id_name, ctx_path_name = pulse_binary_fields
        resource_path = str(result.resource_data.path)

        setattr(self.Context, ctx_id_name, result.resource.id)
        setattr(self.Context, ctx_path_name, resource_path)

        if resource_name == N_PULSE_DIFF_VIEWER:
            diff_viewer_dir = os.path.join(os.getcwd(), 'viewer')

            with tarfile.open(resource_path) as tar:
                tar.extractall(path=diff_viewer_dir)

            if not os.path.exists(diff_viewer_dir) or not os.path.isdir(diff_viewer_dir):
                raise SandboxTaskFailureError('There is no pulse-diff-viewer')

            setattr(self.Context, ctx_path_name, diff_viewer_dir)

    def _process_if_ammo_resource(self, resource_name, result):
        ammo_params = AMMO_CTX_MAP.get(resource_name)
        if not ammo_params:
            return

        ammo_id_field, ammo_path_field = ammo_params
        rr_archive_path = str(result.resource_data.path)

        setattr(self.Context, ammo_id_field, result.resource.id)
        setattr(self.Context, ammo_path_field, rr_archive_path)

    def _adjust_rr_instances(self):
        if not self.Context.rr_actual_id:
            self.Context.rr_actual_id = self.Context.rr_base_id

        if not self.Context.rr_actual_path:
            self.Context.rr_actual_path = self.Context.rr_base_path

        if not self.Context.ynode_actual_path:
            self.Context.ynode_actual_path = self.Context.ynode_base_path

    def _validate_resource_ctx_fields(self):
        not_found = {}
        errors = []
        for field_name in REQUIRED_CTX_FIELDS:
            value = getattr(self.Context, field_name, not_found)
            if value is not_found:
                errors.append('%s not found' % field_name)
            elif value is None:
                errors.append('%s is empty' % field_name)

        if errors:
            raise SandboxTaskFailureError('Resource errors found: %s' % ', '.join(errors))

    def _run_pulse_shooter(self):
        routes_id = self.Parameters.custom_routes_id

        if not self.Parameters.use_custom_routes_id:
            routes_id = ROUTES_CONFIG.get(self.project_name, {}).get(self.service_platform)

        if not routes_id:
            raise SandboxTaskFailureError('There is no route for %s:%s' % (self.project_name, self.service_platform))

        shooter_logs_dir = os.path.join(os.getcwd(), 'pulse-shooter-logs')
        os.makedirs(shooter_logs_dir)

        rr_logs_dir = os.path.join(os.getcwd(), 'rr-logs')
        rr_logs_dir_base = os.path.join(rr_logs_dir, 'base')
        rr_logs_dir_actual = os.path.join(rr_logs_dir, 'actual')

        os.makedirs(rr_logs_dir_base)
        os.makedirs(rr_logs_dir_actual)

        pulse_shooter_results = os.path.join(shooter_logs_dir, 'pulse_shooter_results.json')
        shooter_error_path = os.path.join(shooter_logs_dir, 'stderr.log')

        program_params = self._prepare_pulse_shooter_program_params(routes_id, rr_logs_dir_actual, rr_logs_dir_base)

        with sdk2.helpers.process.ProcessRegistry, open(pulse_shooter_results, 'w') as o_fp, \
                open(shooter_error_path, 'w') as e_fp:
            logging.debug('Using program params: %s', program_params)
            p = Popen(program_params, stdout=o_fp, stderr=e_fp)

            ps_kill_timeout = max(300.0, float(self.Parameters.kill_timeout) - 300)
            logging.info('Using Pulse Shooter kill timeout %s seconds', ps_kill_timeout)

            def kill_by_timeout():
                logging.error('Kill Pulse Shooter by timeout %s seconds', ps_kill_timeout)
                p.kill()
                self.Context.was_timeout = True

            kill_timer = Timer(ps_kill_timeout, kill_by_timeout)
            try:
                kill_timer.start()
                return_code = p.wait()
            finally:
                kill_timer.cancel()

        pulse_shooter_logs_resource = PulseShooterLogs(
            task=self,
            description='Pulse Shooter logs',
            type='pulse-shooter',
            path=shooter_logs_dir,
        )

        sdk2.ResourceData(pulse_shooter_logs_resource).ready()

        try:
            # Can fail if empty logs dir
            rr_logs_resource = PulseShooterLogs(
                task=self,
                description='RR logs',
                type='rr',
                path=rr_logs_dir,
            )
            sdk2.ResourceData(rr_logs_resource).ready()

            self.Context.pulse_shooter_errors_link = pulse_shooter_logs_resource.http_proxy
            self.Context.rr_errors_link = rr_logs_resource.http_proxy

            pulse_shooter_template_errors_base = os.path.join(rr_logs_dir_base, 'template-errors-base.log')
            if os.path.exists(pulse_shooter_template_errors_base):
                self.Context.pulse_shooter_template_errors_base_link = rr_logs_resource.http_proxy + '/base/template-errors-base.log'
                self.Context.pulse_shooter_template_errors_base_count = self._get_errors_count(pulse_shooter_template_errors_base)

            pulse_shooter_template_errors_actual = os.path.join(rr_logs_dir_actual, 'template-errors-actual.log')
            if os.path.exists(pulse_shooter_template_errors_actual):
                self.Context.pulse_shooter_template_errors_actual_link = rr_logs_resource.http_proxy + '/actual/template-errors-actual.log'
                self.Context.pulse_shooter_template_errors_actual_count = self._get_errors_count(pulse_shooter_template_errors_actual)

        except InvalidResource:
            pass

        if self.Context.was_timeout:
            raise RuntimeError('Pulse Shooter was killed by timeout %s seconds' % ps_kill_timeout)

        if return_code != 0:
            self._collect_fail_logs()
            if return_code == REPORT_RENDERER_ERROR_CODE:
                raise ReportRendererError('pulse-shooter raised REPORT RENDERER error and exited with code %s' % return_code)
            elif return_code == PULSE_ERROR_CODE:
                raise PulseError('pulse-shooter raised PULSE ERROR and exited with code %s' % return_code)
            else:
                raise SandboxTaskFailureError('pulse-shooter exited with code %s' % return_code)

        for responses_path in ('rr-responses-base', 'rr-responses-actual'):
            responses_tar = responses_path + '.tar'

            with tarfile.open(responses_tar, 'w') as tar:
                tar.add(responses_path, recursive=True)

        self.base_pulse_shooter_results = self._get_rr_result_log(rr_logs_dir_base)
        self.actual_pulse_shooter_results = self._get_rr_result_log(rr_logs_dir_actual)

    def _prepare_pulse_shooter_program_params(self, routes_id, rr_logs_dir_actual, rr_logs_dir_base):
        program_params = [
            self.Context.pulse_shooter_path,
            '--ynode-base=%s' % self.Context.ynode_base_path,
            '--ynode-actual=%s' % self.Context.ynode_actual_path,
            '--rr-base=%s' % self.Context.rr_base_path,
            '--rr-actual=%s' % self.Context.rr_actual_path,
            '--ammo=%s' % self.Context.ammo_base_path,
            '--routes-id=%s' % routes_id,
            '--routes-base=%s' % self.Context.base_route_path,
            '--rr-logs-base=%s' % rr_logs_dir_base,
            '--rr-logs-actual=%s' % rr_logs_dir_actual,
            '--routes-actual=%s' % self.Context.actual_route_path,
            '--ammo-limit=%s' % (self.Parameters.request_limit_with_memory_capture if self.Parameters.detect_memory_leaks else self.Parameters.request_limit),
            '--memory-capture-requests-per-measurement=%s' % self.Parameters.memory_capture_requests_per_measurement,
            '--workers=%s' % self.rr_workers,
            '--pulse-shooter-workers=%s' % self.pulse_shooter_workers,
            '--threshold-percentile=%s' % self.Parameters.threshold_percentile,
            '--threshold-constant=%s' % self.Parameters.threshold_constant,
        ]

        if self.Context.ammo_actual_path:
            program_params.append('--ammo-actual=%s' % self.Context.ammo_actual_path)

        if self.Parameters.apphost_mode:
            program_params.append('--apphost-mode')

        if self.Parameters.apphost_mode and self.Parameters.ahproxy:
            program_params.append('--ahproxy')

        if self.Parameters.profile_v8:
            program_params.append('--prof')

        if self.Parameters.allow_natives_syntax:
            program_params.append('--allow-natives-syntax')

        if self.Parameters.profile_v8_txt:
            program_params.append('--generate-profile-txt')

        if self.Parameters.no_opt_base:
            program_params.append('--no-opt-base')

        if self.Parameters.no_opt_actual:
            program_params.append('--no-opt-actual')

        if self.Parameters.trace_opt:
            program_params.append('--trace-opt')

        if self.Parameters.trace_deopt:
            program_params.append('--trace-deopt')

        if self.Parameters.detect_memory_leaks:
            program_params.append('--memory-capture-gc-frequency=%s' % self.Parameters.memory_capture_gc_frequency)

        if self.Parameters.heap_snapshots_enabled:
            program_params.extend((
                '--heap-snapshots-enabled',
                '--heap-snapshots-count=%s' % self.Parameters.heap_snapshots_count,
            ))

        if self.Parameters.additional_timing_on:
            self.Parameters.flags_base['additional_timing_on'] = '1'
            self.Parameters.flags_actual['additional_timing_on'] = '1'

        if self.Parameters.flags_base:
            flags_base_file = os.path.join(os.getcwd(), 'flags-base.json')
            self._write_json(flags_base_file, self.Parameters.flags_base)
            program_params.append('--flags-base=%s' % flags_base_file)

        if self.Parameters.flags_actual:
            flags_actual_file = os.path.join(os.getcwd(), 'flags-actual.json')
            self._write_json(flags_actual_file, self.Parameters.flags_actual)
            program_params.append('--flags-actual=%s' % flags_actual_file)

        return program_params

    def _run_pulse_aggregator(self):
        logs_dir = os.path.join(os.getcwd(), 'pulse-aggregator-logs')
        os.makedirs(logs_dir)

        stdout_path = os.path.join(logs_dir, 'stdout.log')
        stderr_path = os.path.join(logs_dir, 'stderr.log')
        excesses_path = os.path.join(logs_dir, 'excesses.log')

        stats_path = os.path.join(logs_dir, 'stats.json')
        program_params = [
            self.Context.pulse_aggregator_path,
            '--platform=%s' % self.service_platform,
            '--rr-base-log=%s' % self.base_pulse_shooter_results,
            '--rr-actual-log=%s' % self.actual_pulse_shooter_results,
            '--stats-out=%s' % stats_path,
            '--excesses-out=%s' % excesses_path,
        ]

        # noinspection PyUnresolvedReferences
        if not self.Parameters.use_custom_routes_id:
            project_limits = pulse_config.load_pulse_genisys_config(self.project_name, 'pulse_shooter_limits')
            limits_file = os.path.join(os.getcwd(), 'project-limits.json')

            with open(limits_file, 'w') as fp:
                json.dump(project_limits, fp)

            program_params.extend((
                '--project-limits={}'.format(limits_file),
                '--significance-threshold={}'.format(self.Parameters.threshold_significance),
            ))

            if self.Parameters.pulse_shooter_ammo_actual:
                program_params.append('--only-significant')

        with sdk2.helpers.process.ProcessRegistry, open(stdout_path, 'w') as o_fp, \
                open(stderr_path, 'w') as e_fp:
            p = Popen(program_params, stdout=o_fp, stderr=e_fp)

            return_code = p.wait()

        pulse_aggregator_logs_resource = PulseShooterLogs(
            task=self,
            description='Pulse Aggregator logs',
            type='pulse-aggregator',
            path=logs_dir,
        )

        sdk2.ResourceData(pulse_aggregator_logs_resource).ready()

        if return_code != 0:
            raise SandboxTaskFailureError('pulse-aggregator exited with code %s' % return_code)

        self.pulse_aggregator_results = stats_path
        self.excesses_results = excesses_path

    def _run_pulse_report(self):
        logs_dir = os.path.join(os.getcwd(), 'pulse-report-logs')
        os.makedirs(logs_dir)

        stderr_path = os.path.join(logs_dir, 'stderr.log')
        report_path = os.path.join(logs_dir, 'report.html')

        with sdk2.helpers.process.ProcessRegistry, open(report_path, 'w') as o_fp, \
                open(stderr_path, 'w') as e_fp:
            p = Popen((
                self.Context.pulse_report_path,
                '--stats=%s' % self.pulse_aggregator_results,
                '--format=html',
            ), stdout=o_fp, stderr=e_fp)

            return_code = p.wait()

        pulse_report_logs_resource = PulseShooterLogs(
            task=self,
            description='Pulse Report logs',
            type='pulse-report',
            path=logs_dir,
        )

        sdk2.ResourceData(pulse_report_logs_resource).ready()

        if return_code != 0:
            raise SandboxTaskFailureError('pulse-report exited with code %s' % return_code)

        self.pulse_report_result = report_path

    def _build_report(self):
        links = [{
            'href': PULSE_WIKI_LINKS.get('pulse_shooter'),
            'text': 'Pulse Shooter documentation',
        }]

        if self.Parameters.pulse_shooter_ammo_actual:
            links += [{
                'href': PULSE_WIKI_LINKS.get('two_shooting_baskets'),
                'text': '<b>How to investigate failure?</b>',
            }]

        if self.Parameters.detect_memory_leaks:
            links += [{
                'href': PULSE_WIKI_LINKS.get('memory_leaks'),
                'text': '<b>How to investigate memory leaks?</b>',
            }]

        if not self.Parameters.use_custom_routes_id:
            links += [{
                'href': PULSE_STAT_LINKS.get(self.project_name + '-' + self.service_platform),
                'text': 'Pulse Statistics',
            }]

        links += [{
            'href': self.Context.html_diff_report_link,
            'text': 'HTML Diff Report',
        }]

        if self.Context.html_filtered_diff_report_link:
            links += [{
                'href': self.Context.html_filtered_diff_report_link,
                'text': 'HTML Diff Report (filtered)',
            }]

        links += [{
            'href': self.Context.pulse_shooter_template_errors_base_link,
            'text': '<span class="status status_error">Errors</span> <span class="status status_error">{count}</span> template-errors-base.log'.format(
                count=self.Context.pulse_shooter_template_errors_base_count
            )
        }, {
            'href': self.Context.pulse_shooter_template_errors_actual_link,
            'text': '<span class="status status_error">Errors</span> <span class="status status_error">{count}</span> template-errors-actual.log'.format(
                count=self.Context.pulse_shooter_template_errors_actual_count
            )
        }, {
            'href': self.Context.pulse_blockstat_log_analyzer_report_link,
            'text': 'Blockstat log analyzer report',
        }]

        links_report = ''
        pulse_report = ''

        for link in links:
            if not link['href']:
                continue

            links_report += '<li><a href="{href}" target="_blank">{text}</a></li>'.format(
                text=link['text'],
                href=link['href'],
            )

        links_report = '<ul>%s</ul>' % links_report

        if self.pulse_aggregator_results:
            with open(self.pulse_aggregator_results) as fp:
                self.report_data = self.Parameters.results = json.load(fp)

        if self.pulse_report_result:
            with open(self.pulse_report_result) as fp:
                pulse_report = fp.read().strip()

        mld_data, mld_plots = self.memory_leaks_detector_result

        if mld_plots:
            heap_total_result = mld_data['heap_total']
            memory_leak_angle = heap_total_result['trend_line_angle']
            memory_leak_angle_color = 'red' if heap_total_result['has_leakage'] else 'green'
            pulse_report += '''
            <style>
                .pulse-memory-plots, {clear: both}
                .pulse-memory-plots img {padding: 8px; float: left}
                .pulse-memory-plots-additional {display: none}
                .pulse-memory-plots-info {clear: both; font-size: 16px}
                #pulse-memory-plots-show-all:checked ~ .pulse-memory-plots .pulse-memory-plots-additional {display: block}
            </style>
            <h3>Memory usage diff plot</h3>
            <input type="checkbox" id="pulse-memory-plots-show-all">
            <label for="pulse-memory-plots-show-all">Show all memory types plots</label>
            <div class="pulse-memory-plots">
                <img alt="Heap Total memory diff plot" width=640 height=480 src="%s">
                <div class="pulse-memory-plots-additional">
                    <img alt="Heap Used memory diff plot" width=640 height=480 src="%s">
                    <img alt="Resident set size memory diff plot" width=640 height=480 src="%s">
                    <img alt="External memory diff plot" width=640 height=480 src="%s">
                </div>
                <div class="pulse-memory-plots-info">Memory leaks angle: <span style="color: %s;">%s˚</span></div>
            </div>
            ''' % (
                mld_plots['heap_total'],
                mld_plots['heap_used'],
                mld_plots['resident_set'],
                mld_plots['external'],
                memory_leak_angle_color,
                memory_leak_angle,
            )

        self.Context.report_html = links_report + pulse_report

    def _run_html_differ(self, use_blockstat_filter=False):
        logs_dir = sdk2.paths.get_unique_file_name(os.getcwd(), 'pulse-html-differ-logs')
        os.makedirs(logs_dir)

        stderr_path = os.path.join(logs_dir, 'stderr.log')
        result_path = os.path.join(logs_dir, 'stdout.log')

        differ_dir = sdk2.paths.get_unique_file_name(os.getcwd(), 'rr-responses')
        os.makedirs(differ_dir)

        command = [
            self.Context.html_differ_path,
            '--custom-log-base=%s' % self.base_pulse_shooter_results,
            '--custom-log-actual=%s' % self.actual_pulse_shooter_results,
            '--responses-base=%s' % os.path.join(os.getcwd(), 'rr-responses-base'),
            '--responses-actual=%s' % os.path.join(os.getcwd(), 'rr-responses-actual'),
            '--report-out=%s/index.html' % differ_dir,
        ]

        if use_blockstat_filter:
            rr_logs_dir = os.path.join(os.getcwd(), 'rr-logs')
            rr_logs_dir_base = os.path.join(rr_logs_dir, 'base')
            rr_logs_dir_actual = os.path.join(rr_logs_dir, 'actual')
            command.extend((
                '--blockstat-log-filter',
                '--blockstat-log-base=%s' % self._get_rr_result_log(rr_logs_dir_base, type='blockstat'),
                '--blockstat-log-actual=%s' % self._get_rr_result_log(rr_logs_dir_actual, type='blockstat'),
            ))

        with sdk2.helpers.process.ProcessRegistry, open(result_path, 'w') as o_fp, open(stderr_path, 'w') as e_fp:
            p = Popen(command, stdout=o_fp, stderr=e_fp)
            return_code = p.wait()

        shutil.copytree(
            self.Context.diff_viewer_path,
            os.path.join(differ_dir, 'viewer')
        )

        pulse_html_differ_logs_resource = PulseShooterLogs(
            task=self,
            description='Pulse HTML Differ logs',
            type='pulse-html-differ',
            path=logs_dir,
        )

        sdk2.ResourceData(pulse_html_differ_logs_resource).ready()

        pulse_html_differ_responses = PulseShooterResponses(
            task=self,
            description='Pulse HTML Differ responses',
            path=differ_dir,
            type='html-differ-result',
        )

        sdk2.ResourceData(pulse_html_differ_responses).ready()

        if return_code != 0:
            raise SandboxTaskFailureError('pulse-html-differ exited with code %s' % return_code)

        if use_blockstat_filter:
            self.Context.html_filtered_diff_report_link = pulse_html_differ_responses.http_proxy + '/index.html'
        else:
            self.Context.html_diff_report_link = pulse_html_differ_responses.http_proxy + '/index.html'

    def _run_memory_leaks_detector(self):
        memory_limits = pulse_config.load_pulse_genisys_config(
            self.project_name, 'memory_limits'
        ).get(self.service_platform, {})

        memory_limits_file = os.path.join(os.getcwd(), 'memory-limits.json')
        with open(memory_limits_file, 'w') as fp:
            json.dump(memory_limits, fp)

        logs_dir = os.path.join(os.getcwd(), 'pulse-memory_leaks_detector-logs')
        os.makedirs(logs_dir)
        result = {}

        try:
            for plot_type in MEMORY_LEAKS_DETECTOR_MEASUREMENTS:
                prefix = plot_type.lower()
                stderr_path = os.path.join(logs_dir, '%s_stderr.log' % prefix)
                stdout_path = os.path.join(logs_dir, '%s_stdout.log' % prefix)
                plot_path = os.path.join(logs_dir, '%s.png' % prefix)

                with sdk2.helpers.process.ProcessRegistry, open(stdout_path, 'w') as o_fp, \
                        open(stderr_path, 'w') as e_fp:
                    args = [
                        self.Context.memory_leaks_detector_path,
                        '--rr-base-log=%s' % self.base_pulse_shooter_results,
                        '--rr-actual-log=%s' % self.actual_pulse_shooter_results,
                        '--memory-measurement-type=%s' % plot_type,
                        '--plot-filename=%s' % plot_path,
                        '--limits=%s' % memory_limits_file,
                    ]

                    if 'leakage_angle' in memory_limits:
                        args.append('--leakage-angle-limit=%s' % float(memory_limits['leakage_angle']))

                    p = Popen(args, stdout=o_fp, stderr=e_fp)

                    return_code = p.wait()

                if return_code != 0:
                    raise SandboxTaskFailureError('pulse-memory-leaks-detector exited with code %s' % return_code)

                with open(os.path.join(logs_dir, stdout_path)) as fp:
                    result[prefix] = json.load(fp)

        finally:
            pulse_memory_leaks_detector_logs_resource = PulseShooterLogs(
                task=self,
                description='Pulse Memory leaks detector logs',
                type='pulse-memory-leaks-detector',
                path=logs_dir,
            )

            sdk2.ResourceData(pulse_memory_leaks_detector_logs_resource).ready()

        http_proxy = pulse_memory_leaks_detector_logs_resource.http_proxy
        plot_urls = {
            'heap_total': http_proxy + '/heap_total.png',
            'heap_used': http_proxy + '/heap_used.png',
            'resident_set': http_proxy + '/resident_set.png',
            'external': http_proxy + '/external.png',
        }

        self.memory_leaks_detector_result = (result, plot_urls)

    def _run_blockstat_log_analyzer(self):
        rr_logs_dir = os.path.join(os.getcwd(), 'rr-logs')
        rr_logs_dir_base = os.path.join(rr_logs_dir, 'base')
        rr_logs_dir_actual = os.path.join(rr_logs_dir, 'actual')

        blockstat_logs_paths = {
            'base': self._get_rr_result_log(rr_logs_dir_base, type='blockstat'),
            'actual': self._get_rr_result_log(rr_logs_dir_actual, type='blockstat')
        }

        logs_dir = os.path.join(os.getcwd(), 'pulse-blockstat_logs_analyzer-logs')
        os.makedirs(logs_dir)

        blockstat_report_filter = self.Parameters.blockstat_report_filter
        blockstat_report_results = {}

        for type in ('base', 'actual'):
            current_logs_dir = os.path.join(logs_dir, type)
            os.makedirs(current_logs_dir)

            stdout_path = os.path.join(current_logs_dir, 'stdout.log')
            stderr_path = os.path.join(current_logs_dir, 'stderr.log')
            report_path = os.path.join(current_logs_dir, 'report.json')

            program_params = [
                self.Context.blockstat_log_analyzer_path,
                '--log-file=%s' % blockstat_logs_paths.get(type),
                '--report-out=%s' % report_path,
            ]

            if blockstat_report_filter:
                program_params.append("--filter=%s" % self.Parameters.blockstat_report_filter)

            logging.debug(program_params)

            with sdk2.helpers.process.ProcessRegistry, open(stdout_path, 'w') as o_fp, open(stderr_path, 'w') as e_fp:
                process = Popen(program_params, stdout=o_fp, stderr=e_fp)
                return_code = process.wait()

            if return_code != 0:
                raise SandboxTaskFailureError('pulse-blockstat-log-analyzer exited with code %s' % return_code)

            with open(report_path) as fd:
                blockstat_report_results[type] = json.load(fd)

        pulse_blockstat_log_analyzer_resource = PulseShooterLogs(
            task=self,
            description='Blockstat log analyzer reports',
            type='pulse-blockstat-log-analyzer',
            path=logs_dir,
        )

        sdk2.ResourceData(pulse_blockstat_log_analyzer_resource).ready()
        self.Context.pulse_blockstat_log_analyzer_report_link = pulse_blockstat_log_analyzer_resource.http_proxy

        if not blockstat_report_results.get('base') or not blockstat_report_results.get('actual'):
            raise PulseBlockstatLogAnalyzerError(
                'Blockstat logs not contains blocks with path: %s.' % blockstat_report_filter
            )

    def get_limits_excesses(self):
        with open(self.excesses_results) as fp:
            warnings = json.load(fp)

        if self.Parameters.detect_memory_leaks:
            mld_data, _ = self.memory_leaks_detector_result
            heap_total = mld_data['heap_total']

            if heap_total and heap_total['has_leakage']:
                trend_line_angle = heap_total['trend_line_angle']
                warnings.append('Detected heap total memory leakage. Trend line angle: %s' % trend_line_angle)

            for measurement_type in MEMORY_LEAKS_DETECTOR_MEASUREMENTS:
                measurement = mld_data[measurement_type.lower()]
                if measurement['is_exceeded']:
                    warnings.append(
                        'Memory limit exceeded: max %s=%s, limit=%s'
                        % (measurement_type, measurement['max'], measurement['limit'])
                    )

        return warnings

    def _check_exist_errors(self):
        errors_actual_count = self.Context.pulse_shooter_template_errors_actual_count or 0
        errors_base_count = self.Context.pulse_shooter_template_errors_base_count or 0

        if (errors_actual_count):
            if (errors_base_count):
                if (errors_actual_count > errors_base_count):
                    raise PulseTemplateError(u'More template errors in Actual than in Base')
            else:
                raise PulseTemplateError(u'Check your templates for errors')

    def _add_tags(self):
        if self.Parameters.use_custom_routes_id:
            if CUSTOM_ROUTE_ID_TAG not in self.Parameters.tags:
                self.Parameters.tags += [CUSTOM_ROUTE_ID_TAG]

        else:
            project_tag = self.project_name.upper()
            platform_tag = self.service_platform.upper()

            if project_tag not in self.Parameters.tags:
                self.Parameters.tags += [project_tag]

            if platform_tag not in self.Parameters.tags:
                self.Parameters.tags += [platform_tag]

        if self.Parameters.apphost_mode and APPHOST_TAG not in self.Parameters.tags:
            self.Parameters.tags += [APPHOST_TAG]

    def _get_rr_result_log(self, directory, type='custom'):
        dir_files = os.listdir(directory)

        rr_logs = filter(lambda x: x.startswith('current-report-renderer_{}'.format(type)), dir_files)
        if not rr_logs:
            raise SandboxTaskFailureError('There is no Pulse custom log')

        return os.path.join(directory, rr_logs[0])

    def _get_errors_count(self, filename):
        count = 0

        with open(filename) as f:
            for line in f:
                # All errors have word 'Error' in stacktrace
                if 'Error' in line:
                    count += 1

        return count

    def _send_statface_report(self):
        if self.Parameters.use_custom_routes_id:
            return

        # noinspection PyUnresolvedReferences
        from statface_client.report import StatfaceReportConfig

        report_data = self.report_data

        components = {
            'RRBase': str(self.Context.rr_base_id),
            'RRActual': str(self.Context.rr_actual_id),
            'AmmoBase': str(self.Context.ammo_base_id),
        }

        current_datetime = datetime.now()

        logging.debug('STATFACE: Start to create data')

        statface_data = PulseShooterStatfaceData(
            current_datetime, report_data, components,
            self.service_platform, self.id,
        )

        logging.debug('STATFACE: Data: {}'.format(statface_data))

        report_name = '{}_dev'.format(self.project_name)

        report = self.statface.get_report('Yandex.Productivity/pulse/{}'.format(report_name))

        logging.debug('STATFACE: Start to create a config')

        config = StatfaceReportConfig(
            title=report_name,
            dimensions=[
                ('fielddate', 'date'),
                ('percentile', 'number'),
                ('chunk_type', 'string'),
                ('platform', 'string'),
            ],
            measures=statface_data.measures()
        )

        report.upload_config(config)

        rows = statface_data.get_rows_for_statface()

        report.upload_data('s', rows)

    def _collect_fail_logs(self):
        fail_log_dir = os.path.join(os.getcwd(), 'fail_logs')
        if not os.path.exists(fail_log_dir):
            os.makedirs(fail_log_dir)

        dmesg_out_file = os.path.join(fail_log_dir, 'dmesg.out')

        with sdk2.helpers.process.ProcessRegistry, open(dmesg_out_file, 'w') as o_fp:
            p = Popen(('dmesg', '-T'), stdout=o_fp, stderr=o_fp)
            p.wait()

        pulse_shooter_logs_resource = PulseShooterLogs(
            task=self,
            description='Pulse Shooter fail logs',
            type='fail',
            path=fail_log_dir,
        )

        sdk2.ResourceData(pulse_shooter_logs_resource).ready()

        # Collect coredumps
        coredump_filter_pattern = re.compile(COREDUMP_FILTER_PATTERN)
        coredumps_dir = self.registry.client.tasks.coredumps_dir
        logging.debug('Checking coredumps dir %s', coredumps_dir)
        for core_name in os.listdir(coredumps_dir):
            if core_name == 'core_watcher.timestamp':
                continue

            logging.debug('Found %s', core_name)
            if coredump_filter_pattern.search(core_name) is not None:
                logging.debug('Saving %s', core_name)
                self._save_coredump(os.path.join(coredumps_dir, core_name))

    def _save_coredump(self, coredump_path):
        coredump_path = os.path.abspath(coredump_path)
        logging.debug('Save coredump {}'.format(coredump_path))
        coredump_filename = os.path.basename(coredump_path)
        saved_coredump_path = str(self.path(coredump_filename))
        gzipped_coredump_path = saved_coredump_path + '.gz'
        try:
            if coredump_path != saved_coredump_path:
                shutil.copy(coredump_path, saved_coredump_path)
            Popen(('gzip', '-f', saved_coredump_path)).wait()
        except OSError:
            logging.exception('Error while copying coredump {}'.format(coredump_filename))
            self.set_info('Cannot dump coredump {}'.format(coredump_filename))
            return None
        mode = stat.S_IMODE(os.stat(gzipped_coredump_path).st_mode)
        mode |= stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH
        os.chmod(gzipped_coredump_path, mode)
        coredump_resource = sdk2.Resource['CORE_DUMP'](
            self, '{} coredump'.format(coredump_filename), gzipped_coredump_path
        )
        sdk2.ResourceData(coredump_resource).ready()
        self.set_info('COREDUMP was saved as resource:{}'.format(coredump_resource.id))
        self.set_info('<hr/>', do_escape=False)
        return coredump_resource

    def _write_json(self, filepath, data):
        with open(filepath, 'w') as f:
            json.dump(data, f)
