# coding=utf-8
import hashlib
import json
import logging
import re
import traceback
from datetime import datetime

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.sandbox_ci.pulse import const as pulse_const
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_params
from sandbox.projects.sandbox_ci.pulse.pulse_shooter import PulseShooter
from sandbox.projects.sandbox_ci.pulse.pulse_shooting_basket import PulseShootingBasket
from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

PULSE_SHOOTER_TAG = 'PULSE_SHOOTER'
CUSTOM_TAG = 'CUSTOM'

space_pattern = re.compile(r'\s+')


class PulseShooterCustom(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
    Проводит обстрел двух версий шаблонов с помощью PULSE_SHOOTER.
    Если нужно, собирает патроны для проекта по счетчику с помощью PULSE_SHOOTING_BASKET.
    Добавляет заданные параметры и флаги в патроны
    """

    class Requirements(sdk2.Requirements):
        disk_space = 40 * 1024
        environments = [
            environments.PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        ]

        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 20 * 3600

        with sdk2.parameters.Group('Custom Shooting Parameters') as custom_params_block:
            with sdk2.parameters.RadioGroup('Ammo query type') as ammo_type:
                ammo_type.values['auto'] = ammo_type.Value('Auto', default=True)
                ammo_type.values['custom'] = ammo_type.Value('Custom')

            with ammo_type.value['auto']:
                counter = sdk2.parameters.String('Counter', default='')
                logs_date = sdk2.parameters.String('Logs date', default='', description='YYYY-MM-DD')

            with ammo_type.value['custom']:
                yql_query = sdk2.parameters.String('YQL query', multiline=True)

        with sdk2.parameters.Group('Pulse plan params') as pulse_plan_params:
            use_soy = sdk2.parameters.Bool(
                'Use Scraper over YT',
                default=False,
            )

            with use_soy.value[True]:
                soy_operation_priority = pulse_params.soy_operation_priority()

            base_query_params = sdk2.parameters.List(
                'Base ammo query params',
                description='Additional flags for base data request',
                default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
            )

            build_actual_ammo = sdk2.parameters.Bool(
                'Build actual ammo',
                description='Build additional (actual) ammo resource',
                default=False,
                sub_fields={'true': ['actual_query_params']},
            )

            actual_query_params = sdk2.parameters.List(
                'Actual ammo query params',
                description='Additional flags for actual data request',
                default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
            )

            requests_number = sdk2.parameters.Integer(
                'Requests number',
                description='Number of requests to prepare',
                default=5400,
                required=True,
            )

            access_log_threshold = sdk2.parameters.Integer(
                'Access log threshold',
                description='Required number of access log records (will raise if not satisfy)',
                default=5200,
                required=True,
            )

            plan_threshold = sdk2.parameters.Integer(
                'Plan threshold',
                description='Required number of plan ammo (will raise if not satisfy)',
                default=5000,
                required=True,
            )

            use_ammo_cache = sdk2.parameters.Bool(
                'Use ammo cache',
                default=True,
            )

        pulse_shooter_params = PulseShooter.Parameters()

    class Context(sdk2.Context):
        shooting_basket_task_id = None
        shooting_task_id = None

        pulse_shooter_ammo_base_id = None
        pulse_shooter_ammo_actual_id = None

        report_html = None

    @classmethod
    def format_github_context(cls, platform):
        return u'Pulse Shooter Custom: {}'.format(platform)

    @property
    def github_context(self):
        platform = self.Parameters.platform
        if self.Parameters.service:
            platform = '{}/{}'.format(self.Parameters.service, self.Parameters.platform)
        return self.format_github_context(platform)

    @property
    def counter(self):
        return space_pattern.sub('', str(self.Parameters.counter))

    def on_execute(self):
        with self.memoize_stage.show_alerts():
            self._show_alerts()

        with self.memoize_stage.shooting_basket(), self.profiler.actions.shooting_basket('Shooting basket'):
            self._prepare_shooting_basket()

        with self.memoize_stage.handle_ammo(), self.profiler.actions.handle_ammo('Handle shooting basket result'):
            self._handle_shooting_basket_results()

        with self.memoize_stage.shooting(), self.profiler.actions.shooting('Shooting'):
            self._launch_shooting()

        with self.memoize_stage.shooting_results(), self.profiler.actions.shooting_results('Shooting results'):
            self._handle_shooting_results()

    def on_finish(self, prev_status, status):
        self._profiler_send_report()
        super(self.__class__, self).on_finish(prev_status, status)

    def _show_alerts(self):
        if self.Parameters.pulse_shooter_ammo_base or self.Parameters.pulse_shooter_ammo_actual:
            self.set_info('Ammo from parameters are ignored. Collect custom ammo')

    def _prepare_shooting_basket(self):
        counter = self.counter
        base_query_params = list(self.Parameters.base_query_params)
        actual_query_params = list(self.Parameters.actual_query_params)

        if not counter and not base_query_params and not actual_query_params:
            raise TaskFailure('You should provide one of: counter, base or actual query params')

        final_platform = str(self.Parameters.platform)
        if self.Parameters.service:
            final_platform = '%s-%s' % (self.Parameters.service, self.Parameters.platform)

        task_params = {
            'requests_number': int(self.Parameters.requests_number),
            'use_soy': bool(self.Parameters.use_soy),
            'soy_operation_priority': str(self.Parameters.soy_operation_priority),
            'access_log_threshold': int(self.Parameters.access_log_threshold),
            'plan_threshold': int(self.Parameters.plan_threshold),

            'base_query_params': base_query_params,
            'build_actual_ammo': bool(self.Parameters.build_actual_ammo),
            'actual_query_params': actual_query_params,

            'prepare_classic_basket': not self.Parameters.apphost_mode,
            'prepare_apphost_basket': bool(self.Parameters.apphost_mode),
            'build_dolbilka_plan': False,
            'prepare_json_basket': False,

            'release_resources': False,
        }

        if self.Parameters.ammo_type == 'auto':
            task_params.update({
                'target': '{project}:{platform}'.format(project=self.Parameters.project, platform=final_platform),
                'counter': self.Parameters.counter,
                'logs_date': self.Parameters.logs_date
            })
        else:
            task_params.update({
                'ammo_type': 'custom',
                'custom_project': str(self.Parameters.project),
                'custom_platform': final_platform,
                'custom_yql': self.Parameters.yql_query,
            })

        logging.debug('Pulse Shooting Basket params: %s', task_params)

        params_json = json.dumps(task_params, sort_keys=True)
        params_hash = hashlib.md5(params_json.encode()).hexdigest()

        should_wait = False
        shooting_basket_task = None

        if self.Parameters.use_ammo_cache:
            if self.Parameters.ammo_type == 'auto':
                shooting_basket_task = self._find_shooting_basket_task(task_params)
            else:
                shooting_basket_task = self._find_shooting_basket_task(dict(custom_params_hash=params_hash))

        if not shooting_basket_task:
            should_wait = True
            shooting_basket_task = PulseShootingBasket(
                self,
                tags=(PULSE_SHOOTER_TAG, CUSTOM_TAG, self.Parameters.project, final_platform),
                description='Custom shooting basket for {}/{} by counter "{}" with custom flags'.format(
                    self.Parameters.project,
                    self.Parameters.platform,
                    counter,
                ),
                custom_params_hash=params_hash,
                **task_params
            ).enqueue()

        self.Context.shooting_basket_task_id = shooting_basket_task.id

        if should_wait:
            raise sdk2.WaitTask(
                tasks=shooting_basket_task,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

    def _find_shooting_basket_task(self, input_parameters):
        return sdk2.Task.find(
            task_type=PulseShootingBasket,
            status=ctt.Status.SUCCESS,
            children=True,
            input_parameters=input_parameters,
        ).order(-sdk2.Task.id).first()

    def _handle_shooting_basket_results(self):
        shooting_basket_task = sdk2.Task[self.Context.shooting_basket_task_id]

        if shooting_basket_task.status != ctt.Status.SUCCESS:
            msg = 'Ammo collecting ended with error: %s\n%s' % (shooting_basket_task.status, shooting_basket_task.info)
            raise TaskFailure(msg)

        ammo_base = shooting_basket_task.Parameters.pulse_shooter_plan_phantom
        ammo_actual = shooting_basket_task.Parameters.pulse_shooter_plan_phantom_actual
        if self.Parameters.apphost_mode:
            ammo_base = shooting_basket_task.Parameters.pulse_shooter_plan_phantom_apphost
            ammo_actual = shooting_basket_task.Parameters.pulse_shooter_plan_phantom_apphost_actual

        if not ammo_base:
            raise TaskFailure('Error when collecting Pulse Shooter ammo. Look to child task')

        self.Context.pulse_shooter_ammo_base_id = ammo_base.id
        if ammo_actual:
            self.Context.pulse_shooter_ammo_actual_id = ammo_actual.id

    def _launch_shooting(self):
        shooter_params = self._get_subtask_default_parameters()
        shooter_params['pulse_shooter_ammo_base'] = self.Context.pulse_shooter_ammo_base_id
        shooter_params['pulse_shooter_ammo_actual'] = self.Context.pulse_shooter_ammo_actual_id

        if self.Parameters.ammo_type == 'auto':
            shooter_params['generate_blockstat_report'] = True
            shooter_params['blockstat_report_filter'] = self.counter

        logging.debug('Pulse Shooter params: %s', shooter_params)
        shooting_task = PulseShooter(self, **shooter_params).enqueue()

        self.Context.shooting_task_id = shooting_task.id

        raise sdk2.WaitTask(
            tasks=shooting_task,
            statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        )

    def _get_subtask_default_parameters(self):
        param_names = (
            # General and other params
            'project',
            'project_tree_hash',

            'fail_on_limits_exceed',
            'send_email_on_limits_exceed',
            'should_report_to_stat',

            'scripts_last_resource',
            'statface_host',

            'send_comment_to_searel',
            'send_comment_to_issue',

            # Templates  and data params
            'service',
            'platform',
            'base_templates_package',
            'actual_templates_package',

            # Shooting components
            'rr_base',
            'rr_actual',
            'pulse_shooter_binary',
            'pulse_aggregator_binary',
            'pulse_report_binary',
            'html_differ_binary',
            'diff_viewer_binary',
            'memory_leaks_detector_binary',

            # Template flags
            'flags_base',
            'flags_actual',
            'additional_timing_on',

            # Shooting params
            'rr_workers',
            'pulse_shooter_workers',
            'threshold_percentile',
            'threshold_constant',
            'request_limit',
            'request_limit_with_memory_capture',
            'apphost_mode',
            'ahproxy',
            'profile_v8',
            'profile_v8_txt',
            'allow_natives_syntax',
            'no_opt_base',
            'no_opt_actual',
            'trace_opt',
            'trace_deopt',
            'detect_memory_leaks',
            'memory_capture_requests_per_measurement',
            'memory_capture_gc_frequency',
            'heap_snapshots_enabled',
            'heap_snapshots_count',

            # Project params
            'project_build_context',
            'build_platform',
            'static_url_template',

            # Environment params
            'environ',
            '_container',

            # params for GitHub status
            'project_github_owner',
            'project_github_repo',
            'project_github_commit',
            'report_github_statuses',
        )

        result = {
            p: getattr(self.Parameters, p)
            for p in param_names
            if hasattr(self.Parameters, p)
        }

        if 'tags' not in result:
            result['tags'] = []

        if CUSTOM_TAG not in result['tags']:
            result['tags'].append(CUSTOM_TAG)

        return result

    def _handle_shooting_results(self):
        shooting_task = sdk2.Task[self.Context.shooting_task_id]

        self.Context.report_html = shooting_task.Context.report_html
        self.Parameters.results = shooting_task.Parameters.results

        # Failure is ok on limits exceeded
        if shooting_task.status not in (ctt.Status.SUCCESS, ctt.Status.FAILURE):
            msg = 'Shooting ended with error: %s\n%s' % (shooting_task.status, shooting_task.info)
            raise TaskFailure(msg)

        if shooting_task.status == ctt.Status.FAILURE:
            raise TaskFailure('Pulse Shooter task ended with failure')

    def _profiler_send_report(self):
        try:
            # catch statface errors, see https://st.yandex-team.ru/FEI-5011
            self.profiler.send_report()
        except Exception as e:
            logging.error('Statface Upload Error: %s', str(e))
            logging.error(traceback.format_exc())

    @sdk2.header()
    def header(self):
        header = ''

        if self.Context.shooting_basket_task_id:
            header += """
                <div style="margin-bottom: 15px">
                <b>Custom Shooting Basket Task:</b> <a href="/task/{id}/view">{id}</a>
                """.format(id=self.Context.shooting_basket_task_id)

            if self.Context.pulse_shooter_ammo_base_id:
                header += """
                    <br><br><b>Pulse Shooter base ammo:</b> <a href="/resource/{resource_id}/view">{resource_id}</a>
                    """.format(resource_id=self.Context.pulse_shooter_ammo_base_id)

            if self.Context.pulse_shooter_ammo_actual_id:
                header += """
                    <br><br><b>Pulse Shooter actual ammo:</b> <a href="/resource/{resource_id}/view">{resource_id}</a>
                    """.format(resource_id=self.Context.pulse_shooter_ammo_actual_id)

            header += '</div>'
        else:
            header += '<div style="margin-bottom: 15px"><b>Custom Shooting Basket Task:</b> In progress…</div>'

        if self.Context.shooting_task_id:
            header += """
                <div style="margin-bottom: 15px">
                <b>Shooting Task:</b> <a href="/task/{id}/view">{id}</a>
                </div>
                """.format(id=self.Context.shooting_task_id)
        else:
            header += '<div style="margin-bottom: 15px"><b>Shooting Task:</b> Pending…</div>'

        if self.Context.report_html:
            header += self.Context.report_html

        return header
