#  -*- coding: utf-8 -*-
import re
import logging

from datetime import datetime
from urlparse import parse_qsl, urlsplit

import sandbox.projects.sandbox_ci.pulse.utils.yql as yql_utils
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.common.types.task import Status
from sandbox.common.utils import singleton_property
from sandbox.projects.common.yasm import push_api
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.sandbox_ci import managers
from sandbox.projects.sandbox_ci.pulse import const as pulse_const
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_params
from sandbox.projects.sandbox_ci.pulse.prepare_report_renderer_plan import PrepareReportRendererPlan
from sandbox.projects.sandbox_ci.pulse.pulse_shooting_basket.sql import build_query
from sandbox.projects.sandbox_ci.pulse.release_pulse_shooter_ammo import ReleasePulseShooterAmmo
from sandbox.projects.sandbox_ci.pulse.resources import ReportRendererPhantomData, \
    ReportRendererPhantomDataApphost, SearchParsedAccessLog, ReportRendererPlan, ReportRendererPlanJson, \
    ReportRendererPlanApphost
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

WAIT_TIMEOUT = 10 * 3600  # максимальное время ожидания сабтасок (10 часов)


class PulseShootingBasket(TasksResourceRequirement, sdk2.Task):
    """Собирает патроны для проекта или по переданному YQL и генерирует SHOOTING_COMPONENTS_META_DATA"""

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 10 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Project params') as project_params:
            # предполагаем, что у нас 2 таска, которых можно долго ждать:
            # RUN_YQL_2 и PREPARE_REPORT_RENDERER_PLAN
            kill_timeout = WAIT_TIMEOUT * 2

            with sdk2.parameters.RadioGroup('Ammo query type') as ammo_type:
                ammo_type.values['auto'] = ammo_type.Value(value='Auto', default=True)
                ammo_type.values['custom'] = ammo_type.Value(value='Custom')

            with ammo_type.value['auto']:
                target = pulse_params.shooting_basket_target()
                counter = sdk2.parameters.String('Counter', default='')
                logs_date = sdk2.parameters.String('Logs date', default='', description='YYYY-MM-DD')

            with ammo_type.value['custom']:
                custom_project = sdk2.parameters.String(
                    'Project',
                    required=True,
                )

                custom_platform = sdk2.parameters.String(
                    'Platform',
                    required=True,
                )

                custom_yql = sdk2.parameters.String(
                    'Custom YQL',
                    description='Result of YQL query should contain as least two fields - "path" and "headers"',
                    multiline=True,
                    required=True,
                )

        with sdk2.parameters.Group('Overlayfs'):
            _use_overlayfs = sdk2.parameters.Bool(
                'Use overlayfs',
                description='Использовать overlayfs (тестовый режим)',
                default=True
            )

        with sdk2.parameters.Group('Pulse plan params') as pulse_plan_params:
            requests_number = sdk2.parameters.Integer(
                'Requests number',
                description='Number of requests to prepare',
                default=30000,
                required=True,
            )

            use_soy = sdk2.parameters.Bool(
                'Use Scraper over YT',
                default=False,
            )

            with use_soy.value[True]:
                soy_operation_priority = pulse_params.soy_operation_priority()

            access_log_threshold = sdk2.parameters.Integer(
                'Access log threshold',
                description='Required number of access log records (will raise if not satisfy)',
                default=28000,
                required=True,
            )

            plan_threshold = sdk2.parameters.Integer(
                'Plan threshold',
                description='Required number of plan ammo (will raise if not satisfy)',
                default_value=15000,
                required=True,
            )

            parallel_request_limit = sdk2.parameters.Integer(
                'Parallel request limit',
                default=5,
            )

            base_query_params = sdk2.parameters.List(
                'Base ammo query params',
                description='Additional flags for base data request',
                default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
            )

            build_actual_ammo = sdk2.parameters.Bool(
                'Build actual ammo',
                description='Build additional (actual) ammo resource',
                default=False,
                sub_fields={'true': ['actual_query_params']},
            )

            actual_query_params = sdk2.parameters.List(
                'Actual ammo query params',
                description='Additional flags for actual data request',
                default=pulse_const.HAMSTER_QUERY_PARAM_LIST,
            )

            release_resources = sdk2.parameters.Bool(
                'Release resources',
                description='Release resource',
                default=False,
            )

            yql_api_retries = sdk2.parameters.Integer(
                'YQL API download retries',
                default_value=10,
                required=True,
            )

        with sdk2.parameters.Group('Basket params') as basket_params:
            prepare_classic_basket = sdk2.parameters.Bool('Prepare classic Pulse basket', default=True)
            prepare_apphost_basket = sdk2.parameters.Bool('Prepare Apphost Pulse basket', default=False)
            build_dolbilka_plan = sdk2.parameters.Bool(
                'Build plan versions in Dolbilka format',
                description='Gather report-renderer ammo in Dolbilka format (for classic and Apphost both)',
                default=True,
            )
            prepare_json_basket = sdk2.parameters.Bool('Prepare JSON Pulse basket', default=False)

        with sdk2.parameters.Group('Cache params') as cache_params:
            custom_params_hash = sdk2.parameters.String(
                'Custom parameters hash',
                description='Technical parameter for caching (don\'t touch this)',
                default='',
            )

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Pulse Shooter base ammo') as pulse_shooter_base_ammo:
                pulse_shooter_plan_phantom = sdk2.parameters.Resource('Pulse Shooter classic Phantom plan')
                pulse_shooter_plan_dolbilka = sdk2.parameters.Resource('Pulse Shooter classic Dolbilka plan')
                pulse_shooter_plan_json = sdk2.parameters.Resource('Pulse Shooter classic JSON plan')

                pulse_shooter_plan_phantom_apphost = sdk2.parameters.Resource(
                    'Pulse Shooter Apphost Phantom plan'
                )
                pulse_shooter_plan_dolbilka_apphost = sdk2.parameters.Resource(
                    'Pulse Shooter Apphost Dolbilka plan'
                )

            with sdk2.parameters.Group('Pulse Shooter actual ammo') as pulse_shooter_actual_ammo:
                pulse_shooter_plan_phantom_actual = sdk2.parameters.Resource('Pulse Shooter classic Phantom plan')
                pulse_shooter_plan_dolbilka_actual = sdk2.parameters.Resource('Pulse Shooter classic Dolbilka plan')
                pulse_shooter_plan_json_actual = sdk2.parameters.Resource('Pulse Shooter classic JSON plan')

                pulse_shooter_plan_phantom_apphost_actual = sdk2.parameters.Resource(
                    'Pulse Shooter Apphost Phantom plan'
                )
                pulse_shooter_plan_dolbilka_apphost_actual = sdk2.parameters.Resource(
                    'Pulse Shooter Apphost Dolbilka plan'
                )

    class Context(sdk2.Context):
        project = None
        platform = None
        release_resources = None
        yql_query = None

        access_log_task_id = None
        rr_plan_task_id = None
        rr_plan_apphost_task_id = None
        release_pulse_shooter_ammo_task_id = None

        access_log_resource_id = None
        access_log_table_path = None

    def on_prepare(self):
        self.Context.custom_params_hash = self.Parameters.custom_params_hash
        self.Context.release_resources = bool(self.Parameters.release_resources)

        if self.Parameters.ammo_type == 'custom':
            self.Context.project = self.Parameters.custom_project
            self.Context.platform = self.Parameters.custom_platform
            self.Context.release_resources = False
            self.Context.yql_query = self.Parameters.custom_yql
        elif ':' in self.Parameters.target:
            self.Context.project, self.Context.platform = self.Parameters.target.split(':', 1)
            self.Context.yql_query = build_query(
                self.Context.project,
                self.Context.platform,
                counter=self.counter,
                logs_date=self.logs_date,
                limit=self.Parameters.requests_number,
            )
        else:
            raise RuntimeError('Invalid target: %s' % self.Parameters.target)

        if self.Parameters.build_actual_ammo:
            self.Context.release_resources = False

    @singleton_property
    def task_reports(self):
        return managers.Reports(self)

    @singleton_property
    def counter(self):
        space_pattern = re.compile(r'\s+')
        return space_pattern.sub('', str(self.Parameters.counter))

    @singleton_property
    def logs_date(self):
        return str(self.Parameters.logs_date) or datetime.now().strftime('%Y-%m-%d')

    @property
    def is_custom(self):
        return self.Parameters.ammo_type == 'custom' or 'CUSTOM' in self.Parameters.tags

    def on_execute(self):
        with self.memoize_stage.show_alerts():
            if self.Parameters.release_resources and not self.Context.release_resources:
                self.set_info('Could not release resources when building custom ammo, parameter ignored')

        # Fetch access-log data for ammo
        with self.memoize_stage.prepare_access_log(commit_on_entrance=False):
            self._fetch_access_data()

        # Prepare parsed access log resource
        with self.memoize_stage.prepare_parsed_access_log(commit_on_entrance=False):
            self._check_subtask_status(self.Context.access_log_task_id, 'RUN_YQL_2')
            self._prepare_parsed_access_log()

        # Prepare Pulse Shooter ammo
        if self.Parameters.prepare_classic_basket or self.Parameters.prepare_apphost_basket:
            with self.memoize_stage.prepare_rr_plan(commit_on_entrance=False):
                self._prepare_rr_plan()

            with self.memoize_stage.check_rr_plan(commit_on_entrance=False):
                if self.Parameters.prepare_classic_basket:
                    self._check_subtask_status(self.Context.rr_plan_task_id, 'PREPARE_REPORT_RENDERER_PLAN')

                if self.Parameters.prepare_apphost_basket:
                    self._check_subtask_status(self.Context.rr_plan_apphost_task_id, 'PREPARE_REPORT_RENDERER_PLAN')

            with self.memoize_stage.publish_pulse_shooter_ammo(commit_on_entrance=False):
                self._publish_pulse_shooter_ammo_from_originals()

            ammo_task_id = self.Context.release_pulse_shooter_ammo_task_id
            if ammo_task_id:
                with self.memoize_stage.check_pulse_shooter_ammo_publishing(commit_on_entrance=False):
                    self._check_subtask_status(ammo_task_id, 'RELEASE_PULSE_SHOOTER_AMMO')
                    self._release_pulse_shooter_ammo()
                    self._publish_pulse_shooter_ammo_from_release()


    def on_before_end(self, status):
        if self.is_custom:
            return logging.debug('Do not send a signal for custom runs')

        # Заменяем "-" и ":" в параметре target на подчёркивания, чтобы можно было
        # передавать это в yasm как тег. См. https://nda.ya.ru/t/lZ_110LS3bUNfe.
        target_tag = re.sub(
            r'[-:]',
            lambda matchobj: '__' if matchobj.group(0) == ':' else '_',
            self.Parameters.target
        )

        try:
            push_api.push_signals(
                signals={
                    'status_{}_mmmm'.format(status.lower()): 1,
                },
                tags={
                    'itype': 'sandboxci',
                    'prj': 'sandboxci',
                    'service_name': target_tag,
                    'task_type': str(self.type),
                }
            )
        except Exception:
            logging.exception('Exception while sending status signal to yasm')

    def on_finish(self, prev_status, status):
        self.on_before_end(status)
        super(PulseShootingBasket, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.on_before_end(status)
        super(PulseShootingBasket, self).on_break(prev_status, status)

    def _fetch_access_data(self):
        subtask = RunYQL2(
            self,
            owner=yql_utils.YQL_TOKEN_OWNER,
            kill_timeout=self.Parameters.kill_timeout,
            description=self.Parameters.description,
            query=self.Context.yql_query,
            yql_token_vault_name=yql_utils.YQL_TOKEN_NAME,
            use_v1_syntax=True,
            publish_query=True,
            publish_download_link=True,
            public_download_link=True,
            download_format='JSON_WITH_BINARY',
            trace_query=True,
            retry_period=60,
        ).enqueue()

        self.Context.access_log_task_id = subtask.id

        self._wait_task(subtask)

    def _prepare_parsed_access_log(self):
        access_log_url = sdk2.Task[self.Context.access_log_task_id].Parameters.full_results_download_url
        logging.debug('Result download link: %s', access_log_url)

        if not access_log_url:
            raise TaskFailure('YQL did not find any data by your query')

        dest_file_name = 'access_log'
        yql_utils.download_from_yql(access_log_url, dest_file_name, max_retries=self.Parameters.max_restarts)

        access_log_resource = SearchParsedAccessLog(
            task=self,
            description=dest_file_name,
            path=dest_file_name,
            project=self.Context.project,
            platform=self.Context.platform,
            project_type=self.Context.platform,
        )

        sdk2.ResourceData(access_log_resource).ready()

        self.Context.access_log_resource_id = access_log_resource.id
        self.Context.access_log_table_path = dict(parse_qsl(urlsplit(access_log_url).query)).get('path')

    def _prepare_rr_plan(self):
        subtask_params = {
            'kill_timeout': self.Parameters.kill_timeout,
            'description': self.Parameters.description,

            'project': self.Context.project,
            'platform': self.Context.platform,

            'use_soy': self.Parameters.use_soy,
            'soy_operation_priority': self.Parameters.soy_operation_priority,
            'request_limit': self.Parameters.requests_number,
            'threshold': self.Parameters.access_log_threshold,
            'access_log': self.Context.access_log_resource_id,
            'access_log_table': self.Context.access_log_table_path,
            'parallel_request_limit': self.Parameters.parallel_request_limit,

            'build_json_plan': self.Parameters.prepare_json_basket,
            'build_dolbilka_plan': self.Parameters.build_dolbilka_plan,

            'build_actual_ammo': self.Parameters.build_actual_ammo,
            'base_query_params': self.Parameters.base_query_params,
            'actual_query_params': self.Parameters.actual_query_params,
        }

        if self.Context.project == 'news':
            # Новостей нет на хамстере, ходим в приемку.
            subtask_params['report_host'] = 'news.stable.priemka.yandex.ru'

            # Приемка Новостей не работает с дефолтными параметрами, прокидываем только exp_flags
            subtask_params['query_params'] = {
                'exp_flags': 'test_tool=pulse',
                'flags': 'yxnews_core_render_with_neo_desktop=1;yxnews_core_actions_with_neo=index,rubric,story',
            }

        classic_plan_task = None
        apphost_plan_task = None
        if self.Parameters.prepare_classic_basket:
            classic_subtask_params = subtask_params.copy()
            classic_plan_task = PrepareReportRendererPlan(self, **classic_subtask_params)

        if self.Parameters.prepare_apphost_basket:
            apphost_subtask_params = subtask_params.copy()
            apphost_subtask_params.update({
                'apphost_format': True,
                'build_json_plan': False,
            })
            apphost_plan_task = PrepareReportRendererPlan(self, **apphost_subtask_params)

        subtasks = []

        if self.Parameters.prepare_classic_basket:
            subtasks.append(classic_plan_task)
            self.Context.rr_plan_task_id = classic_plan_task.id

        if apphost_plan_task:
            subtasks.append(apphost_plan_task)
            self.Context.rr_plan_apphost_task_id = apphost_plan_task.id

        for task in subtasks:
            task.enqueue()

        self._wait_task(subtasks)

    def _publish_pulse_shooter_ammo_from_originals(self):
        plan_dolbilka, plan_dolbilka_apphost, plan_json, plan_phantom, plan_phantom_apphost = \
            self._get_all_pulse_shooter_ammo_resources(
                self.Context.rr_plan_task_id,
                self.Context.rr_plan_apphost_task_id,
                pulse_const.AMMO_TYPE_BASE
            )

        plan_dolbilka_actual = None
        plan_dolbilka_apphost_actual = None
        plan_json_actual = None
        plan_phantom_actual = None
        plan_phantom_apphost_actual = None

        if self.Parameters.build_actual_ammo:
            plan_dolbilka_actual, plan_dolbilka_apphost_actual, plan_json_actual, plan_phantom_actual, \
                plan_phantom_apphost_actual = self._get_all_pulse_shooter_ammo_resources(
                    self.Context.rr_plan_task_id,
                    self.Context.rr_plan_apphost_task_id,
                    pulse_const.AMMO_TYPE_ACTUAL
                )

        if not self.Context.release_resources:
            self.Parameters.pulse_shooter_plan_phantom = plan_phantom
            self.Parameters.pulse_shooter_plan_dolbilka = plan_dolbilka
            self.Parameters.pulse_shooter_plan_json = plan_json

            self.Parameters.pulse_shooter_plan_phantom_apphost = plan_phantom_apphost
            self.Parameters.pulse_shooter_plan_dolbilka_apphost = plan_dolbilka_apphost

            self.Parameters.pulse_shooter_plan_phantom_actual = plan_phantom_actual
            self.Parameters.pulse_shooter_plan_dolbilka_actual = plan_dolbilka_actual
            self.Parameters.pulse_shooter_plan_json_actual = plan_json_actual

            self.Parameters.pulse_shooter_plan_phantom_apphost_actual = plan_phantom_apphost_actual
            self.Parameters.pulse_shooter_plan_dolbilka_apphost_actual = plan_dolbilka_apphost_actual
            return

        subtask = ReleasePulseShooterAmmo(
            self,
            kill_timeout=self.Parameters.kill_timeout,
            description='Release Pulse Shooter ammo for project "{project}" and platform "{platform}"'.format(
                project=self.Context.project,
                platform=self.Context.platform,
            ),
            project=self.Context.project,
            platform=self.Context.platform,

            parsed_access_log=self.Context.access_log_resource_id,

            report_renderer_plan_phantom=plan_phantom,
            report_renderer_plan_dolbilka=plan_dolbilka,
            report_renderer_plan_json=plan_json,

            report_renderer_plan_phantom_apphost=plan_phantom_apphost,
            report_renderer_plan_dolbilka_apphost=plan_dolbilka_apphost,
        ).enqueue()

        self.Context.release_pulse_shooter_ammo_task_id = subtask.id

        self._wait_task(subtask.id)

    def _publish_pulse_shooter_ammo_from_release(self):
        plan_dolbilka, plan_dolbilka_apphost, plan_json, plan_phantom, plan_phantom_apphost = \
            self._get_all_pulse_shooter_ammo_resources(
                self.Context.release_pulse_shooter_ammo_task_id,
                self.Context.release_pulse_shooter_ammo_task_id,
                pulse_const.AMMO_TYPE_BASE,
            )

        self.Parameters.pulse_shooter_plan_phantom = plan_phantom
        self.Parameters.pulse_shooter_plan_dolbilka = plan_dolbilka
        self.Parameters.pulse_shooter_plan_json = plan_json

        self.Parameters.pulse_shooter_plan_phantom_apphost = plan_phantom_apphost
        self.Parameters.pulse_shooter_plan_dolbilka_apphost = plan_dolbilka_apphost

    def _get_all_pulse_shooter_ammo_resources(self, rr_plan_task_id, rr_plan_apphost_task_id, ammo_type):
        plan_phantom = None
        plan_dolbilka = None
        plan_json = None
        if (self.Parameters.prepare_classic_basket or self.Parameters.prepare_apphost_basket) and rr_plan_task_id:
            if self.Parameters.prepare_classic_basket:
                plan_phantom = sdk2.Resource.find(
                    type=ReportRendererPhantomData,
                    task_id=rr_plan_task_id,
                    attrs=dict(ammo_type=ammo_type),
                ).first()

                if not plan_phantom:
                    raise TaskFailure('Unable to find resource={type} within task={task_id}'.format(
                        type=ReportRendererPhantomData,
                        task_id=rr_plan_task_id,
                    ))

                if self.Parameters.build_dolbilka_plan:
                    plan_dolbilka = sdk2.Resource.find(
                        type=ReportRendererPlan,
                        task_id=rr_plan_task_id,
                        attrs=dict(ammo_type=ammo_type),
                    ).first()

                    if not plan_dolbilka:
                        raise TaskFailure('Unable to find resource={type} within task={task_id}'.format(
                            type=ReportRendererPlan,
                            task_id=rr_plan_task_id,
                        ))

                if self.Parameters.prepare_json_basket:
                    plan_json = sdk2.Resource.find(
                        type=ReportRendererPlanJson,
                        task_id=rr_plan_task_id,
                        attrs=dict(ammo_type=ammo_type),
                    ).first()

                    if not plan_json:
                        raise TaskFailure('Unable to find resource={type} within task={task_id}'.format(
                            type=ReportRendererPlanJson,
                            task_id=rr_plan_task_id,
                        ))

        plan_phantom_apphost = None
        plan_dolbilka_apphost = None
        if self.Parameters.prepare_apphost_basket:
            plan_phantom_apphost = sdk2.Resource.find(
                type=ReportRendererPhantomDataApphost,
                task_id=rr_plan_apphost_task_id,
                attrs=dict(ammo_type=ammo_type),
            ).first()

            if not plan_phantom_apphost:
                raise TaskFailure('Unable to find resource={type} within task={task_id}'.format(
                    type=ReportRendererPhantomDataApphost,
                    task_id=rr_plan_apphost_task_id,
                ))

            if self.Parameters.build_dolbilka_plan:
                plan_dolbilka_apphost = sdk2.Resource.find(
                    type=ReportRendererPlanApphost,
                    task_id=rr_plan_apphost_task_id,
                    attrs=dict(ammo_type=ammo_type),
                ).first()

                if not plan_dolbilka_apphost:
                    raise TaskFailure('Unable to find resource={type} within task={task_id}'.format(
                        type=ReportRendererPlanApphost,
                        task_id=rr_plan_apphost_task_id,
                    ))

        return plan_dolbilka, plan_dolbilka_apphost, plan_json, plan_phantom, plan_phantom_apphost

    def _release_pulse_shooter_ammo(self):
        self.server.release(
            params=dict(
                task_id=self.Context.release_pulse_shooter_ammo_task_id,
                type='stable',
                subject='Pulse Shooter ammo release',
            ),
        )

    def _wait_task(self, task):
        raise sdk2.WaitTask(
            task,
            Status.Group.FINISH | Status.Group.BREAK,
            timeout=self.Parameters.kill_timeout,
            wait_all=True,
        )

    def _check_subtask_status(self, task_id, label):
        if not task_id:
            return logging.warn('Wrong task ID for check: %s', task_id)

        subtask = sdk2.Task[task_id]

        if subtask.status != Status.SUCCESS:
            raise TaskFailure(
                'Subtask %s ended with error: %s\n%s' % (
                    label,
                    subtask.status,
                    subtask.info,
                )
            )
