# -*- coding: utf-8 -*-
import datetime
import fnmatch
import glob
import json
import logging
import os
import shutil
import tarfile
from collections import defaultdict
from subprocess import Popen

from sandbox import sdk2
from sandbox.common.utils import singleton_property, classproperty
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.pulse import parameters
from sandbox.projects.sandbox_ci.pulse.config import load_pulse_genisys_config
from sandbox.projects.sandbox_ci.pulse.resource_finder import ResourceTemplatesFinder
from sandbox.projects.sandbox_ci.pulse.resources import PulseBinary, PulseStaticLogs, PulseStaticDiff
from sandbox.projects.sandbox_ci.pulse.statface import PulseStaticStatfaceData
from sandbox.projects.sandbox_ci.pulse.pulse_static.assets_json_handler import AssetsJsonHandler, \
    ASSETS_JSON_HANDLER_PARAMS
from sandbox.projects.sandbox_ci.task import BasePulseTask


class PulseStatic(BasePulseTask):
    """Check static file sizes"""

    VIEWER_DIR = 'diff-viewer'

    class Requirements(BasePulseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BasePulseTask.Parameters):
        with sdk2.parameters.Group('Pulse binaries') as binaries_block:
            pulse_static_binary = sdk2.parameters.Resource(
                'Pulse Static binary',
                resource_type=PulseBinary,
                required=False,
            )

            pulse_report_binary = sdk2.parameters.Resource(
                'Pulse Report binary',
                resource_type=PulseBinary,
                required=False,
            )

            diff_viewer_binary = sdk2.parameters.Resource(
                'Diff Viewer binary',
                resource_type=PulseBinary,
                required=False,
                default=None,
            )

        with sdk2.parameters.Group('Static packages') as templates_block:
            base_static_package = sdk2.parameters.Resource(
                'Base static package',
            )

            actual_static_package = sdk2.parameters.Resource(
                'Actual static package',
                required=True,
            )

            check_static_files = parameters.check_static_files()

            use_assets_json = sdk2.parameters.Bool(
                'Use assets.json config',
                default=False,
                description='Использовать конфиг assets.json для формирования списка файлов статики'
            )

            with use_assets_json.value[True]:
                use_assets_json_v2 = sdk2.parameters.Bool(
                    'Use assets.json config V2',
                    default=False,
                    description='Использовать конфиги assets.json, сгенерированные новой сборкой Webpack с разделением по платформам и ручным main chunk'
                )

            raw_mode = sdk2.parameters.Bool(
                'Compare raw sizes instead of gzipped',
                default=False,
            )

        with sdk2.parameters.Output():
            results = sdk2.parameters.JSON('Results')

            final_pulse_static_binary = sdk2.parameters.Resource('Final Pulse Static binary')
            final_pulse_report_binary = sdk2.parameters.Resource('Final Pulse Report binary')
            final_diff_viewer_binary = sdk2.parameters.Resource('Final Diff Viewer binary')

    class Context(BasePulseTask.Context):
        report_html = ''

    @staticmethod
    def format_github_context():
        return u'Pulse Static'

    @classproperty
    def github_context(self):
        return self.format_github_context()

    @classproperty
    def task_label(self):
        return 'pulse.static'

    @singleton_property
    def base_static_package(self):
        return self.Parameters.base_static_package or self._find_static_package(self.base_static_package_id)

    @singleton_property
    def actual_static_package(self):
        return self.Parameters.actual_static_package or self._find_static_package(self.actual_static_package_id)

    def _find_static_package(self, resource_id):
        return next(iter(sdk2.Resource.find(id=resource_id).limit(1)), None)

    @singleton_property
    def base_static_package_id(self):
        if self.Parameters.base_static_package:
            return self.Parameters.base_static_package.id

        logging.debug('base static package is not specified, trying to find resource')

        return self._resource_templates_finder.base_static_resource_id()

    @singleton_property
    def actual_static_package_id(self):
        if self.Parameters.actual_static_package:
            return self.Parameters.actual_static_package.id

        logging.debug('actual static package is not specified, trying to find resource')

        return self._resource_templates_finder.actual_static_resource_id()

    @singleton_property
    def _resource_templates_finder(self):
        return ResourceTemplatesFinder(self.parent)

    @singleton_property
    def cache_parameters(self):
        parameters = super(PulseStatic, self).cache_parameters

        parameters.update(
            base_static_package=self.base_static_package_id,
            actual_static_package=self.actual_static_package_id,
        )

        return parameters

    @property
    def _report(self):
        return self.Context.report_html

    @property
    def project_dir(self):
        # fiji and web4 has different project layouts
        if self.project_name == 'fiji':
            return ''

        # что за нерпа?
        if self.project_name == 'nerpa':
            return 'nerpa/projects'

        return self.project_name

    @property
    def use_assets_json_v2(self):
        return self.Parameters.use_assets_json_v2

    @singleton_property
    def _static_files_names(self):
        """
        :rtype: dict
        """
        return load_pulse_genisys_config(self.project_name, 'static_files')

    def _process_assets_json_file(self, static_path, assets_json_config):
        file = assets_json_config
        params = {}

        if isinstance(assets_json_config, dict):
            file = assets_json_config.get('file')
            params.update(
                {
                    k: v
                    for k, v in assets_json_config.iteritems()
                    if k in ASSETS_JSON_HANDLER_PARAMS and v is not None
                }
            )

        logging.info('Processing assets.json file: {}'.format(file))
        try:
            with open(os.path.join(static_path, file)) as fd:
                params['config'] = json.load(fd)
                return params
        except IOError:
            logging.warn('Unable to process assets.json file {}'.format(file))
            return

    @in_case_of('use_assets_json_v2', 'process_assets_json_v2')
    def process_assets_json(self, static_package):
        static_path = os.path.join(static_package, 'static', self.project_dir)
        assets_json_config = load_pulse_genisys_config(self.project_name, 'static_files_config')

        assets_json_handler = AssetsJsonHandler()

        if isinstance(assets_json_config, list):
            for config_item in assets_json_config:
                config = self._process_assets_json_file(static_path, config_item)
                if config:
                    assets_json_handler.parse(**config)
        else:
            config = self._process_assets_json_file(static_path, assets_json_config)
            if config:
                assets_json_handler.parse(**config)

        create_symlinks(assets_json_handler.get_symlinks(), static_path)

        return assets_json_handler.get_files()

    def process_assets_json_v2(self, static_package):
        static_path = os.path.join(static_package, 'static', self.project_dir)
        assets_json_config = load_pulse_genisys_config(self.project_name, 'static_files_config_v2')

        assets_json_handler = AssetsJsonHandler()

        for config_item in assets_json_config:
            config = self._process_assets_json_file(static_path, config_item)
            if config:
                assets_json_handler.parse(**config)

        create_symlinks(assets_json_handler.get_symlinks(), static_path)

        return assets_json_handler.get_files()

    def on_enqueue(self):
        super(PulseStatic, self).on_enqueue()
        self._add_tags()

    def execute(self):
        with self.profiler.actions.preparation('Preparation'):
            base_static_package, actual_static_package = self._sync_static_packages()

            pulse_static_binary = self._sync_pulse_static_binary()
            pulse_report_binary = self._sync_pulse_report_binary()
            diff_viewer_path = self._sync_diff_viewer()

            static_file_patterns = load_pulse_genisys_config(self.project_name, 'static_files')

            if self.Parameters.use_assets_json:
                base_assets_files = self.process_assets_json(base_static_package)
                actual_assets_files = self.process_assets_json(actual_static_package)

                static_file_patterns = merge_file_configs(
                    static_file_patterns,
                    base_assets_files,
                    actual_assets_files,
                )

            logging.debug('Using static file patterns: %s', static_file_patterns)

        with self.profiler.actions.diff_creation('Diff creation'):
            self._create_diff(base_static_package, actual_static_package, diff_viewer_path, static_file_patterns)

        with self.profiler.actions.measurement('Measurement'):
            self._run_pulse_static(
                pulse_static_binary, actual_static_package,
                base_static_package, static_file_patterns
            )

        with self.profiler.actions.creating_report('Creating report'):
            self._run_pulse_report(pulse_report_binary)

        with self.profiler.actions.report_to_stat('Report to Stat'):
            if self.should_report_to_stat:
                self._send_statface_report_safe()

        with self.profiler.actions.limits_checking('Limits checking'):
            self.check_exceeded_limits()

    def get_limits_excesses(self):
        """
        :return: Массив строк, которые описывают, какие лимиты превышены и на сколько
        :rtype: list of str
        """
        with open(self._exceeded_json_path) as fp:
            return json.load(fp)

    def _sync_static_packages(self):
        base_static_package = self._sync_base_static_package()
        actual_static_package = self._sync_actual_static_package()

        return base_static_package, actual_static_package

    def _run_pulse_static(
            self, pulse_static_binary, actual_static_package, base_static_package,
            static_file_patterns
    ):
        static_patterns_json = os.path.join(os.getcwd(), 'static-file-patterns.json')
        with open(static_patterns_json, 'w') as fp:
            json.dump(static_file_patterns, fp)

        limits = load_pulse_genisys_config(self.project_name, 'limits')

        project_limits_delta = limits.get('static_delta', {})
        project_limits_delta_json = os.path.join(os.getcwd(), 'project-limits.json')
        with open(project_limits_delta_json, 'w') as fp:
            json.dump(project_limits_delta, fp)

        project_limits_file_size = limits.get('static_file_size', {})

        self._exceeded_json_path = exceeded_json = os.path.join(os.getcwd(), 'exceeded.json')

        static_logs_dir = os.path.join(os.getcwd(), 'pulse-static-logs')
        os.makedirs(static_logs_dir)

        static_out_path = os.path.join(static_logs_dir, 'stdout.log')
        static_error_path = os.path.join(static_logs_dir, 'stderr.log')
        self._report_path = report_path = os.path.join(static_logs_dir, 'report.json')

        with open(static_out_path, 'w') as o_fp, open(static_error_path, 'w') as e_fp:
            args = [
                str(pulse_static_binary),
                '--project-name=%s' % self.project_name,
                '--package-path-base=%s' % base_static_package,
                '--package-path-actual=%s' % actual_static_package,
                '--file-patterns=%s' % static_patterns_json,
                '--project-limits=%s' % project_limits_delta_json,
                '--excesses-out=%s' % exceeded_json,
                '--report-out=%s' % report_path,
                '--differ-base-url=%s' % self._diff_viewer_url,
            ]

            if self.Parameters.check_static_files:
                args.append('--check-files')

            if self.Parameters.raw_mode:
                args.append('--raw-mode')

            if project_limits_file_size:
                project_limits_file_size_json = os.path.join(os.getcwd(), 'project-limits-file-size.json')
                with open(project_limits_file_size_json, 'w') as fp:
                    json.dump(project_limits_file_size, fp)
                args.append('--project-limits-file-size=%s' % project_limits_file_size_json)

            p = Popen(args, stdout=o_fp, stderr=e_fp)

            return_code = p.wait()

            pulse_static_logs_resource = PulseStaticLogs(
                task=self,
                description='Pulse Static logs',
                type='pulse-static',
                path=static_logs_dir,
            )

            sdk2.ResourceData(pulse_static_logs_resource).ready()

            if return_code != 0:
                raise RuntimeError('pulse-static exited with code %s' % return_code)

            with open(report_path) as fp:
                self._report_metrics = json.load(fp)

            self.Parameters.results = self._report_metrics

    def _run_pulse_report(self, pulse_report_binary):
        static_logs_dir = os.path.join(os.getcwd(), 'pulse-report-logs')
        os.makedirs(static_logs_dir)

        static_out_path = os.path.join(static_logs_dir, 'stdout.log')
        static_error_path = os.path.join(static_logs_dir, 'stderr.log')

        report_path = os.path.join(static_logs_dir, 'report.html')

        with open(static_out_path, 'w') as o_fp, open(static_error_path, 'w') as e_fp:
            p = Popen((
                str(pulse_report_binary),
                '--stats=%s' % self._report_path,
                '--report-out=%s' % report_path,
                '--format=html',
            ), stdout=o_fp, stderr=e_fp)

            return_code = p.wait()

            pulse_report_logs_resource = PulseStaticLogs(
                task=self,
                description='Pulse Report logs',
                type='pulse-report',
                path=static_logs_dir,
            )

            sdk2.ResourceData(pulse_report_logs_resource).ready()

            if return_code != 0:
                raise RuntimeError('pulse-report exited with code %s' % return_code)

            with open(report_path) as fp:
                self.Context.report_html = fp.read()

    def _send_statface_report(self):
        # noinspection PyUnresolvedReferences
        from statface_client.report import StatfaceReportConfig

        current_datetime = datetime.datetime.now()

        statface_data = PulseStaticStatfaceData(current_datetime, self._report_metrics, self.id)

        report_name = '{}_dev'.format(self.project_name)
        report = self.statface.get_report('Yandex.Productivity/pulse-static/{}'.format(report_name))

        config = StatfaceReportConfig(
            title=report_name,
            dimensions=[
                ('fielddate', 'date'),
                ('platform', 'string'),
                ('filename', 'string'),
            ],
            measures=statface_data.measures()
        )

        report.upload_config(config)

        rows = statface_data.get_rows_for_statface()

        report.upload_data('s', rows)

    def _sync_actual_static_package(self):
        return self._sync_static_package(
            self.actual_static_package,
            'static_actual'
        )

    def _sync_base_static_package(self):
        return self._sync_static_package(
            self.base_static_package,
            'static_base'
        )

    def _sync_pulse_static_binary(self):
        resource = self._find_pulse_binary('pulse-static', self.Parameters.pulse_static_binary)
        if not resource:
            raise RuntimeError('There is no Pulse Static binary')
        self.Parameters.final_pulse_static_binary = resource.id

        return sdk2.ResourceData(resource).path

    def _sync_pulse_report_binary(self):
        resource = self._find_pulse_binary('pulse-report', self.Parameters.pulse_report_binary)
        if not resource:
            raise RuntimeError('There is no Pulse Report binary')
        self.Parameters.final_pulse_report_binary = resource.id

        return sdk2.ResourceData(resource).path

    def _sync_diff_viewer(self):
        resource = self._find_diff_viewer()

        binary_path = str(sdk2.ResourceData(resource).path)
        diff_viewer_dir = os.path.join(os.getcwd(), 'viewer')

        with tarfile.open(binary_path) as tar:
            tar.extractall(path=diff_viewer_dir)

        if not os.path.exists(diff_viewer_dir) or not os.path.isdir(diff_viewer_dir):
            raise RuntimeError('There is no pulse-diff-viewer')

        return diff_viewer_dir

    def _find_diff_viewer(self):
        resource = self.Parameters.diff_viewer_binary
        if resource:
            self.Parameters.final_diff_viewer_binary = resource
            return resource

        resource = sdk2.Resource.find(
            type=PulseBinary,
            attrs={
                'released': 'stable',
                'project': 'pulse-diff-viewer',
            }
        ).first()

        if resource:
            self.Parameters.final_diff_viewer_binary = resource
            return resource

        resource = sdk2.Resource.find(
            type=PulseBinary,
            attrs={'project': 'pulse-diff-viewer'}
        ).first()

        if resource:
            self.Parameters.final_diff_viewer_binary = resource
            return resource

        if not resource:
            raise RuntimeError('There is no Diff Viewer binary')

    def _find_pulse_binary(self, project, resource):
        if resource:
            return resource

        resource = sdk2.Resource.find(
            type=PulseBinary,
            attrs={
                'platform': 'linux',
                'released': 'stable',
                'project': project,
            }
        ).first()

        if resource:
            return resource

        resource = sdk2.Resource.find(
            type=PulseBinary,
            attrs={
                'platform': 'linux',
                'project': project,
            }
        ).first()

        if resource:
            return resource

    def _sync_static_package(self, resource, target):
        self.artifacts.unpack_build_artifacts([resource], sdk2.path.Path(target))
        return str(self.path(target))

    def _create_diff(self, base_static_package, actual_static_package, diff_viewer_path, static_file_patterns):
        viewer_dir = os.path.join(os.getcwd(), self.VIEWER_DIR)
        os.makedirs(viewer_dir)

        file_names = []
        for platform_files in static_file_patterns.itervalues():
            file_names.extend(platform_files)

        base_platform_dir = os.path.join(base_static_package, 'static')
        base_dir = find_dir(base_platform_dir, file_names[0])
        actual_platform_dir = os.path.join(actual_static_package, 'static')
        actual_dir = find_dir(actual_platform_dir, file_names[0])

        if base_dir is None or actual_dir is None:
            self._diff_viewer_url = ''
            self.set_info('WARN: Unable to create diff because of error: %s' % 'project files not found')
            return None

        # copy viewer files
        shutil.copytree(diff_viewer_path, os.path.join(viewer_dir, 'viewer'))

        # У разных проектов разные форматы расположения проектов внутри пакета
        project_path = base_dir.replace(base_platform_dir, '').strip('/')

        dest_base = os.path.join(viewer_dir, 'base', 'static', project_path)
        dest_actual = os.path.join(viewer_dir, 'actual', 'static', project_path)

        for file_name in file_names:
            # copy files from base templates
            for found in glob.iglob(os.path.join(base_dir, file_name)):
                copy_file(found, dest_base, os.path.relpath(found, base_dir))

            # copy files from actual templates
            for found in glob.iglob(os.path.join(actual_dir, file_name)):
                copy_file(found, dest_actual, os.path.relpath(found, actual_dir))

        try:
            resource = PulseStaticDiff(self, 'Pulse Static Diff', path=viewer_dir)
            sdk2.ResourceData(resource).ready()
            self._diff_viewer_url = resource.http_proxy
        except Exception as e:
            self._diff_viewer_url = ''
            logging.error(e)
            self.set_info('WARN: Unable to create diff because of error: %s' % e)

    def _add_tags(self):
        project_tag = self.Parameters.project.upper()

        if project_tag not in self.Parameters.tags:
            self.Parameters.tags += [project_tag]


def copy_file(src, dst, file):
    directory = os.path.dirname(file)
    dst_file = os.path.join(dst, file)

    if not os.path.exists(os.path.join(dst, directory)):
        os.makedirs(os.path.join(dst, directory))

    if os.path.exists(src):
        shutil.copyfile(src, dst_file)


def find_dir(start_dir, file_name):
    MAX_FOLDER_DEEP_LEN = 20
    found_dir = []
    deep = ''

    while len(found_dir) == 0:
        if len(deep) > MAX_FOLDER_DEEP_LEN:
            break

        pattern = os.path.join(start_dir, deep, file_name)
        found_dir = glob.glob(pattern)
        deep = os.path.join(deep, '*')

    return None if len(found_dir) == 0 else subtract_path(found_dir[0], file_name)


def subtract_path(path, pattern):
    head, tail = os.path.split(path)

    while not fnmatch.fnmatch(tail, pattern):
        if not head:
            break

        head, tail_part = os.path.split(head)
        tail = os.path.join(tail_part, tail)

    return head


def create_symlinks(symlinks, root_path):
    for src, dest in symlinks:
        src_full = os.path.join(root_path, src)
        dest_full = os.path.join(root_path, dest)
        try:
            os.symlink(src_full, dest_full)
        except Exception as e:
            logging.error('Unable to create symlink from %s to %s: %s', src_full, dest_full, e)


def merge_file_configs(*configs):
    result = defaultdict(list)

    for config in configs:
        for platform, files in config.iteritems():
            for file_item in files:
                if file_item not in result[platform]:
                    result[platform].append(file_item)

    return result
