# -*- coding: utf-8 -*-
import os
from collections import defaultdict, namedtuple

DEFAULT_PLATFORM = 'common'
DEFAULT_FREEZE_DIR = 'freeze'
DEFAULT_BUILD_DIR = '.build'
DEFAULT_MAIN_REACT_BUNDLE_ALIAS = 'main-react-bundle'

ASSETS_JSON_HANDLER_PARAMS = (
    'platform',
    'service',
    'build_dir',
    'freeze_dir',
    'use_main_chunk',
    'use_extended_paths',
)

Asset = namedtuple('Asset', ('name', 'js', 'js_source', 'css', 'css_source', 'platform'))


class AssetsJsonHandler(object):
    u"""Парсер артефактов assets.json, генерируемых сборкой Webpack+Taburet."""

    def __init__(self):
        self._assets = []

    def parse(
        self,
        config,
        platform=DEFAULT_PLATFORM,
        service=None,
        build_dir=DEFAULT_BUILD_DIR,
        freeze_dir=DEFAULT_FREEZE_DIR,
        use_main_chunk=False,
        use_extended_paths=False,
    ):
        u"""Обработать assets.json

        :param config: dict-представление assets.json
        :type config: dict
        :param platform: явное указание платформы
        :type platform: str, optional
        :param service: имя сервиса, которое будет выводиться в тайтле платформы
        :type service: str, optional
        :param build_dir:
        :type build_dir: str, optional
        :param freeze_dir:
        :type freeze_dir: str, optional
        :param use_main_chunk:
        :type use_main_chunk: bool, optional
        :param use_extended_paths: Использовать расширенные пути для исходных JS-файлов у бандлов, по умолчанию False
        :type use_extended_paths: bool, optional
        """
        if use_main_chunk:
            self._assets.append(self._get_main_chunk(config, platform))

        for entry_name, entry_data in config.get('entries').iteritems():
            entry_platform = self._get_platform(entry_name, service)
            entry_main_chunk_id = entry_data.get('main', {}).get('ru')

            entry_main_chunk_file_name = self._get_chunk_file_name(entry_main_chunk_id, config)
            if entry_main_chunk_file_name is None:
                continue
            entry_main_chunk_base = os.path.join(
                freeze_dir,
                os.path.dirname(entry_name) if use_extended_paths else '',
                entry_main_chunk_file_name,
            )

            entry_main_chunk_js = '{}.js'.format(entry_main_chunk_base)
            entry_main_chunk_css = '{}.css'.format(entry_main_chunk_base)

            entry_asset = Asset(
                name=entry_name,
                js=os.path.join(build_dir, '{}.js'.format(entry_name)),
                js_source=entry_main_chunk_js,
                css=os.path.join(build_dir, '{}.css'.format(entry_name)),
                css_source=entry_main_chunk_css,
                platform='{} bundles'.format(entry_platform),
            )

            self._assets.append(entry_asset)

        for entry_name, entry_data in config.get('componentsExperiments', {}).iteritems():
            entry_platform = self._get_platform_for_experiment(entry_name)
            # В assets-*.json могут оказаться эксперименты всех платформ - игнорируем остальные платформы
            if entry_platform != platform:
                continue

            entry_hash = entry_data.get('hash')
            # Совместимость с предыдущей версией assets-*.json
            if entry_hash is None:
                continue

            entry_asset = Asset(
                name=entry_name,
                js=os.path.join(build_dir, '{}.js'.format(entry_name)),
                js_source=os.path.join(build_dir, '{}.{}.js'.format(entry_name, entry_hash)),
                # CSS может отсутствовать
                css=os.path.join(build_dir, '{}.css'.format(entry_name)),
                css_source=os.path.join(build_dir, '{}.{}.css'.format(entry_name, entry_hash)),
                platform='{} component experiments'.format(platform),
            )

            self._assets.append(entry_asset)

    def get_files(self):
        result = defaultdict(list)
        for asset in self._assets:
            if asset.css:
                result[asset.platform].append(asset.css)
            if asset.js:
                result[asset.platform].append(asset.js)
        return dict(result)

    def get_symlinks(self):
        links = []

        for asset in self._assets:
            if asset.js_source:
                links.append((asset.js_source, asset.js))

            if asset.css_source:
                links.append((asset.css_source, asset.css))

        return links

    def _get_platform(self, entry_name, service):
        platform = entry_name.split('@')[1]
        if service:
            platform = '{}-{}'.format(service, platform)
        return platform

    def _get_platform_for_experiment(self, entry_name):
        return os.path.basename(entry_name)

    def _get_chunk_file_name(self, id, config):
        chunks = config.get('chunks')
        chunk = chunks.get(id) if isinstance(chunks, dict) else chunks[id]
        chunk_js = chunk[0].get('js', {})

        if isinstance(chunk_js, dict):
            chunk_url = chunk_js.get('url')
            chunk_base_name = os.path.basename(chunk_url)
            return chunk_base_name.rsplit(os.path.extsep, 1)[0]

        return None

    def _get_main_chunk(self, config, platform):
        main = config.get('main')
        main_chunk_url = main.get('js', {}).get('url') if isinstance(main, dict) else main
        return Asset(
            name='main-react-bundle',
            js=os.path.join(DEFAULT_FREEZE_DIR, '{}-{}.js'.format(DEFAULT_MAIN_REACT_BUNDLE_ALIAS, platform)),
            js_source=os.path.join(DEFAULT_FREEZE_DIR, os.path.basename(main_chunk_url)),
            css=None,
            css_source=None,
            platform=platform,
        )
