# coding=utf-8
import logging
import os
import shutil

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.sandbox_ci.pulse.pulse_shooter import PulseShooter
from sandbox.projects.sandbox_ci.pulse.resources import SearchParsedAccessLog, ReportRendererPhantomData, \
    ReportRendererPhantomDataApphost, ReportRendererPlan, ReportRendererPlanJson, ReportRendererPlanApphost
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

RELEASE_TAG = 'AMMO_RELEASE'


class ReleasePulseShooterAmmo(TasksResourceRequirement, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String(
            'Project',
            required=True,
        )

        platform = sdk2.parameters.String(
            'Platform',
            required=True,
        )

        parsed_access_log = sdk2.parameters.Resource(
            'Parsed access-log resource',
            resource_type=SearchParsedAccessLog,
        )

        report_renderer_plan_phantom = sdk2.parameters.Resource(
            'Report renderer Phantom classic plan',
            resource_type=ReportRendererPhantomData,
        )

        report_renderer_plan_dolbilka = sdk2.parameters.Resource(
            'Report renderer Dolbilka classic plan',
            resource_type=ReportRendererPlan,
        )

        report_renderer_plan_json = sdk2.parameters.Resource(
            'Report renderer JSON classic plan',
            resource_type=ReportRendererPlanJson,
        )

        report_renderer_plan_phantom_apphost = sdk2.parameters.Resource(
            'Report renderer Phantom Apphost plan',
            resource_type=ReportRendererPhantomDataApphost,
        )

        report_renderer_plan_dolbilka_apphost = sdk2.parameters.Resource(
            'Report renderer Dolbilka Apphost plan',
            resource_type=ReportRendererPlanApphost,
        )

    class Context(sdk2.Context):
        report_renderer_plan_phantom_id = None
        report_renderer_plan_phantom_apphost_id = None

        pulse_shooter_classic_id = None
        pulse_shooter_apphost_id = None

    def on_execute(self):
        with self.memoize_stage.test_new_ammo(commit_on_entrance=False):
            self._claim_resources()
            self._run_tests()

        with self.memoize_stage.check_test_results(commit_on_entrance=False):
            self._check_test_results()

    def _claim_resources(self):
        # noinspection PyTypeChecker
        self.Context.report_renderer_plan_phantom_id = self._claim_resource(
            self.Parameters.report_renderer_plan_phantom
        )
        # noinspection PyTypeChecker
        self.Context.report_renderer_plan_phantom_apphost_id = self._claim_resource(
            self.Parameters.report_renderer_plan_phantom_apphost
        )

        resources = [
            self.Parameters.parsed_access_log,
            self.Parameters.report_renderer_plan_dolbilka,
            self.Parameters.report_renderer_plan_json,
            self.Parameters.report_renderer_plan_dolbilka_apphost,
        ]

        for resource in resources:
            # noinspection PyTypeChecker
            self._claim_resource(resource)

    def _claim_resource(self, resource):
        if not resource:
            return

        res_data = sdk2.ResourceData(resource)

        old_path = str(res_data.path)
        new_path = os.path.join(os.getcwd(), os.path.basename(old_path))

        logging.debug('Copy resource file from %s to %s', old_path, new_path)
        shutil.copy(old_path, new_path)

        attributes = resource.__attrs__.copy()
        attributes.update(dict(
            task=self,
            path=new_path,
            description=resource.description,
        ))

        logging.debug('New resource (%s) attributes: %s', resource, attributes)

        new_res = type(resource)(**attributes)
        sdk2.ResourceData(new_res).ready()

        return new_res.id

    def _run_tests(self):
        test_tasks = []

        if self.Parameters.report_renderer_plan_phantom:
            task_id = self._run_test(apphost_mode=False)
            if task_id:
                test_tasks.append(task_id)
                self.Context.pulse_shooter_classic_id = task_id

        if self.Parameters.report_renderer_plan_phantom_apphost:
            task_id = self._run_test(apphost_mode=True)
            if task_id:
                test_tasks.append(task_id)
                self.Context.pulse_shooter_apphost_id = task_id

        if test_tasks:
            raise sdk2.WaitTask(
                test_tasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

    def _run_test(self, apphost_mode):
        ammo_id = self.Context.report_renderer_plan_phantom_id
        if apphost_mode:
            ammo_id = self.Context.report_renderer_plan_phantom_apphost_id

        if not ammo_id:
            return

        project = str(self.Parameters.project)
        service = ''
        platform = str(self.Parameters.platform)

        if project == 'fiji':
            service, platform = platform.split('-', 1)

        input_parameters = dict(
            project=project,
            platform=platform,
            service=service,
        )

        base_task = sdk2.Task.find(
            task_type=PulseShooter,
            status=ctt.Status.SUCCESS,
            input_parameters=input_parameters,
            children=True,
        ).order(
            -sdk2.Task.id
        ).first()

        if not base_task:
            msg = 'There is no base task for chosen params. So OK with new ammo. Params: %s' % input_parameters
            self.set_info(msg)
            return

        new_task = PulseShooter(
            self,
            description=self.Parameters.description,
            tags=base_task.Parameters.tags + [RELEASE_TAG],

            project=project,
            service=service,
            platform=platform,

            project_github_owner=base_task.Parameters.project_github_owner,
            project_github_repo=base_task.Parameters.project_github_repo,
            project_build_context=base_task.Parameters.project_build_context,
            project_tree_hash=base_task.Parameters.project_tree_hash,

            reuse_task_cache=False,
            reuse_subtasks_cache=False,

            environ=base_task.Parameters.environ,

            # Use checked base twice for more stability
            base_templates_package=base_task.Parameters.base_templates_package,
            actual_templates_package=base_task.Parameters.base_templates_package,

            pulse_shooter_ammo_base=ammo_id,

            send_comment_to_searel=False,
            is_release=False,
            report_github_statuses=False,

            apphost_mode=apphost_mode,
            pulse_shooter_workers=base_task.Parameters.pulse_shooter_workers,
        )

        new_task.enqueue()

        return new_task.id

    def _check_test_results(self):
        self._check_task(self.Context.pulse_shooter_classic_id)
        self._check_task(self.Context.pulse_shooter_apphost_id)

    def _check_task(self, task_id):
        if not task_id:
            return

        task = sdk2.Task[task_id]
        if not task:
            raise TaskFailure('Task %s not found' % task_id)

        if task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Test task %s ended with status %s' % (task_id, task.status))
