# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State
from sandbox.projects.resource_types import REPORT_STATIC_PACKAGE
from sandbox.projects.report_renderer.resource_types import REPORT_RENDERER_BUNDLE
from sandbox.projects.sandbox_ci.resources import WebMicroPackage
from sandbox.projects.sandbox_ci.resources import WebNewsMicroPackage
from sandbox.projects.sandbox_ci.resources import WebTurboMicroPackage
from sandbox.projects.sandbox_ci.resources import WebGrannyMicroPackage, WebGrannyFreezeMicroPackage
from sandbox.projects.sandbox_ci.resources import ImagesMicroPackage
from sandbox.projects.sandbox_ci.resources import VideoMicroPackage
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.pulse import const as pulse_const
from sandbox.projects.sandbox_ci.pulse.resources import (
    PulseBinary, ReportRendererPhantomData,
    ReportRendererPhantomDataApphost,
)


def get_dynamic_resource_attr_type_by_project(project_name):
    """
    :type project_name: str
    :return: str Name of package with dynamic
    """
    res_type = {
        'web4': 'web4-micropackage',
        'turbo': 'turbo-micropackage',
        'granny': 'granny-micropackage',

        'fiji': 'fiji-micropackage',

        'news': 'news-templates',
    }.get(project_name)

    if res_type is None:
        raise Exception('There is no resource type for project {}'.format(project_name))

    return res_type


def get_dynamic_resource_type_by_project(project_name, sub_project=None):
    name = '{project_name}{postfix_name}'.format(
        project_name=project_name,
        postfix_name='_{}'.format(sub_project) if sub_project else '',
    )

    task_type = {
        'web4': WebMicroPackage,
        'turbo': WebTurboMicroPackage,
        'granny': WebGrannyMicroPackage,
        'granny_freeze': WebGrannyFreezeMicroPackage,
        'news': WebNewsMicroPackage,
        'fiji': SANDBOX_CI_ARTIFACT,
        'fiji_images': ImagesMicroPackage,
        'fiji_video': VideoMicroPackage,
    }.get(name)

    if task_type is None:
        raise Exception('There is no resource type for project {}'.format(name))

    return task_type


class ResourceTemplatesFinder(object):

    def __init__(self, master_task):
        """
        :type master_task: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
        """
        self._master_task = master_task
        self._project = self._master_task.project_name
        self._sub_project = self._master_task.sub_project

    def base_dynamic_resource_id(self, **query):
        return self._find_base_resource_id(self._templates_resource_type, self._templates_attr_type, query)

    def base_static_resource_id(self, **query):
        return self._find_base_resource_id(REPORT_STATIC_PACKAGE, self._static_attr_type, query)

    def base_image_resource_id(self, **query):
        return self._find_base_resource_id(SANDBOX_CI_ARTIFACT, self._image_attr_type, query)

    def actual_dynamic_resource_id(self):
        return self._master_task.get_registered_artifact_resource(self._templates_attr_type).id

    def actual_static_resource_id(self):
        return self._master_task.get_registered_artifact_resource(self._static_attr_type).id

    def actual_image_resource_id(self):
        return self._master_task.get_registered_artifact_resource(self._image_attr_type).id

    def _find_base_resource_id(self, res_type, attr_type, query=None):
        return self._find_resource_by_query(dict(
            query or {},
            resource_type=res_type,
            state=State.READY,
            type=attr_type,
            project_tree_hash=self._base_tree_hash(),
        )).id

    def _find_resource_by_query(self, query):
        params = dict(query, YENV='testing')

        if getattr(self._master_task.Parameters, 'is_release', False):
            params.update(YENV='production')

        resource = self._master_task.artifacts.get_resource(**params)

        if not resource:
            raise ResourceNotFoundException('Unable to find base resource by query "{}"'.format(params))

        return resource

    def _base_tree_hash(self):
        tree_hash = self._master_task.Parameters.project_range_base_tree_hash

        if not tree_hash:
            raise BaseHashNotFoundException('Master task parameter project_range_base_tree_hash is undefined')

        return tree_hash

    @property
    def _templates_attr_type(self):
        return get_dynamic_resource_attr_type_by_project(self._project)

    @property
    def _templates_resource_type(self):
        return get_dynamic_resource_type_by_project(self._project, self._sub_project)

    @property
    def _static_attr_type(self):
        return '{}-static'.format(self._project)

    @property
    def _image_attr_type(self):
        return '{}.squashfs'.format(self._project)


class ResourceNotFoundException(TaskFailure):
    pass


class BaseHashNotFoundException(TaskFailure):
    pass


def find_latest_report_renderer_bundle():
    resource = sdk2.Resource.find(
        type=REPORT_RENDERER_BUNDLE,
        attrs={'released': 'stable', 'platform': 'linux'},
    ).order(-sdk2.Resource.id).first()

    if resource:
        return resource.id


def find_latest_pulse_binary(binary_name, platform='linux'):
    resource = sdk2.Resource.find(
        type=PulseBinary,
        attrs={'platform': platform, 'released': 'stable', 'project': binary_name},
    ).order(-sdk2.Resource.id).first()

    if resource:
        return resource.id


class LatestPulseBinaryRegistry(object):
    def __init__(self):
        self.shooter = find_latest_pulse_binary(pulse_const.PULSE_BINARY_SHOOTER)
        self.static = find_latest_pulse_binary(pulse_const.PULSE_BINARY_STATIC)
        self.aggregator = find_latest_pulse_binary(pulse_const.PULSE_BINARY_AGGREGATOR)
        self.report = find_latest_pulse_binary(pulse_const.PULSE_BINARY_REPORT)
        self.html_differ = find_latest_pulse_binary(pulse_const.PULSE_BINARY_HTML_DIFFER)
        self.diff_viewer = find_latest_pulse_binary(pulse_const.PULSE_BINARY_DIFF_VIEWER, platform=None)


def find_latest_ammo(project, service, platform, apphost_mode):
    resource_type = ReportRendererPhantomData
    if apphost_mode:
        resource_type = ReportRendererPhantomDataApphost

    full_platform = platform
    if service:
        full_platform = '%s-%s' % (service, platform)

    resource = sdk2.Resource.find(
        type=resource_type,
        attrs={'project': project, 'platform': full_platform, 'released': 'stable'},
    ).order(-sdk2.Resource.id).first()

    if resource:
        return resource.id
