# -*- coding: utf-8 -*-
import re


class PulseStaticStatfaceData(object):
    COMMON_MEASURES = {
        'task_id': 'number',
        'size': 'number'
    }

    def __init__(self, current_datetime, report_data, task_id):
        self._current_datetime = current_datetime
        self._report_data = report_data
        self._task_id = task_id

    def get_rows_for_statface(self):
        files_metrics = []

        for platform_data in self._report_data:
            platform_name = platform_data['title']

            for file_data in platform_data['rows']:
                if not file_data['values']:
                    continue

                files_metrics.append(dict(
                    size=file_data['values'][0]['actual'],
                    filename=file_data['raw_name'],
                    platform=platform_name
                ))

        return self._extend_metrics_with_common_data(files_metrics)

    def measures(self):
        """
        Возращает декларацию схемы данных для statface
        :rtype: list
        """
        return self.COMMON_MEASURES.items()

    def _extend_metrics_with_common_data(self, metrics):
        """
        Расширяет уже сформированные данные, данными, которые идентичны для всех перцентилей
        :param metrics:
        :rtype : list of dict
        """

        def collect(item):
            return dict(item, **{
                'task_id': self._task_id,
                'fielddate': self._current_datetime.strftime('%Y-%m-%d %H:%M:%S'),
            })

        return map(collect, metrics)


class PulseShooterStatfaceData(object):
    """
    Подготавливает данные об обстреле для отправки в statface
    """
    REQUIRED_AGGREGATIONS = ('p25', 'p50', 'p75', 'p95', 'p99')

    COMMON_MEASURES = {
        'components': 'string',
        'task_id': 'number',
    }

    entry_prefix_blacklist_pattern = re.compile(r'\W+')

    def __init__(self, current_datetime, report_data, components, platform, task_id):
        """
        :type report_data: dict
        :type components: dict
        :type platform: string
        :type task_id: int

        Схема данных (report_data):
        report_data = (
            {
                'title': project_platform,
                'subtitle': 'Subtitle',
                'row_names': [p50, p80, …],
                'statistics': [1, 2, …],
            },
            …
        )
        """
        self._current_datetime = current_datetime
        self._report_data = report_data
        self._components = components
        self._platform = platform
        self._task_id = task_id

    def get_rows_for_statface(self):
        """
        Возвращает список метрик для base и actual обстрелов, для каждой из перцентилей
        :return: list
        """

        base = self._create_metrics('base')
        actual = self._create_metrics('actual')

        return self._extend_metrics_with_common_data(base + actual)

    def measures(self):
        """
        Возращает декларацию схемы данных для statface
        :rtype: list
        """
        metric_names = self._extract_metric_names()
        metric_measures = self._measures_by_metric_names(metric_names)

        return self.COMMON_MEASURES.items() + metric_measures.items()

    def _create_metrics(self, stage_name):
        result = []

        for aggregation_name in self.REQUIRED_AGGREGATIONS:
            metrics = self._extract_metrics_for_aggregated_data(aggregation_name, stage_name)
            metrics.update({
                'percentile': aggregation_name[1:],
                'chunk_type': stage_name,
            })
            result.append(metrics)

        return result

    def _extract_metrics_for_aggregated_data(self, aggregation_name, stage_name):
        """
        Формирует словарь со значениями метрик для конкретного перцентиля
        :type aggregation_name: str
        :type stage_name: str
        :rtype: dict

        {
            'metric_name': 22.1,
            ...
        }
        """
        metrics = {}

        for table_data in self._report_data:
            rows = table_data['rows']
            entry_prefix = self._get_entry_prefix(table_data)

            if rows:
                for row_data in rows:
                    metric_data = self._find_metric_aggregated_data(row_data['values'], aggregation_name)
                    if metric_data:
                        name = self._get_metric_name(entry_prefix, row_data)
                        metrics[name] = metric_data.get(stage_name)

        return metrics

    def _get_metric_name(self, entry_prefix, row_data):
        return entry_prefix + '_' + row_data['raw_name'] if entry_prefix else row_data['raw_name']

    def _extract_metric_names(self):
        metric_names = []

        for table_data in self._report_data:
            rows = table_data['rows']
            entry_prefix = self._get_entry_prefix(table_data)

            if rows:
                for row_data in rows:
                    name = self._get_metric_name(entry_prefix, row_data)
                    metric_names.append(name)

        return metric_names

    def _get_entry_prefix(self, table_data):
        return self.entry_prefix_blacklist_pattern.sub('_', table_data['subtitle'])

    def _extend_metrics_with_common_data(self, metrics):
        """
        Расширяет уже сформированные данные, данными, которые идентичны для всех перцентилей
        :param metrics:
        :rtype : list of dict
        """

        def collect(item):
            return dict(item, **{
                'platform': self._platform,
                'task_id': self._task_id,
                'components': self._format_components(),
                'fielddate': self._current_datetime.strftime('%Y-%m-%d %H:%M:%S'),
            })

        return map(collect, metrics)

    def _format_components(self):
        return '; '.join('%s:%s' % (n, v) for n, v in self._components.iteritems())

    @staticmethod
    def _find_metric_aggregated_data(metric_data, aggregation_name):
        """
        Находит объект со значениями для конкретной агрегации
        :param metric_data: list of dict
        :param aggregation_name: number
        :return:
        """
        return next((p for p in metric_data if p.get('aggregation_name', p.get('aggr_name')) == aggregation_name), None)

    @staticmethod
    def _measures_by_metric_names(metric_names):
        """
        :param metric_names: list of str
        :return: dict

        {
            'metric_name1': 'metric_type',
            ...
        }
        """
        return reduce(lambda measures, name: dict(measures, **{name: 'number'}), metric_names, {})
