# coding=utf-8
import logging

from sandbox.projects.sandbox_ci.pulse.pulse_shooter import PulseShooter
from sandbox.projects.sandbox_ci.pulse.pulse_shooter_custom import PulseShooterCustom
from sandbox.projects.sandbox_ci.pulse.pulse_static import PulseStatic
from sandbox.projects.sandbox_ci.pulse.ticount import Ticount
from sandbox.projects.sandbox_ci.decorators.skip_subtask import skip_subtask
from sandbox.projects.sandbox_ci.utils import request
from sandbox.projects.sandbox_ci.pulse.notification import notify_template_error
from sandbox.projects.sandbox_ci.pulse.resource_finder import (
    ResourceTemplatesFinder,
    BaseHashNotFoundException,
    ResourceNotFoundException,
    LatestPulseBinaryRegistry,
    find_latest_ammo, find_latest_report_renderer_bundle)

FAMILY = 'pulse'


def handle_exception(master_task, shooting_task, static_task, platforms, err):
    if type(platforms) is dict:
        for service, service_platforms in platforms.iteritems():
            for platform in service_platforms:
                service_platform = shooting_task.format_service_platform(platform, service)
                formatted_platform = shooting_task.format_platform(service_platform)

                notify_template_error(
                    master_task=master_task,
                    label=service_platform,
                    context=shooting_task.format_github_context(formatted_platform),
                )
    else:
        for platform in platforms:
            notify_template_error(
                master_task=master_task,
                label='pulse.{}'.format(platform),
                context=shooting_task.format_github_context(platform),
            )

    notify_template_error(
        master_task=master_task,
        label='pulse.static',
        context=static_task.format_github_context()
    )

    logging.error(err)


def create_pulse_subtasks(
        master_task,
        platforms,
        reusable=True,
        shoot_dynamic=False,
        shoot_static=False,
        waitable=True,
        base_dynamic_resource_id=None,
        base_static_resource_id=None,
        base_image_resource_id=None,
        pr_number=None,
        check_static_files=False,
        use_assets_json=False,
        use_assets_json_v2=False,
        notifications=None,
):
    """
    :type master_task: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
    :type platforms: list or dict
    :type reusable: bool
    :type shoot_dynamic: bool
    :type shoot_static: bool
    :type waitable: bool
    :type base_dynamic_resource_id: int
    :type base_static_resource_id: int
    :type pr_number: int or None
    :type check_static_files: bool
    :type notifications: list of sdk2.Notification
    :rtype: list of sandbox.projects.sandbox_ci.task.BasePulseTask.BasePulseTask
    """
    shoot_dynamic = bool(master_task.config.get_deep_value(['tests', 'pulse', 'shoot_dynamic'], shoot_dynamic))
    shoot_static = bool(master_task.config.get_deep_value(['tests', 'pulse', 'shoot_static'], shoot_static))

    shoot_classic = shoot_dynamic and bool(master_task.config.get_deep_value(['tests', 'pulse', 'classic'], True))
    shoot_apphost = shoot_dynamic and bool(master_task.config.get_deep_value(['tests', 'pulse', 'apphost'], False))

    run_ticount = bool(master_task.config.get_deep_value(['tests', 'ticount', 'enabled'], False))

    declare_pulse_shooter_custom = bool(
        master_task.config.get_deep_value(['tests', 'pulse', 'declare_custom'], False)
    )
    custom_apphost_mode = bool(
        master_task.config.get_deep_value(['tests', 'pulse', 'custom_apphost_mode'], True)
    )

    finder = ResourceTemplatesFinder(master_task)
    binary_registry = LatestPulseBinaryRegistry()

    project_vcs_ref = ''
    project_vcs_ref_list = getattr(master_task.Parameters, 'project_git_merge_ref', None)
    if project_vcs_ref_list:
        project_vcs_ref = str(project_vcs_ref_list[0])

    shooting_task = PulseShooter
    static_task = PulseStatic

    subtasks = []

    actual_dynamic_resource_id = None
    actual_static_resource_id = None
    actual_image_resource_id = None
    try:
        if shoot_dynamic:
            actual_dynamic_resource_id = finder.actual_dynamic_resource_id()
        if shoot_static:
            actual_static_resource_id = finder.actual_static_resource_id()
        if run_ticount:
            actual_image_resource_id = finder.actual_image_resource_id()
    except (BaseHashNotFoundException, ResourceNotFoundException) as err:
        logging.exception(err)
        handle_exception(master_task, shooting_task, static_task, platforms, err)
        return []

    try:
        if shoot_dynamic:
            base_dynamic_resource_id = base_dynamic_resource_id or finder.base_dynamic_resource_id()
        if shoot_static:
            base_static_resource_id = base_static_resource_id or finder.base_static_resource_id()
        if run_ticount:
            base_image_resource_id = base_image_resource_id or finder.base_image_resource_id()
        build_base_commit_subtask = None
    except BaseHashNotFoundException as err:
        logging.exception(err)
        handle_exception(master_task, shooting_task, static_task, platforms, err)
        return []
    except ResourceNotFoundException as err:
        logging.debug('Base resources not found, trying build base resources')
        base_dynamic_resource_id = None
        base_static_resource_id = None
        build_base_commit_subtask = master_task.create_build_base_commit_subtask(
            report_github_statuses=False,
            waitable=waitable,
        )
        subtasks.append(build_base_commit_subtask)

        if not build_base_commit_subtask:
            logging.exception(err)
            handle_exception(master_task, shooting_task, static_task, platforms, err)
            return []

    wait_output_parameters = {build_base_commit_subtask.id: 'is_artifacts_ready'} if build_base_commit_subtask else None
    reusable = False if build_base_commit_subtask else reusable

    if shoot_dynamic:
        pulse_shooter_common_params = dict(
            base_templates_package_id=base_dynamic_resource_id,
            actual_templates_package_id=actual_dynamic_resource_id,
            platforms=platforms,
            reusable=reusable,
            wait_output_parameters=wait_output_parameters,
            waitable=waitable,
            notifications=notifications,
            project_vcs_ref=project_vcs_ref,
            pr_number=pr_number,
        )

        if shoot_classic:
            subtasks += create_shooting_subtasks(
                master_task,
                binary_registry,
                apphost_mode=False,
                **pulse_shooter_common_params
            )

        if shoot_apphost:
            subtasks += create_shooting_subtasks(
                master_task,
                binary_registry,
                apphost_mode=True,
                **pulse_shooter_common_params
            )

    if declare_pulse_shooter_custom:
        declare_pulse_shooter_custom_draft(
            master_task,
            binary_registry,
            base_templates_package_id=base_dynamic_resource_id,
            actual_templates_package_id=actual_dynamic_resource_id,
            notifications=notifications,
            apphost_mode=custom_apphost_mode,
            project_vcs_ref=project_vcs_ref,
            pr_number=pr_number,
        )

    if shoot_static:
        static_subtask = create_static_subtask(
            master_task,
            binary_registry,
            base_static_package_id=base_static_resource_id,
            actual_static_package_id=actual_static_resource_id,
            reusable=reusable,
            wait_output_parameters=wait_output_parameters,
            waitable=waitable,
            check_static_files=check_static_files,
            use_assets_json=use_assets_json,
            use_assets_json_v2=use_assets_json_v2,
            project_vcs_ref=project_vcs_ref,
            pr_number=pr_number,
            notifications=notifications,
        )
        subtasks.append(static_subtask)

    if run_ticount:
        subtasks += create_ticount_subtasks(
            master_task,
            base_image_resource_id=base_image_resource_id,
            actual_image_resource_id=actual_image_resource_id,
            platforms=('desktop', 'touch'),
            reusable=reusable,
            waitable=waitable,
            pr_number=pr_number,
            notifications=notifications,
            wait_output_parameters=wait_output_parameters,
        )

    if not subtasks:
        raise RuntimeError('Empty subtasks list. Maybe arguments are invalid')

    if pr_number:
        update_pr_description(
            master_task,
            pr_number,
            base_dynamic_resource_id,
            actual_dynamic_resource_id
        )

    return subtasks


def create_shooting_subtasks(
        master_task,
        binary_registry,
        base_templates_package_id,
        actual_templates_package_id,
        platforms,
        reusable,
        wait_output_parameters=None,
        waitable=True,
        notifications=None,
        pr_number=None,
        project_vcs_ref='',
        apphost_mode=False,
):
    """
    :param pr_number:
    :param binary_registry:
    :param project_vcs_ref:
    :param apphost_mode:
    :param waitable:
    :param wait_output_parameters:
    :type master_task: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
    :type base_templates_package_id: int
    :type actual_templates_package_id: int
    :type platforms: list or dict
    :type reusable: bool
    :type notifications: list of sdk2.Notification
    :rtype: list of sandbox.projects.sandbox_ci.pulse.pulse_shooter.PulseShooter
    """
    shooting_tasks = []
    project = master_task.project_name
    should_report_to_stat = master_task.Parameters.pulse_should_report_to_stat
    fail_on_limits_exceed = master_task.Parameters.fail_on_limits_exceed
    send_email_on_limits_exceed = master_task.Parameters.send_email_on_limits_exceed

    request_limit = master_task.config.get_deep_value(
        ['tests', 'pulse_shooter', 'request_limit'], 5000)
    detect_memory_leaks = master_task.config.get_deep_value(
        ['tests', 'pulse_shooter', 'detect_memory_leaks'], False)
    request_limit_with_memory_capture = master_task.config.get_deep_value(
        ['tests', 'pulse_shooter', 'request_limit_with_memory_capture'], 50000)
    memory_capture_requests_per_measurement = master_task.config.get_deep_value(
        ['tests', 'pulse_shooter', 'memory_capture_requests_per_measurement'], 10)

    project_tree_hash = getattr(master_task.Parameters, 'project_tree_hash', '')

    def create_subtask(platform_name, service_name=None):
        apphost_mode_per_platform = master_task.config.get_deep_value(
            ['tests', 'pulse', platform, 'apphost'], None)

        final_apphost_mode = apphost_mode
        if apphost_mode_per_platform is not None:
            final_apphost_mode = bool(apphost_mode_per_platform)

        params = dict(
            family=FAMILY,
            task_type=PulseShooter,
            platform=platform_name,
            base_templates_package=base_templates_package_id,
            actual_templates_package=actual_templates_package_id,
            project=project,
            should_report_to_stat=should_report_to_stat,
            reusable=reusable,
            fail_on_limits_exceed=fail_on_limits_exceed,
            send_email_on_limits_exceed=send_email_on_limits_exceed,
            wait_output_parameters=wait_output_parameters,
            waitable=waitable,
            project_tree_hash=project_tree_hash,
            project_vcs_ref=project_vcs_ref,
            pr_number=pr_number,
            notifications=notifications,
            apphost_mode=final_apphost_mode,

            pulse_shooter_ammo_base=find_latest_ammo(project, service_name, platform_name, apphost_mode),
            rr_base=find_latest_report_renderer_bundle(),

            pulse_shooter_binary=binary_registry.shooter,
            pulse_aggregator_binary=binary_registry.aggregator,
            pulse_report_binary=binary_registry.report,
            html_differ_binary=binary_registry.html_differ,
            diff_viewer_binary=binary_registry.diff_viewer,

            request_limit=request_limit,
            detect_memory_leaks=detect_memory_leaks,
            request_limit_with_memory_capture=request_limit_with_memory_capture,
            memory_capture_requests_per_measurement=memory_capture_requests_per_measurement,
        )

        if service_name:
            params = dict(params, service=service_name)

        return master_task.meta.declare_subtask(**params)

    if type(platforms) is dict:
        for service, service_platforms in platforms.iteritems():
            for platform in service_platforms:
                test_label = '{}.{}.{}'.format(FAMILY, service, platform)
                if master_task.need_to_skip_check(test_label):
                    platform_name = '{}/{}'.format(service, platform)
                    master_task.meta.skip_step(
                        github_context=PulseShooter.format_github_context(platform_name),
                        description=u'Не менялся код для данной проверки',
                        label=test_label,
                        reason='not modified',
                    )
                else:
                    shooting_tasks.append(create_subtask(platform, service))
    else:
        for platform in platforms:
            test_label = '{}.{}'.format(FAMILY, platform)
            if master_task.need_to_skip_check(test_label):
                master_task.meta.skip_step(
                    github_context=PulseShooter.format_github_context(platform),
                    description=u'Не менялся код для данной проверки',
                    label=test_label,
                    reason='not modified',
                )
            else:
                shooting_tasks.append(create_subtask(platform))

    return shooting_tasks


def declare_pulse_shooter_custom_draft(
        master_task,
        binary_registry,
        base_templates_package_id=None,
        actual_templates_package_id=None,
        notifications=None,
        apphost_mode=True,
        project_vcs_ref='',
        pr_number=None,
):
    params = dict(
        family=FAMILY,
        task_type=PulseShooterCustom,
        project=master_task.project_name,
        project_vcs_ref=project_vcs_ref,
        pr_number=pr_number,
        notifications=notifications,
        apphost_mode=apphost_mode,

        rr_base=find_latest_report_renderer_bundle(),

        base_templates_package=base_templates_package_id,
        actual_templates_package=actual_templates_package_id,

        pulse_shooter_binary=binary_registry.shooter,
        pulse_aggregator_binary=binary_registry.aggregator,
        pulse_report_binary=binary_registry.report,
        html_differ_binary=binary_registry.html_differ,
        diff_viewer_binary=binary_registry.diff_viewer,
    )

    return master_task.meta.create_subtask(**params)


def update_pr_description(
        master_task,
        pr_number,
        base_templates_package_id,
        actual_templates_package_id,
):
    config_path = master_task.Parameters.pr_description_config_path
    url = (
        "https://prosperity.si.yandex-team.ru/v1/sandbox/update-pull-request-description"
        "?configPath={}"
        "&sectionId=pulse-shooter-custom"
        "&prNumber={}"
    ).format(config_path, pr_number)

    data = {
        "owner": "SANDBOX_CI_SEARCH_INTERFACES",
        "custom_fields": [
            {
                "name": "arcanum_review_request_id",
                "value": master_task.Parameters.arcanum_review_request_id
            },
            {
                "name": "project_github_owner",
                "value": master_task.Parameters.project_github_owner
            },
            {
                "name": "project_github_repo",
                "value": master_task.project_name
            },
            {
                "name": "project_build_context",
                "value": "pull-request"
            },
            {
                "name": "base_templates_package_id",
                "value": base_templates_package_id
            },
            {
                "name": "actual_templates_package_id",
                "value": actual_templates_package_id
            },
            {
                "name": "pr_number",
                "value": pr_number
            },
        ],
        "output": [
            {
                "name": "ammo_counters",
                "value": master_task.Parameters.ammo_counters
            }
        ]
    }

    res = request.send_request(
        method='post',
        url=url,
        headers={'Content-Type': 'application/json'},
        json=data,
        timeout=120,
    )

    logging.debug('response for {url} with code {status_code} is: {text}'.format(
        url=url,
        status_code=res.status_code,
        text=res.text,
    ))


@skip_subtask(PulseStatic, u'Не менялся код для данной проверки')
def create_static_subtask(
        master_task,
        binary_registry,
        base_static_package_id,
        actual_static_package_id,
        reusable,
        wait_output_parameters=None,
        waitable=True,
        check_static_files=False,
        use_assets_json=False,
        use_assets_json_v2=False,
        notifications=None,
        project_vcs_ref='',
        pr_number=None,
):
    """
    :param binary_registry:
    :param project_vcs_ref:
    :param check_static_files:
    :param use_assets_json:
    :param waitable:
    :param wait_output_parameters:
    :type master_task: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
    :type base_static_package_id: int
    :type actual_static_package_id: int
    :type reusable: bool
    :type notifications: list of sdk2.Notification
    :rtype: sandbox.projects.sandbox_ci.pulse.pulse_static.PulseStatic | None
    """
    if not master_task.config.is_enabled('tests', 'pulse-static'):
        return master_task.meta.skip_step(
            github_context=PulseStatic.github_context,
            description=u'Запуск отключён в Genisys',
            label=PulseStatic.task_label,
        )

    fail_on_limits_exceed = master_task.Parameters.fail_on_limits_exceed
    send_email_on_limits_exceed = master_task.Parameters.send_email_on_limits_exceed
    should_report_to_stat = master_task.Parameters.pulse_should_report_to_stat

    return master_task.meta.declare_subtask(
        task_type=PulseStatic,
        description=master_task.Parameters.description,
        base_static_package=base_static_package_id,
        actual_static_package=actual_static_package_id,
        project=master_task.project_name,
        project_vcs_ref=project_vcs_ref,
        pr_number=pr_number,
        reusable=reusable,
        should_report_to_stat=should_report_to_stat,
        fail_on_limits_exceed=fail_on_limits_exceed,
        send_email_on_limits_exceed=send_email_on_limits_exceed,
        wait_output_parameters=wait_output_parameters,
        waitable=waitable,
        check_static_files=check_static_files,
        use_assets_json=use_assets_json,
        use_assets_json_v2=use_assets_json_v2,
        notifications=notifications,

        pulse_static_binary=binary_registry.static,
        pulse_report_binary=binary_registry.report,
        diff_viewer_binary=binary_registry.diff_viewer,
    )


def create_ticount_subtasks(
    master_task,
    base_image_resource_id,
    actual_image_resource_id,
    platforms,
    reusable,
    waitable=True,
    pr_number=None,
    notifications=None,
    wait_output_parameters=None,
):

    requests_limit = master_task.config.get_deep_value(['tests', 'ticount', 'requests'], 1)
    repeat = master_task.config.get_deep_value(['tests', 'ticount', 'repeat'], 100)

    def create_ticount_subtask(platform):
        return master_task.meta.declare_subtask(
            # TODO https://st.yandex-team.ru/VELOCITY-1389#5f6462dca54aeb07fcd10841
            _container=1719034549,

            task_type=Ticount,
            project=master_task.project_name,
            description=master_task.Parameters.description,

            platform=platform,
            use_overlayfs=master_task.use_overlayfs,
            build_artifacts_resources_base=base_image_resource_id,
            build_artifacts_resources_actual=actual_image_resource_id,
            project_hash_base=master_task.Parameters.project_range_base_hash,
            project_hash_actual=master_task.Parameters.project_hash,
            wait_output_parameters=wait_output_parameters,
            reusable=reusable,
            waitable=waitable,
            pr_number=pr_number,
            notifications=notifications,
            requests_limit=requests_limit,
            repeat=repeat,
        )

    subtasks = []

    for platform in platforms:
        test_label = '{}.{}'.format(FAMILY, platform)
        if master_task.need_to_skip_check(test_label):
            master_task.meta.skip_step(
                github_context=Ticount.format_github_context(platform),
                description=u'Не менялся код для данной проверки',
                label=test_label,
                reason='not modified',
            )
        else:
            subtasks.append(create_ticount_subtask(platform))

    return subtasks
