# coding=utf-8

import json
import logging
import os
import tempfile
import urllib
from datetime import datetime
from urlparse import parse_qsl, urlsplit

from sandbox import common, sdk2
from sandbox.common.errors import TemporaryError
from sandbox.common.types import misc as ctm
from sandbox.common.utils import singleton_property
from sandbox.projects.sandbox_ci import parameters as sandbox_ci_parameters
from sandbox.projects.sandbox_ci.pulse import build_url_path, parse_query_params
from sandbox.projects.sandbox_ci.pulse.config import load_pulse_genisys_config
from sandbox.projects.sandbox_ci.pulse.resources import TicountLogs, TicountPlan
from sandbox.projects.sandbox_ci.pulse.ticount.config import YT_PATH, YT_TOKEN_NAME, YT_TOKEN_OWNER
from sandbox.projects.sandbox_ci.pulse.ticount.templates_runner import TemplatesRunner
from sandbox.projects.sandbox_ci.pulse.ticount.ticount_runner import TicountRunner
from sandbox.projects.sandbox_ci.task import BasePulseTask, OverlayfsMixin, PrepareWorkingCopyMixin
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.sandboxsdk.environments import PipEnvironment


class Ticount(BasePulseTask, PrepareWorkingCopyMixin, OverlayfsMixin):
    TASK_MAX_RETRIES = 2

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL
        environments = [
            PipEnvironment('yandex-yt')
        ]

    class Parameters(BasePulseTask.Parameters):
        with sdk2.parameters.Group('Ticount Parameters') as ticount_parameters:
            with sdk2.parameters.RadioGroup('Requests origin', per_line=2) as requests_origin:
                requests_origin.values['plan'] = requests_origin.Value('Plan', default=True)
                requests_origin.values['manual'] = requests_origin.Value('Manual')

            with requests_origin.value['plan']:
                plan_resource = sdk2.parameters.Resource(
                    'Plan resource',
                    resource_type=TicountPlan,
                )

                requests_limit = sdk2.parameters.Integer('Requests limit', default=1)

            with requests_origin.value['manual']:
                queries = sdk2.parameters.String(
                    'Queries',
                    description='One query per line',
                    required=True,
                    multiline=True,
                )

                base_query_params = sdk2.parameters.List(
                    'Base query params',
                    default=[
                        'no-tests=1',
                        'promo=nomooa',
                    ]
                )

                use_actual_query_params = sdk2.parameters.Bool(
                    'Use actual query params',
                    default=False,
                )

                with use_actual_query_params.value[True]:
                    actual_query_params = sdk2.parameters.List(
                        'Actual query params',
                        default=[
                            'no-tests=1',
                            'promo=nomooa',
                        ]
                    )

            repeat = sdk2.parameters.Integer('Repeat', default=100)
            browser_workers = sdk2.parameters.Integer('Browser workers', default=10)

            with sdk2.parameters.RadioGroup('Save traces', per_line=2) as save_traces:
                save_traces.values['on_limits_exceed'] = save_traces.Value('On limits exceed', default=True)
                save_traces.values['always'] = save_traces.Value('Always')

            with sdk2.parameters.String('Platform') as platform:
                platform.values['desktop'] = platform.Value('desktop', default=True)
                platform.values['touch'] = platform.Value('touch')

            ticount_package = sdk2.parameters.String(
                'Ticount package',
                default='@yandex-int/ticount@0.9',
                required=True,
            )

            ticount_report_package = sdk2.parameters.String(
                'Ticount report package',
                default='@yandex-int/ticount-report@1.0.4',
                required=True,
            )

            ammo_counters = sdk2.parameters.JSON(
                'Счётчики фичи',
                description=(
                    'Счётчики затронутой фичи для запуска PULSE_SHOOTER_CUSTOM\n'
                    '(в формате {"touch-phone": "/$page/$main/$result/#type:wizard/#wizard_name:images", "desktop": "/$page/$parallel/$result/#type:wizard/#wizard_name:images"})'
                )
            )

        with sdk2.parameters.Group('Base Resources') as base_resources_parameters:
            build_artifacts_resources_base = sandbox_ci_parameters.build_artifacts_resources()
            project_hash_base = sandbox_ci_parameters.project_hash()

        with sdk2.parameters.Group('Actual Resources') as actual_resources_parameters:
            build_artifacts_resources_actual = sandbox_ci_parameters.build_artifacts_resources()
            project_hash_actual = sandbox_ci_parameters.project_hash()

    class Context(BasePulseTask.Context):
        report = ''
        excesses = []
        metrics_base = []
        metrics_actual = []
        compare_results = []
        report_resource_path = ''
        table_resource_path = ''

    lifecycle_steps = {
        'npm_install': 'npm ci --registry=https://npm.yandex-team.ru',
    }

    @classmethod
    def format_github_context(cls, platform):
        return u'Ticount: {}'.format(platform)

    @property
    def github_context(self):
        return self.format_github_context(self.Parameters.platform)

    @singleton_property
    def cache_parameters(self):
        parameters = super(Ticount, self).cache_parameters

        parameters.update(
            project_hash_base=self.Parameters.project_hash_base,
            project_hash_actual=self.Parameters.project_hash_actual,
            platform=self.Parameters.platform,
        )

        return parameters

    @property
    def _report(self):
        return self.Context.report or 'Report is not ready yet'

    @property
    def mobile(self):
        return self.Parameters.platform == 'touch'

    @singleton_property
    def metrics_logs_dir(self):
        return os.path.join(os.getcwd(), 'metrics')

    @singleton_property
    def traces_dir(self):
        return os.path.join(os.getcwd(), 'traces')

    @singleton_property
    def report_html_dir(self):
        return os.path.join(os.getcwd(), 'report')

    @singleton_property
    def result_tables_dir(self):
        tables_dir = os.path.join(os.getcwd(), 'tables')
        if not os.path.exists(tables_dir):
            os.mkdir(tables_dir)
        return tables_dir

    @singleton_property
    def queries(self):
        return self.Parameters.queries.encode('utf8').split('\n')

    @singleton_property
    def requests(self):
        return (
            self.queries
            if self.Parameters.requests_origin == 'manual'
            else [self.get_name_from_url(url) for url in self.plan]
        )

    @singleton_property
    def plan(self):
        plan_resource = self.Parameters.plan_resource

        if not plan_resource:
            plan_resource = self.artifacts.get_last_artifact_resource(
                TicountPlan,
                project=self.Parameters.project,
                platform=self.Parameters.platform,
            )

        if not plan_resource:
            raise Exception(
                'Released Ticount plan for {}:{} not found'.format(self.Parameters.project, self.Parameters.platform)
            )

        plan_file = self.artifacts.sync_build_artifact(plan_resource)

        with open(str(plan_file)) as fd:
            return [line.strip() for line in fd.readlines()[:self.Parameters.requests_limit]]

    @singleton_property
    def urls_base(self):
        if self.Parameters.requests_origin == 'plan':
            return self.plan

        return self.format_urls(query_params=self.Parameters.base_query_params)

    @singleton_property
    def urls_actual(self):
        if self.Parameters.requests_origin == 'plan':
            return self.plan

        if self.Parameters.use_actual_query_params:
            return self.format_urls(query_params=self.Parameters.actual_query_params)

        return self.urls_base

    @singleton_property
    def limits(self):
        return load_pulse_genisys_config(
            self.project_name,
            'ticount_limits'
        ).get(self.Parameters.platform)

    def format_urls(self, query_params):
        query_params_dict = parse_query_params(query_params)
        return [
            build_url_path(
                self.Parameters.project,
                self.Parameters.platform,
                dict(text=text, **query_params_dict),
            )
            for text in self.queries
        ]

    def get_name_from_url(self, url):
        get_params = dict(parse_qsl(urlsplit(url).query))
        text = urllib.unquote(get_params.get('text', ''))
        return text.decode('utf8')

    @singleton_property
    def ticount(self):
        return TicountRunner(package=self.Parameters.ticount_package, node_js_version=self.Parameters.node_js_version)

    def execute(self):
        if not self.Parameters.build_artifacts_resources_base:
            self.set_info('Base resource not ready, skipping')
            return

        if not self.ticount.test():
            if self.agentr.iteration < self.TASK_MAX_RETRIES:
                raise TemporaryError('CPU instructions count not supported on this agent, task will be restarted')
            else:
                self.Context.report = 'CPU instructions count is not supported, manual restart is needed.'
                return

        metrics_logs_dir_base = os.path.join(self.metrics_logs_dir, 'base')
        metrics_logs_dir_actual = os.path.join(self.metrics_logs_dir, 'actual')

        with self.prepare_working_copy_context(dict(commit=self.Parameters.project_hash_base)):
            with self.mount_overlayfs(self.Parameters.build_artifacts_resources_base):
                with self.profiler.actions.shooting('Shooting base'):
                    self.collect_metrics(
                        urls=self.urls_base,
                        metrics_logs_dir=metrics_logs_dir_base,
                        traces_dir=os.path.join(self.traces_dir, 'base'),
                    )

        with self.prepare_working_copy_context(dict(commit=self.Parameters.project_hash_actual)):
            with self.mount_overlayfs(self.Parameters.build_artifacts_resources_actual):
                with self.profiler.actions.shooting('Shooting actual'):
                    self.collect_metrics(
                        urls=self.urls_actual,
                        metrics_logs_dir=metrics_logs_dir_actual,
                        traces_dir=os.path.join(self.traces_dir, 'actual'),
                    )

        with self.profiler.actions.save_logs('Save logs'):
            ticount_logs_resource = TicountLogs(
                task=self,
                description='Ticount metrics logs',
                type='metrics',
                path=self.metrics_logs_dir,
            )

            sdk2.ResourceData(ticount_logs_resource).ready()

        with self.profiler.actions.aggregation('Aggregation'):
            self.compare(metrics_logs_dir_base, metrics_logs_dir_actual)

        with self.profiler.actions.save_traces('Save traces'):
            self.save_traces()

        with self.profiler.actions.report_html('Report HTML'):
            self.report_html(metrics_logs_dir_base, metrics_logs_dir_actual)

        with self.profiler.actions.report_header('Report Header'):
            self.header_report()

        with self.profiler.actions.uploading_to_yt('Uploading to YT'):
            self.upload_to_yt_safe()

        with self.profiler.actions.check_limits('Check limits'):
            self.check_exceeded_limits()

    def mount_overlayfs(self, resources):
        return self._overlayfs(lower_dirs=[self.arc_project_dir], resources=resources, target_dir=self.project_dir)

    def get_limits_excesses(self):
        return self.Context.excesses

    def collect_metrics(self, urls, metrics_logs_dir, traces_dir):
        if not os.path.exists(metrics_logs_dir):
            os.makedirs(metrics_logs_dir)

        if not os.path.exists(traces_dir):
            os.makedirs(traces_dir)

        templates = TemplatesRunner(
            work_dir=self.project_dir,
            synchrophazotron_path=self.synchrophazotron,
            node_js_version=self.Parameters.node_js_version,
        )

        templates.start()

        for idx, url in enumerate(urls):
            self.ticount.write(
                url='http://{}:{}{}'.format(templates.host, templates.port, url),
                repeat=self.Parameters.repeat,
                workers=self.Parameters.browser_workers,
                metrics_log=os.path.join(metrics_logs_dir, 'metrics_{}.json'.format(idx)),
                traces_dir=os.path.join(traces_dir, 'traces_{}'.format(idx)),
                mobile=self.mobile,
            )

        templates.stop()

    def report_html(self, metrics_logs_dir_base, metrics_logs_dir_actual):
        for idx, url in enumerate(self.requests):
            metrics_base = os.path.join(metrics_logs_dir_base, 'metrics_{}.json'.format(idx))
            metrics_actual = os.path.join(metrics_logs_dir_actual, 'metrics_{}.json'.format(idx))
            output = os.path.join(self.report_html_dir, 'report_{}.html'.format(idx))

            command = [
                '--url {}'.format(url),
                '--output {}'.format(output),
                '--data {} {}'.format(metrics_base, metrics_actual),
            ]

            with Node(self.Parameters.node_js_version):
                run_process(
                    ['npx', self.Parameters.ticount_report_package] + command,
                    shell=True,
                    log_prefix='ticount_report',
                    environment=dict(
                        os.environ,
                        npm_config_registry='https://npm.yandex-team.ru',
                    ),
                )

        ticount_report_resource = TicountLogs(
            task=self,
            description='Ticount report html',
            type='report',
            path=self.report_html_dir,
        )

        sdk2.ResourceData(ticount_report_resource).ready()
        self.Context.report_resource_path = ticount_report_resource.http_proxy

    def compare(self, metrics_logs_dir_base, metrics_logs_dir_actual):
        for idx, url in enumerate(self.requests):
            excessess_log = os.path.join(os.getcwd(), 'excesses_{}.log'.format(idx))
            metrics_base = os.path.join(metrics_logs_dir_base, 'metrics_{}.json'.format(idx))
            metrics_actual = os.path.join(metrics_logs_dir_actual, 'metrics_{}.json'.format(idx))
            report_file = os.path.join(os.getcwd(), 'report_{}.txt'.format(idx))
            result_table_file = os.path.join(self.result_tables_dir, 'table_{}.txt'.format(idx))

            compare_result_table = self.ticount.compare(
                metrics_base=metrics_base,
                metrics_actual=metrics_actual,
                excesses_log=excessess_log,
                limits=self.limits,
                report=report_file,
            )

            self.Context.metrics_base.append(metrics_base)
            self.Context.metrics_actual.append(metrics_actual)

            with open(report_file) as fd:
                self.Context.compare_results.append(fd.read())

            with open(excessess_log) as fd:
                self.Context.excesses += fd.read().splitlines()

            with open(result_table_file, 'w') as fd:
                fd.write(compare_result_table)

        ticount_tables_resource = TicountLogs(
            task=self,
            description='Ticount result tables',
            type='table',
            path=self.result_tables_dir,
        )

        sdk2.ResourceData(ticount_tables_resource).ready()
        self.Context.table_resource_path = ticount_tables_resource.http_proxy

    def header_report(self):
        for idx, (request, compare_result) in enumerate(zip(self.requests, self.Context.compare_results)):
            chart_href = '{}/report_{}.html'.format(self.Context.report_resource_path, idx)
            table_href = '{}/table_{}.txt'.format(self.Context.table_resource_path, idx)

            self.Context.report += '''
                <div>
                    <b>{request}</b>
                    <a href={chart_href}>Графики</a>
                    <a href={table_href}>Полные результаты</a>
                </div>
                <pre>{compare_result}</pre>
            '''.format(
                request=request,
                chart_href=chart_href,
                table_href=table_href,
                compare_result=compare_result,
            )

    def save_traces(self):
        if self.Parameters.save_traces == 'on_limits_exceed':
            exceeded_limits = self.get_limits_excesses()
            if not len(exceeded_limits):
                logging.info('Limits are not exceeded - skip saving traces')
                return

        ticount_traces_resource = TicountLogs(
            task=self,
            description='Ticount traces',
            type='traces',
            path=self.traces_dir,
        )

        sdk2.ResourceData(ticount_traces_resource).ready()

    def upload_to_yt_safe(self):
        try:
            self._upload_to_yt()
        except Exception as e:
            logging.exception('Error during uploading data to YT: %s', e)

    def _upload_to_yt(self):
        from yt.wrapper import JsonFormat, YtClient

        yt_client = YtClient(
            proxy='hahn',
            token=sdk2.Vault.data(YT_TOKEN_OWNER, YT_TOKEN_NAME)
        )

        base_data = self._prepare_yt_data(self.urls_base, self.Context.metrics_base, {
            'type': 'base',
        })

        actual_data = self._prepare_yt_data(self.urls_actual, self.Context.metrics_actual, {
            'type': 'actual',
        })

        table_name = datetime.now().strftime('%Y-%m-%d')
        table_path = os.path.join(YT_PATH, table_name)

        if not yt_client.exists(table_path):
            schema = [{'name': name, 'type': 'any'} for name in base_data[0].keys()]
            yt_client.create('table', table_path, attributes={'schema': schema}, recursive=True)

        yt_client.write_table(
            yt_client.TablePath(table_path, append=True),
            base_data+actual_data,
            format=JsonFormat()
        )

    def _prepare_yt_data(self, urls, log_files, additional_fields={}):
        result = []
        for idx, file in enumerate(log_files):
            with open(file) as fd:
                for line in fd:
                    record = json.loads(line)
                    table_line = dict(
                        task_id=self.id,
                        host=common.config.Registry().this.id,
                        created=self.created.strftime('%Y-%m-%d %H:%M:%S'),
                        url=urls[idx],
                        tags=self.Parameters.tags,
                        client_tags=common.config.Registry().client.tags,
                    )

                    table_line.update(record)
                    table_line.update(additional_fields)
                    result.append(table_line)

        return result
