import logging
import os
import socket
import time

import requests
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.process import run_process

START_POLLING_MAX_ATTEMPTS = 10
START_POLLING_DELAY = 10


class TemplatesRunner(object):
    host = 'localhost'

    def __init__(self, work_dir, node_js_version, synchrophazotron_path):
        self._work_dir = work_dir
        self._node_js_version = node_js_version
        self._synchrophazotron_path = synchrophazotron_path

    @property
    def port(self):
        return os.getenv('ARCHON_KOTIK_PORT', 17999)

    @property
    def admin_port(self):
        return os.getenv('ARCHON_RENDERER_DEVOPS_PORT', 18005)

    @property
    def command(self):
        return [
            'npm run dev:testing --',
            '--kotik-incognito=true',
            '--renderer-devops-port={}'.format(self.admin_port),
        ]

    def start(self):
        with Node(self._node_js_version):
            self._process = run_process(
                self.command,
                wait=False,
                log_prefix='templates',
                work_dir=str(self._work_dir),
                environment=dict(
                    os.environ,
                    SYNCHROPHAZOTRON_PATH=self._synchrophazotron_path,
                    DISABLE_TEMPLAR='true',
                )
            )

        self.wait_for_start()

    def wait_for_start(self):
        args = socket.getaddrinfo(self.host, self.port, socket.AF_INET6, socket.SOCK_STREAM)
        attempts = 0

        logging.info('Checking http server')

        started = False
        while started is False:
            logging.debug(
                'Attempt to check templates http server, attempt %d/%d',
                attempts,
                START_POLLING_MAX_ATTEMPTS
            )
            for family, socktype, proto, _, sockaddr in args:
                s = socket.socket(family, socktype, proto)
                try:
                    s.connect(sockaddr)
                except socket.error:
                    logging.debug('Templates is down')
                else:
                    started = True
                    s.close()

            attempts += 1
            if attempts >= START_POLLING_MAX_ATTEMPTS:
                break

            time.sleep(START_POLLING_DELAY)

        if not started:
            raise Exception('Templates is not started')

    def stop(self):
        requests.get('http://localhost:{}/admin?action=shutdown'.format(self.admin_port))
