# -*- coding: utf-8 -*-
import inspect
import logging
import numbers
import os
import tarfile
import glob

from sandbox import sdk2
from sandbox.common import rest
from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


def find_resource(resource_type, attrs):
    if not attrs:
        attrs = {}

    resource = apihelpers.get_last_resource_with_attrs(resource_type, attrs, all_attrs=True)

    if not resource:
        raise SandboxTaskFailureError('Unable to find {} with attrs {}'.format(resource_type, attrs))

    return resource


def find_resource_id(resource_type, attrs=None):
    if attrs is None:
        attrs = {}

    resource = apihelpers.get_last_resource_with_attrs(resource_type, attrs, all_attrs=True)

    if not resource:
        raise SandboxTaskFailureError('Unable to find {} with attrs {}'.format(resource_type, attrs))

    return str(resource.id)


def create_sdk1_subtask(task_type, description, input_parameters, owner, priority=None):
    """
    Костылик для запуска SDK1 задач из SDK2 задачи
    """
    api = rest.Client()

    task_id = api.task({'type': task_type, 'children': True}).get('id')

    params_to_update = {
        'description': description,
        'owner': owner,
        'custom_fields': map(lambda (k, v): dict(name=k, value=v), input_parameters.iteritems())
    }

    if priority:
        if isinstance(priority, (list, tuple)) and len(priority) == 2:
            cls, scls = priority

            params_to_update.update({
                'priority': {
                    'class': cls,
                    'subclass': scls,
                }
            })
        else:
            params_to_update.update({
                'priority': priority.as_dict()
            })

    api.task[task_id].update(params_to_update)
    api.batch.tasks.start.update(task_id)

    return task_id


def find_in_dict(element, d):
    """
    :param element: путь разделенный точками, например 'some.data.info'
    :type element: str
    :type d: dict
    """
    keys = element.split('.')
    rv = d
    for key in keys:
        rv = rv[key]
    return rv


def unpack_templates(templates_resource_data, directory):
    resource_path = str(templates_resource_data.path)
    logging.info('Unpacking templates: %s', resource_path)

    with tarfile.open(resource_path, 'r:gz') as tar:
        tar.chown = lambda *args: None
        for entry in tar:
            if not entry.name.startswith('/') and '..' not in entry.name.split('/'):
                dst = os.path.join(directory, *entry.name.split('/'))

                if entry is not None and (entry.isfile() or entry.isdir()):
                    if dst:
                        if dst.endswith(os.sep):
                            dst = dst[:-1]
                        try:
                            tar._extract_member(entry, dst)
                        except tarfile.ExtractError:
                            pass

    micro_package_routes_path = get_micropackage_routes(directory)
    if micro_package_routes_path:
        return micro_package_routes_path

    old_fashion_package_routes_path = os.path.join(directory, 'report-templates', 'YxWeb', 'routes.json')
    if os.path.exists(old_fashion_package_routes_path):
        return old_fashion_package_routes_path

    raise RuntimeError('Unknown package format')


def get_micropackage_routes(directory):
    """
    В заданной директории ищет routes.js или routes.json файлы и возвращает путь найденного файла.

    :param directory: директория
    :type directory: str
    :rtype: str or None
    """
    routes_path_glob = os.path.join(directory, 'routes.js*')
    routes_files = glob.glob(routes_path_glob)

    logging.debug('"{glob}" matches: {files}'.format(
        glob=routes_path_glob,
        files=routes_files,
    ))

    return routes_files[0] if routes_files else None


def get_subtask_params(task_parameters):
    """
    Копировать параметры

    :rtype: dict
    """

    param_items = inspect.getmembers(task_parameters)
    params = {}

    for name, val in param_items:
        # Не берём приватные свойства
        if not name.startswith('_'):
            new_val = _get_param_value(val)
            if new_val is not None:
                params[name] = new_val

    return params


def _get_param_value(val):
    if isinstance(val, sdk2.Resource):
        return val.id

    if isinstance(val, basestring) or isinstance(val, bool) or isinstance(val, numbers.Number):
        return val

    if isinstance(val, list):
        new_val = []
        for v in val:
            new_v = _get_param_value(v)
            if new_v is not None:
                new_val.append(new_v)
        return new_val

    if isinstance(val, dict):
        # В словарях допускается None, не проверяем
        return {k: _get_param_value(v) for k, v in val.iteritems()}
