import os
import traceback
import logging

from datetime import datetime
from string import Template

from sandbox.common.fs import read_file
from sandbox.common import system


class YQLQueryBuilder(object):
    def __init__(self, tasks_path):
        self._tasks_path = tasks_path

    def build(self, project, platform, date=None, **tpl_vars):
        tpl_local_path = '{project}/{platform}.sql'.format(
            project=project,
            platform=platform
        )
        tpl_path = os.path.join(self._tasks_path, tpl_local_path)

        common_local_path = '{project}/_common.sql'.format(project=project)
        common_path = os.path.join(self._tasks_path, common_local_path)

        global_common_path = os.path.join(self._tasks_path, '_common.sql')

        query = self._read_source(tpl_path)

        if self._file_exists(common_path):
            common_yql = self._read_source(common_path)
            query = common_yql + '\n' + query

        if self._file_exists(global_common_path):
            global_common_yql = self._read_source(global_common_path)
            query = global_common_yql + '\n' + query

        return Template(query).safe_substitute(
            date=date or datetime.now().strftime('%Y-%m-%d'),
            **tpl_vars
        )

    def build_from_file(self, filename, **tpl_vars):
        tpl_path = os.path.join(self._tasks_path, filename)
        query = self._read_source(tpl_path)

        return Template(query).safe_substitute(**tpl_vars)

    def _file_exists(self, path):
        if system.inside_the_binary():
            from library.python import resource
            return bool(resource.find(path))
        return os.path.exists(path)

    def _read_source(self, path):
        try:
            return read_file(path)
        except (OSError, IOError) as e:
            logging.error(e)
            raise Exception('Unable to find YQL file by path - {}\n{}'.format(
                path,
                traceback.format_exc()
            ))
