# -*- coding: utf-8 -*-

import json
import logging

from requests import codes

from sandbox.projects.sandbox_ci.qloud.enums import QloudEnvironmentStatus
from sandbox.projects.sandbox_ci.qloud.errors import QloudReadonly, QloudDeployFailed
from sandbox.projects.sandbox_ci.utils.request import send_request

MINUTE = 60
QLOUD_URL = 'https://qloud.yandex-team.ru'


class QloudApi(object):
    def __init__(self, token, url=QLOUD_URL):
        self._url = url
        self._token = token
        self._headers = {
            'Authorization': 'OAuth {token}'.format(token=self._token),
            'Content-Type': 'application/json',
        }

    def get_full_url(self, path):
        """
        Создает полный URL-адрес, используя базовый адрес и путь

        :param path: запрашиваемый путь
        :type path: str
        :return: путь объединенный с базовым адресом
        :rtype: str
        """
        return '{}/{}'.format(self._url, path)

    def get_application(self, application_id):
        """
        Получает приложение

        :param application_id: id qloud приложения
        :type application_id: str
        :return: response
        :rtype: dict
        """
        url = self.get_full_url('api/v1/application/{application_id}'.format(
            application_id=application_id,
        ))
        res = send_request('get', url, headers=self._headers, verify=False)

        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def upload_environment(self, environment):
        """
        Загрузка и запуск выкладки окружения.

        :param environment: настройки окружения
        :type environment: object
        :raises: QloudReadonly
        :rtype: None
        """
        self.check_not_readonly()

        url = self.get_full_url('api/v1/environment/upload')
        data = json.dumps(environment)
        res = send_request('post', url, headers=self._headers, data=data, verify=False)

        logging.info('Uploading qloud environment: {}'.format(data))

        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}

    def request_environment(self, environment_id):
        """
        Запрашивает рабочую (развернутую) версию окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :return: response
        :rtype: requests.Response
        """
        url = self.get_full_url('api/v1/environment/stable/{environment_id}'.format(
            environment_id=environment_id,
        ))
        return send_request('get', url, headers=self._headers, verify=False)

    def dump_environment(self, environment_id):
        """
        Запрашивает текущие настройки окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :return: response
        :rtype: dict
        """
        url = self.get_full_url('api/v1/environment/dump/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = send_request('get', url, headers=self._headers, verify=False)

        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def delete_environment(self, environment_id):
        """
        Удаление окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :raises: QloudReadonly
        :rtype: None
        """
        self.check_not_readonly()

        url = self.get_full_url('api/v1/environment/stable/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = send_request('delete', url, headers=self._headers, verify=False)

        assert res.status_code == codes.no_content, {'status_code': res.status_code, 'text': res.text}

    def create_domain(self, environment_id, settings):
        """
        Создает новый домен в окружении

        :param environment_id: id qloud окружения
        :type environment_id: str
        :param settings: настройки домена
        :type settings: dict
        :raises: QloudReadonly
        :rtype: None
        """
        self.check_not_readonly()

        url = self.get_full_url('api/v1/environment-domain/{environment_id}?silent=true'.format(
            environment_id=environment_id,
        ))
        data = json.dumps(settings)
        res = send_request('put', url, headers=self._headers, data=data, verify=False)

        logging.info('Creating qloud environment {} domain: {}'.format(environment_id, settings))

        assert res.status_code == codes.no_content, {'status_code': res.status_code, 'text': res.text}

    def get_environment_status(self, environment_id):
        """
        Запрашивает статус окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :rtype: requests.Response
        """
        url = self.get_full_url('api/v1/environment/status/{environment_id}'.format(
            environment_id=environment_id,
        ))
        return send_request('get', url, headers=self._headers, verify=False)

    def get_environment_domains(self, environment_id):
        """
        Запрашивает домены окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :rtype: dict
        """
        url = self.get_full_url('api/v1/domains/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = send_request('get', url, headers=self._headers, verify=False)

        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def is_environment_exist(self, environment_id):
        """
        Проверка существования окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :rtype: bool
        """
        res = self.get_environment_status(environment_id)
        return res.status_code == codes.ok

    def get_environment(self, environment_id):
        """
        Получает информацию о рабочей копии окружения

        :param environment_id: id qloud окружения
        :type environment_id: str
        :rtype: dict
        """
        res = self.request_environment(environment_id)
        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def is_environment_deployed(self, environment_id):
        """
        Проверяет, развернуто ли окружение

        :param environment_id: id qloud окружения
        :type environment_id: str
        :raises: QloudDeployFailed
        :rtype: bool
        """
        res = self.get_environment_status(environment_id)

        logging.info('Request environment {environment_id} status (status={status_code}, body={body})'.format(
            environment_id=environment_id,
            status_code=res.status_code,
            body=res.text,
        ))

        if res.status_code >= codes.forbidden:
            return False

        data = json.loads(res.text)
        status = data.get('status', None)

        if status in (QloudEnvironmentStatus.TEMPLATE, QloudEnvironmentStatus.FAILED):
            raise QloudDeployFailed(data)

        return status == QloudEnvironmentStatus.DEPLOYED

    def is_in_readonly(self):
        """
        Проверяет, находится ли qloud в режиме чтения

        :rtype: bool
        """
        url = self.get_full_url('api/management/readonly')

        headers = dict(**self._headers)
        headers.update({
            'Content-Type': 'text/plain',
        })

        res = send_request('get', url, headers=headers, verify=False)
        assert res.status_code == codes.ok, {'status_code': res.status_code, 'text': res.text}

        logging.info('Check qloud readonly status (status={text})'.format(text=res.text))

        return res.text == 'true'

    def check_not_readonly(self):
        if self.is_in_readonly():
            raise QloudReadonly()
