# -*- coding: utf-8 -*-
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm, task as ctt
from sandbox.common.errors import TaskFailure

from sandbox.sandboxsdk.process import SandboxSubprocessError

from sandbox.projects.sandbox_ci.utils import prioritizer
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.utils.context import Debug, GitRetryWrapper, GitWithoutLfsProcessing
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

DEBUG_NAMESPACE = 'si:ci:*,serp:*'


class CommitIdentifierParameters(sdk2.Parameters):
    commit_identifiers_info = sdk2.parameters.Info('Номер задачи должен быть с ключом очереди')
    commit_identifiers = sdk2.parameters.List(
        'Номер задачи или пулл-реквеста',
        description='Например, задача — SERP-67923, пулл-реквест — 16716',
    )


class ReleaseIssueActions(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """Добавить задачу или пулл-реквест в релизную ветку"""

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL

    class Parameters(parameters.CommonParameters):
        dump_disk_usage = False
        kill_timeout = 3 * 3600

        scripts_sources = parameters.ScriptsSourcesParameters
        _container = parameters.environment_container()

        with sdk2.parameters.Group('Проект') as project_block:
            project_owner = sdk2.parameters.String(
                'Владелец',
                default='serp',
                choices=[
                    ('serp', 'serp'),
                    ('news', 'news'),
                    ('mm-interfaces', 'mm-interfaces'),
                    ('search-interfaces', 'search-interfaces'),
                    ('testpalm', 'testpalm'),
                    ('tools', 'tools'),
                ],
                description='Владелец репозитория проекта на GitHub. '
                            'Если здесь чего-то не хватает, обращайтесь в рассылку infraduty@.',
                required=True,
            )
            project_repo = sdk2.parameters.String(
                'Проект',
                default='web4',
                choices=[
                    ('advanced', 'advanced'),
                    ('blogs', 'blogs'),
                    ('chat', 'chat'),
                    ('mischandlers', 'mischandlers'),
                    ('redir-warning', 'redir-warning'),
                    ('sitesearch2', 'sitesearch2'),
                    ('turbo', 'turbo'),
                    ('web4', 'web4'),
                    ('nerpa', 'nerpa'),
                    ('fiji', 'fiji'),
                    ('frontend', 'frontend'),
                    ('testpalm', 'testpalm'),
                    ('staff-www', 'staff-www'),
                    ('goals-frontend', 'goals-frontend'),
                ],
                description='Название репозитория на GitHub.',
                required=True,
            )

        node_js = parameters.NodeJsParameters

    @property
    def script_path(self):
        raise NotImplementedError

    def get_commit_identifiers(self):
        raise NotImplementedError

    def get_branch_id(self):
        raise NotImplementedError

    def get_script_params(self):
        raise NotImplementedError

    def working_path(self, *args):
        return self.path(*args)

    def on_create(self):
        if not self.Parameters.description:
            self.Parameters.description = self.__doc__

    def on_save(self):
        super(ReleaseIssueActions, self).on_save()

        weight = 1
        exec_limit = 1
        semaphore_name = self.get_branch_id()

        self.Requirements.semaphores = ctt.Semaphores(
            [(semaphore_name, weight, exec_limit)],
            ctt.Status.Group.BREAK | ctt.Status.Group.FINISH,
            )

        self.Parameters.priority = prioritizer.get_priority(self)

    def on_prepare(self):
        with self.profiler.actions.clone_scripts('Cloning scripts'):
            self.scripts.sync_resource()

        env.export_from_vault(self)

    def on_execute(self):
        self.attach_tags()
        self.update_description()
        self.execute()

    def attach_tags(self):
        self.Parameters.tags = [
            self.Parameters.project_owner,
            self.Parameters.project_repo,
        ]

    def update_description(self):
        self.Parameters.description = '{title}: {items}'.format(
            title=self.get_action_title(),
            items=', '.join(self.get_commit_identifier_html_links()),
        )

    def get_action_title(self):
        return 'Добавить в {}:'.format(self.get_branch_id())

    def get_commit_identifier_html_links(self):
        links = []
        owner = self.Parameters.project_owner
        repo = self.Parameters.project_repo

        for commit_identifier in self.get_commit_identifiers():
            # если только цифры — значит, это ПР, иначе — задача
            if commit_identifier.isdigit():
                html_link = self.get_pr_html_link(owner, repo, commit_identifier)
            else:
                html_link = self.get_issue_html_link(commit_identifier.upper())

            links.append(html_link)

        return links

    @staticmethod
    def get_pr_html_link(owner, repo, pr_number):
        pr_url = 'https://github.yandex-team.ru/{owner}/{repo}/pull/{pr_number}'.format(
            owner=owner,
            repo=repo,
            pr_number=pr_number,
        )

        return '<a href="{pr_url}">{owner}/{repo}#{pr_number}</a>'.format(
            owner=owner,
            repo=repo,
            pr_number=pr_number,
            pr_url=pr_url,
        )

    @staticmethod
    def get_issue_html_link(issue_key):
        return '<a href="https://st.yandex-team.ru/{issue_key}">{issue_key}</a>'.format(issue_key=issue_key)

    def execute(self):
        params = self.get_script_params()
        git_url = self.get_repo_git_url(self.Parameters.project_owner, self.Parameters.project_repo)

        try:
            with Debug(DEBUG_NAMESPACE), GitRetryWrapper(), self.vault.ssh_key(), GitWithoutLfsProcessing(git_url):

                logging.debug('Running script with options: {}'.format(params))

                self.scripts.run_js(self.script_path, params)
        except SandboxSubprocessError as e:
            self.set_info(e.get_task_info(), do_escape=False)
            raise TaskFailure('Could not add issue or pr, see logs for details')

    @staticmethod
    def get_repo_git_url(owner, repo):
        return 'git@github.yandex-team.ru:{owner}/{repo}.git'.format(
            owner=owner,
            repo=repo,
        )
