# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.errors import TaskFailure

from sandbox.sandboxsdk.process import SandboxSubprocessError

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.utils.context import Debug, GitRetryWrapper, GitWithoutLfsProcessing
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement


class SandboxCiCreateRelease(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """Отведение новой релизной ветки"""

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL

    class Parameters(parameters.CommonParameters):
        dump_disk_usage = False
        kill_timeout = 3 * 3600

        scripts_sources = parameters.ScriptsSourcesParameters
        _container = parameters.environment_container()

        with sdk2.parameters.Group('Проект') as project_block:
            project_owner = sdk2.parameters.String(
                'Владелец',
                default='serp',
                choices=[
                    ('serp', 'serp'),
                    ('search-interfaces', 'search-interfaces'),
                    ('news', 'news'),
                    ('mm-interfaces', 'mm-interfaces'),
                    ('direct', 'direct'),
                ],
                description='Владелец репозитория проекта на GitHub. '
                            'Если здесь чего-то не хватает, обращайтесь в рассылку infraduty@.',
                required=True,
            )
            project_repo = sdk2.parameters.String(
                'Проект',
                default='web4',
                choices=[
                    ('advanced', 'advanced'),
                    ('blogs', 'blogs'),
                    ('chat', 'chat'),
                    ('mischandlers', 'mischandlers'),
                    ('redir-warning', 'redir-warning'),
                    ('sitesearch2', 'sitesearch2'),
                    ('turbo', 'turbo'),
                    ('web4', 'web4'),
                    ('nerpa', 'nerpa'),
                    ('fiji', 'fiji'),
                    ('ydo-frontend', 'ydo-frontend'),
                    ('frontend', 'frontend'),
                ],
                description='Название репозитория на GitHub.',
                required=True,
                sub_fields={
                    'web4': ['platform', 'use_arc'],
                    'frontend': ['platform', 'use_arc'],
                    'fiji': ['platform', 'use_arc'],
                    'nerpa': ['platform'],
                },
            )

            with sdk2.parameters.String(
                    'Подпроект',
                    description='Создать релиз для указанной платформы, экспериментов или фриза. '
                                'Актуально для проектов web4 – эксперименты, fiji и nerpa – платформы.',
            ) as platform:
                platform.ui = platform.UI('select')
                platform.choices = [
                    ('all', '', True),
                    ('desktop', 'desktop'),
                    ('touch', 'touch'),
                    ('freeze', 'freeze'),
                    ('images', 'images'),
                    ('video', 'video'),
                    ('images-smart', 'images-smart'),
                    ('images-desktop-granny', 'images-desktop-granny'),
                    ('video-desktop-granny', 'video-desktop-granny'),
                    ('video-phone-granny', 'video-phone-granny'),
                ]

            experiment = sdk2.parameters.String(
                'Эксперимент',
                default='',
                description='Создать эксперимент для проекта. '
                            'В качестве версии эксперимента используется предыдущая версия этого эксперимента. '
                            'Актуально только для web4.',
                choices=[
                    (u'без эксперимента', ''),
                    ('exp', 'exp'),
                    ('exp2', 'exp2'),
                    ('exp3', 'exp3'),
                    ('exp4', 'exp4'),
                ]
            )

        with sdk2.parameters.Group('Версия') as startrek_block:
            from_branch = sdk2.parameters.String(
                'Откуда отвести ветку',
                default='dev',
                description='Ветка dev или тег, например: v1.17.1. '
                            'На основе этого автоматически вычисляется версия. '
                            'При dev инкрементируется второе число, при тегах – третье.',
                required=True,
            )
            hotfix = sdk2.parameters.Bool(
                'Хотфикс',
                default=False,
                description='Добавить в заголовок созданной задачи суффикс "(hotfix)".',
            )

        with sdk2.parameters.Group('Трекер') as nanny_block:
            create_issue = sdk2.parameters.Bool(
                'Создать задачу',
                default=True,
                description='Создать релизную задачу с чейнджлогом.',
                sub_fields={
                    'true': [
                        'components_whitelist',
                        'issue_template',
                        'filter_id',
                    ],
                },
            )
            issue_template = sdk2.parameters.JSON(
                'Шаблон релизной задачи',
                description='Должен быть валидный JSON.',
                required=True,
            )
            components_whitelist = sdk2.parameters.List(
                'Компоненты для фильтрации задач',
                description='Отфильтровать задачи из чейнджлога по заданным компонентам. '
                            'В чейнджлог попадут те задачи у которых есть хотя бы один из указанных компонентов.',
                required=True,
            )
            filter_id = sdk2.parameters.String(
                'Фильтр для дежурных',
                default='15300',
                description='Актуально для Серповых проектов. '
                            'В указанный фильтр будет добавлен ключ созданной задачи и удалены все закрытие задачи. '
                            'Возможно вам это никогда не понадобится.',
            )

            tag_linked_issues = sdk2.parameters.Bool(
                'Проставить тег к тикетам',
                default=False,
                description='Проставить тег к попавшим в релиз тикетам.',
                sub_fields={
                    'true': ['linked_issue_tags'],
                }
            )

            linked_issue_tags = sdk2.parameters.List(
                'Теги'
            )

            extra_holidays = sdk2.parameters.List(
                'Дополнительные выходные',
                description='Эти даты будут считаться выходными независимо от данных Календаря. '
                            'Например: 2019-12-31.',
            )

            only_work_days = sdk2.parameters.Bool(
                'Отводить только в рабочие дни',
                default=True,
                description='Снимите эту галочку, если хотите отводить релиз, даже если это выходной день. '
                            'Только для запуска с помощью шедулера. '
            )

        with sdk2.parameters.Group('Arc (в тестовом режиме)') as arc_block:
            use_arc = sdk2.parameters.Bool(
                'Использовать arc',
                default=False,
                description='Отвести релиз в arc.',
                sub_fields={
                    'true': [
                        'release_tag_prefix',
                        'release_branch_prefix',
                        'path_in_arcadia',
                    ],
                },
            )
            release_tag_prefix = sdk2.parameters.String(
                'Префикс релизных тегов в arc',
                default='tags/releases/frontend/web4/'
            )
            release_branch_prefix = sdk2.parameters.String(
                'Префикс релизных веток в arc',
                default='releases/frontend/web4/'
            )
            path_in_arcadia = sdk2.parameters.String(
                'Директория проекта в Аркадии',
                default='frontend/projects/web4'
            )

        node_js = parameters.NodeJsParameters

    def working_path(self, *args):
        return self.path(*args)

    def on_create(self):
        if not self.Parameters.description:
            self.Parameters.description = 'Отведение новой релизной ветки'
        if not self.Parameters.issue_template:
            author = self.author
            self.Parameters.issue_template = {
                'assignee': author,
                'followers': [author],
                'createdBy': author,
                'tags': [],
                'components': [],
            }

    def on_prepare(self):
        with self.profiler.actions.clone_scripts('Cloning scripts'):
            self.scripts.sync_resource()

        env.export_from_vault(self)

    def on_execute(self):
        self.attach_tags()
        self.create_release_branch()

    def attach_tags(self):
        tags = [
            self.Parameters.project_owner,
            self.Parameters.project_repo,
        ]

        if self.Parameters.platform:
            tags.append(self.Parameters.platform)

        if self.Parameters.experiment:
            tags.append(self.Parameters.experiment)

        if self.Parameters.hotfix:
            tags.append('hotfix')

        self.Parameters.tags = list(set(map(lambda tag: tag.lower(), self.Parameters.tags + tags)))

    def create_release_branch(self):
        opts = [{
            'task-id': str(self.id),
            'owner': self.Parameters.project_owner,
            'repo': self.Parameters.project_repo,
            'from': self.Parameters.from_branch,
            'release-branch-prefix': 'release/',
            'version-prefix': self.get_version_prefix(self.Parameters.platform),
            'experiment-prefix': self.Parameters.experiment,
        }]

        # Сейчас поддерживается либо отведение релиза для платформы, либо для эксперимента
        # Подробнее: FEI-9244
        if self.Parameters.platform or self.Parameters.experiment:
            opts.append({
                'platform': self.Parameters.experiment if self.Parameters.experiment else self.Parameters.platform,
            })

        if self.Parameters.hotfix:
            opts.append('--force-branch')
            opts.append('--hotfix')

        if self.Parameters.create_issue:
            opts.append('--create-issue')
            opts.append({
                'report': self.get_issue_report(self.Parameters.project_owner, self.Parameters.platform),
                'components-whitelist': ','.join(self.Parameters.components_whitelist),
                'issue-template': json.dumps(self.Parameters.issue_template),
            })

        if self.Parameters.filter_id and self.Parameters.project_owner == 'serp':
            opts.append({
                'update-filter': self.Parameters.filter_id,
            })

        if self.scheduler > 0 and self.Parameters.only_work_days:
            opts.append('--working-days-only')

        if self.Parameters.extra_holidays:
            opts.append({
                'extra-holidays': ','.join(self.Parameters.extra_holidays),
            })

        if self.Parameters.tag_linked_issues:
            opts.append({
                'linked-issue-tags': ','.join(self.Parameters.linked_issue_tags),
            })

        if self.Parameters.use_arc:
            opts.append('--use-arc')
            opts.append({
                'tag-prefix': self.Parameters.release_tag_prefix,
                'branch-prefix': self.Parameters.release_branch_prefix,
                'path-in-arcadia': self.Parameters.path_in_arcadia,
            })

        git_url = 'git@github.yandex-team.ru:{owner}/{repo}.git'.format(
            owner=self.Parameters.project_owner,
            repo=self.Parameters.project_repo,
        )

        with Debug('*'), GitRetryWrapper(), self.vault.ssh_key(), GitWithoutLfsProcessing(git_url):
            try:
                logging.debug('Running release creation script with options: {}'.format(opts))
                self.scripts.run_js('script/release/new-release.js', *opts)
            except SandboxSubprocessError as e:
                self.set_info(e.get_task_info(), do_escape=False)
                raise TaskFailure('Could not create release, see logs for details')

    @staticmethod
    def get_version_prefix(platform):
        return '{}/v'.format(platform) if platform else 'v'

    @staticmethod
    def get_issue_report(owner, platform):
        if owner == 'serp':
            return 'report-templates'

        if owner == 'news':
            'report-news-templates'

        if owner == 'mm-interfaces':
            if platform == 'video':
                return 'report-video-templates'

            if platform == 'images':
                return 'report-images-templates'
