# -*- coding: utf-8 -*-

import logging
import time
import dateutil
import json
import os

import jinja2

from sandbox import sdk2
from sandbox.common.utils import classproperty, singleton_property
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils.context import Debug
from sandbox.projects.sandbox_ci.task import BaseTask, PrepareWorkingCopyMixin

AB_EXPERIMENTS_API_URL = 'https://ab.yandex-team.ru'
SECS_IN_DAY = 24 * 60 * 60
BETA_SLOT_OUTPUT_PARAM = 'beta_slot'


class SandboxCiAbExperiments(PrepareWorkingCopyMixin, BaseTask):
    """Задача для заведения эксперимента в сервисе AB-экспериментов"""

    class Requirements(BaseTask.Requirements):
        cores = 1

        environments = BaseTask.Requirements.environments.default + (
            PipEnvironment('jinja2'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        project_hash = parameters.project_hash()
        build_artifacts_resources = parameters.build_artifacts_resources()

        with sdk2.parameters.Group('AB Experiments') as ab_experiments_block:
            ab_experiments_api_url = sdk2.parameters.String(
                'AB Expriments URL',
                description='URL сервиса AB экспериментов',
                default=AB_EXPERIMENTS_API_URL,
            )
            config_path = sdk2.parameters.String(
                'Путь до конфига',
                description='Путь до конфига в предоставленном ресурсе.',
            )
            experiment_title = sdk2.parameters.String(
                'Заголовок эксперимента',
                description='Заголовок используется как уникалньый и человекочитаемый идентификатор эксперимента.',
            )
            experiment_domain_prefix = sdk2.parameters.String(
                'Префикс домена эксперимента',
                description='Используется для перенаправления сетевых запросов за шаблонами на беты.',
            )
            cookie_ttl_days = sdk2.parameters.Integer(
                'TTL для cookie',
                description='Время жизни cookie в днях, которое устанавливается пользователю',
            )
            redirect_ttl_days = sdk2.parameters.Integer(
                'TTL для ссылки',
                description='Время жизни ссылки в днях',
            )
            redirect_url = sdk2.parameters.String(
                'URL для редиректа',
                description='URL для редиректа в случае успеха',
                default='https://yandex.ru/search?text=test',
            )
            redirect_error_url = sdk2.parameters.String(
                'Страница ошибки для редиректа',
                description='URL для редиректа в случае ошибки. Если не указан, будет возвращено описание ошибки',
            )
            create_sticking_url_force = sdk2.parameters.Bool(
                'Создать URL для залипания независимо от его существования',
                default=True,
            )
            is_dry_run = sdk2.parameters.Bool(
                'Тестовый прогон, ничего не заводить',
                default=False,
            )
            use_slot_from_yappy = sdk2.parameters.Bool(
                'Использовать слот из Yappy-бет для srcrwr',
                default=False,
            )
            experiment_beta_slot = sdk2.parameters.String(
                'Используется для перенаправления сетевых запросов за шаблонами на беты.',
                default='',
            )

        webhook_urls = BaseTask.Parameters.webhook_urls(
            default=[],
        )

        with sdk2.parameters.Group('Project params') as project_params:
            git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория', default={})

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('AB Experiments') as ab_experiments:
                ab_experiment_testid = sdk2.parameters.Integer('Идентификатор эксперимента (testid)')
                ab_experiment_exist = sdk2.parameters.Bool(
                    'Эксперимент уже существует',
                    description='Параметр будет выставлен в True, если эксперимент был создан ранее. Если эксперимента не было, то значение параметра будет равно False.',
                )
                ab_experiment_sticking_url = sdk2.parameters.String('Ссылка для залипания в эксперимент')

    skip_ci_scripts_checkout = False

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Создание эксперимента в AB'

    @classproperty
    def task_label(self):
        return 'ab-experiments'

    @singleton_property
    def project_name(self):
        return self.Parameters.project_github_repo

    @singleton_property
    def experiment_tag_name(self):
        return 'crowdtest_{owner}_{repo}'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
        )

    @singleton_property
    def redirect_ttl_days(self):
        if self.Parameters.redirect_ttl_days:
            return self.Parameters.redirect_ttl_days

        return self.config.get_deep_value(['deploy', 'ab_experiment', 'redirect_ttl_days'], 31)

    @singleton_property
    def cookie_ttl_days(self):
        if self.Parameters.cookie_ttl_days:
            return self.Parameters.cookie_ttl_days

        return self.config.get_deep_value(['deploy', 'ab_experiment', 'cookie_ttl_days'])

    def format_experiment_url(self, experiment_testid):
        return '{base_url}/testid/{testid}'.format(
            base_url=self.Parameters.ab_experiments_api_url,
            testid=experiment_testid,
        )

    @in_case_of('use_overlayfs', 'execute_with_checkout')
    def execute(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)

        self.create_ab_experiment()

    def execute_with_checkout(self):
        with self.prepare_working_copy_context():
            os.symlink(str(self.project_sources_dir), str(self.project_dir))

            self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)

            self.create_ab_experiment()

    def create_ab_experiment(self):
        same_task = self.find_first_task_with_same_experiment_name()
        is_outdated = self.check_is_outdated(same_task)

        sticky_url_needed = False

        if self.Parameters.create_sticking_url_force or is_outdated:
            logging.debug('Creating AB-Experiment sticky url for {title}, outdated: {is_outdated}, force: {force}'.format(
                title=self.Parameters.experiment_title,
                is_outdated=is_outdated,
                force=self.Parameters.create_sticking_url_force,
            ))

            self.Parameters.ab_experiment_exist = False
            sticky_url_needed = True
        else:
            existed_test_id, existed_sticky_url = self.extract_ab_experiment_from_task(same_task)
            logging.debug('Skip creating AB-Experiment sticky url for {title}. Already exists for test-id "{testid}" in task "{task_id}"'.format(
                title=self.Parameters.experiment_title,
                testid=existed_test_id,
                task_id=same_task.id,
            ))

            self.Parameters.ab_experiment_exist = True

        testid, sticky_url = self.create_experiment(sticky_url_needed)

        if not sticky_url_needed:
            sticky_url = existed_sticky_url

        self.Parameters.ab_experiment_testid = testid
        self.set_info('AB Experiment: <a href="{url}">#{testid}</a>'.format(
            url=self.format_experiment_url(testid),
            testid=testid,
        ), do_escape=False)

        self.Parameters.ab_experiment_sticking_url = sticky_url
        self.set_info('URL for sticking: <a href="{url}">#{testid}</a>'.format(
            url=sticky_url,
            testid=testid,
        ), do_escape=False)

    def find_first_task_with_same_experiment_name(self):
        return sdk2.Task.find(
            type=self.type,
            children=True,
            input_parameters=dict(
                experiment_title=self.Parameters.experiment_title,
                experiment_domain_prefix=self.Parameters.experiment_domain_prefix,
                is_dry_run=False,
            ),
            output_parameters=dict(
                ab_experiment_exist=False,
            ),
        ).order(-sdk2.Task.id).first()

    def check_is_outdated(self, same_task):
        if not same_task:
            return True

        testid, sticking_url = self.extract_ab_experiment_from_task(same_task)
        if not testid or not sticking_url:
            return True

        if self.check_is_sticking_url_outdated(same_task):
            return True

        return False

    def extract_ab_experiment_from_task(self, task):
        testid = task.Parameters.ab_experiment_testid
        sticking_url = task.Parameters.ab_experiment_sticking_url

        return testid, sticking_url

    def check_is_sticking_url_outdated(self, same_task):
        # Преобразуем строку с датой в timestamp
        same_task_created_datetime = dateutil.parser.parse(str(same_task.created))
        same_task_created_timestamp = int(time.mktime(same_task_created_datetime.timetuple()))

        # Проверяем, что найденный URL для залипания до сих пор валиден
        invalidation_timestamp = same_task_created_timestamp + self.redirect_ttl_days * SECS_IN_DAY

        return invalidation_timestamp <= int(time.time())

    def create_experiment(self, sticky_url_needed):
        config = self.get_experiment_config()
        config_path = self.get_experiment_config_path(config)

        data = self.run_create_command(config_path, sticky_url_needed)

        testid = data.get('testid')
        sticking_url = data.get('stickyUrl', None)

        return testid, sticking_url

    def get_experiment_config(self):
        config = self.render_experiment_config()
        template = config.get('template')

        template['params'] = json.dumps(template['params'])

        return template

    def render_experiment_config(self):
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(str(self.project_dir)))
        config_template = env.get_template(self.Parameters.config_path)

        if self.Parameters.use_slot_from_yappy:
            experiment_beta_slot = self.task_dependencies.get_output_parameter(BETA_SLOT_OUTPUT_PARAM)
        elif self.experiment_beta_slot:
            experiment_beta_slot = self.Parameters.experiment_beta_slot
        else:
            experiment_beta_slot = self.Parameters.experiment_domain_prefix + '.si.yandex-team.ru:80'

        if not experiment_beta_slot:
            raise TaskFailure('Could not detect beta slot for experiment')

        experiment_beta_slot_list = experiment_beta_slot.split(';')
        experiment_beta_slot = experiment_beta_slot_list[0]

        logging.debug('Beta for experiment has slots: [{}]'.format(', '.join(experiment_beta_slot_list)))

        # https://github.yandex-team.ru/serp/turbo/blob/50d9a4f98a0065b57a89c9b7464a36b163dc2614/.config/pr-experiment.json#L32
        experiment_beta_slot_host, experiment_beta_slot_port = experiment_beta_slot.split(':')

        content = config_template.render({
            "experiment_title": self.Parameters.experiment_title,
            "experiment_beta_slot": experiment_beta_slot,
            "experiment_beta_slot_host": experiment_beta_slot_host,
            "experiment_beta_slot_port": experiment_beta_slot_port,
            "experiment_beta_slot_list": experiment_beta_slot_list,
            # for backwards compatibility
            "experiment_domain_prefix": self.Parameters.experiment_domain_prefix,
        })

        return json.loads(content)

    def get_experiment_config_path(self, config):
        config = self.get_experiment_config()

        path = str(self.working_path('.testid-config.json'))

        with open(path, 'w') as config_file:
            json.dump(config, config_file)

        logging.debug('config json saved to {path}'.format(path=path))

        return path

    def run_create_command(self, config_path, sticky_url_needed):
        params = {
            'config': config_path,
            'tag': [self.experiment_tag_name],
            'dry': self.Parameters.is_dry_run,
            'update-if-changed': True,
            'await-deploy': True,
        }

        if sticky_url_needed:
            params.update({
                'create-sticky-url': True,
                'sticky-url-redirect-to': self.Parameters.redirect_url,
                'sticky-url-ttl': self.cookie_ttl_days,
                'sticky-url-ts': self.redirect_ttl_days,
            })

        with Debug('*'):
            return self.scripts.run_js(
                'script/testid-cli.js',
                'create',
                params,
            )
