# -*- coding: utf-8 -*-

import json
import logging
import os

from sandbox.common import utils
from sandbox.projects.sandbox_ci.utils.request import send_request

AB_EXPERIMENTS_API_URL = 'https://ab.yandex-team.ru'


class ABExperimentsApi(object):
    def __init__(self, token, url=AB_EXPERIMENTS_API_URL):
        self._url = url
        self._token = token
        self._headers = {
            'Authorization': 'OAuth {token}'.format(token=self._token),
            'Content-Type': 'application/json',
            'User-Agent': 'ab_experiments_api (sandbox task, {task_id})'.format(
                task_id=os.environ.get('SANDBOX_TASK_ID', '')
            ),
        }

    class ApplicationStatus(utils.Enum):
        # только что создана
        NEW = 'NEW'
        # попала в конфиг экспериментов (но ещё не выкатилась)
        IN_CONFIG = 'IN_CONFIG'
        # выкатилась в продакшен
        ONLINE = 'ONLINE'
        # выключена навсегда (назад вернуть нельзя, только если перевыкатить)
        ARCHIVED = 'ARCHIVED'

    def get_full_url(self, path):
        """
        Возвращает полный URL-адрес, используя базовый адрес и путь.

        :param path: Запрашиваемый путь
        :type path: str
        :rtype: str
        """
        return '{}/{}'.format(self._url, path)

    def create_experiment(self, experiment):
        """
        Создаёт эксперимент с заданными параметрами и заголовком.

        :param experiment: Данные для создания эксперимента
        :type experiment: dict
        :rtype: dict
        """
        url = self.get_full_url('api/testid')
        data = json.dumps(experiment)
        res = send_request('post', url, headers=self._headers, data=data, verify=False)

        logging.info('Creating AB-Experiment: {}'.format(data))

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def add_tag_to_testid(self, testid, tag_name):
        """
        Ставит тег на указанный эексперимент.

        :param testid: Идентификатор эксперимента
        :type testid: int
        :param tag_name: Наименование устанавливаемого тега
        :type tag_name: str
        :rtype: dict
        """
        url = self.get_full_url('api/tag?name={}'.format(tag_name))

        tag_data = json.dumps({
            'name': tag_name,
            'target_type': 2,  # 2 – testid
            'target_id': testid,
        })

        res = send_request('post', url, headers=self._headers, data=tag_data, verify=False)

        logging.info('Adding tag with "{tag_name}" name to "{testid}" test-id'.format(
            tag_name=tag_name,
            testid=testid,
        ))

        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def get_experiments_by_tag(self, tag_name, status='active'):
        """
        Возвращает все экперименты, у которых указан переданный тег.

        :param tag_name: Наименование тега
        :type tag_name: str
        :param status: Статус экспериментов
        :type status: bool
        :rtype: dict
        """
        url = self.get_full_url('api/testid?form=short&type=ABT&status={status}&tags={tag_name}'.format(
            status=status,
            tag_name=tag_name,
        ))

        res = send_request('get', url, headers=self._headers, verify=False)

        logging.debug('Getting AB-Experiments for tag with "{tag_name}" name in "{status}" status'.format(
            tag_name=tag_name,
            status=status,
        ))

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def get_test_id(self, test_id):
        """
        Возвращает данные для указанного test_id
        :param test_id: test_id
        :type test_id: str
        :rtype: dict
        """
        url = self.get_full_url('api/testid/{}?format=json'.format(test_id))

        res = send_request('get', url, headers=self._headers, verify=False)

        logging.debug('Getting test id {} info'.format(test_id))

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def create_application(self, **data):
        """
        Создает заявку на тестирование в проде.

        Все аргументы будут сериализованы в тело запроса:
        https://nda.ya.ru/3TcGw4

        :rtype: dict
        """
        url = self.get_full_url('api/zero_testing/application')
        logging.debug('Creating application for {}'.format(data['testids']))

        res = send_request('post', url, headers=self._headers, data=json.dumps(data), verify=False)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def get_application(self, app_id):
        """
        Возвращает информацию о конкретной заявке на тестирование в проде.

        https://nda.ya.ru/3TcGyG

        :param app_id: идентификатор заявки, полученный при создании
        :type app_id: str
        :rtype: dict
        """
        url = self.get_full_url('api/zero_testing/application/{}'.format(app_id))

        logging.debug('Getting info about application#{}'.format(app_id))

        res = send_request('get', url, headers=self._headers, verify=False)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)

    def get_flag(self, handler_name, flag_name):
        """
        Возвращает информацию об указанном флаге
        :param handler_name: Имя хэндлера
        :type handler_name: str
        :param flag_name: Имя флага
        :type flag_name: str
        :rtype: dict
        """
        url = self.get_full_url('api/v1/flag_storage/flag/{}/{}?format=json'.format(
            handler_name,
            flag_name
        ))

        res = send_request('get', url, headers=self._headers, verify=False)

        logging.debug('Getting flag {} info for handler {}'.format(flag_name, handler_name))

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}
        return json.loads(res.text)
