# -*- coding: utf-8 -*-

import os
import json
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.sandboxsdk import process

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.process import format_args_for_run_process
from sandbox.projects.sandbox_ci.utils.context import Debug, Node

TESTID_CLI_PACKAGE = '@yandex-int/si.ci.testid-cli@6.2'

logger = logging.getLogger(__name__)


class TaskRequirements(sdk2.Requirements):
    dns = ctm.DnsType.LOCAL
    cores = 1

    class Caches(sdk2.Requirements.Caches):
        pass


class TaskParameters(parameters.CommonParameters):
    _container = parameters.environment_container()

    with sdk2.parameters.Group('Experiments') as common_block:
        experiment_tag = sdk2.parameters.String(
            'Тэг эксперимента',
            description='Тэг, который должен иметь искомый эксперимент.',
            required=True,
        )
        experiment_title = sdk2.parameters.String(
            'Заголовок эксперимента',
            description='Заголовок эксперимента, по которому будет осуществляться поиск.',
            required=True,
        )

    with sdk2.parameters.Group('Flow') as flow_block:
        dry_run = sdk2.parameters.Bool(
            'Dry Run',
            description='Вывести операции, которые будут вызваны, но не выполнять их.',
            default=False,
        )

    node_js = parameters.NodeJsParameters


class SandboxCiAbExperimentsCleanup(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
    Таск, запускающий архивацию созданных для пулл-реквеста экспериментов в AB.
    """

    class Requirements(TaskRequirements):
        pass

    class Parameters(TaskParameters):
        pass

    # Lifecycle manager requires the following props/methods: `project_name`, `project_dir`, `working_path`.
    @property
    def project_name(self):
        return ''

    @property
    def project_dir(self):
        return self.working_path()

    def working_path(self, *args):
        return self.path(*args)

    def on_execute(self):
        self._set_env_variables()

        with Debug('*'), Node(self.Parameters.node_js_version):
            experiments = self._search_experiments()

            logger.debug('Found {length} experiments: {experiments}'.format(
                length=len(experiments),
                experiments=experiments
            ))

            if len(experiments) > 0:
                self._archive_experiments(experiments)

    def _set_env_variables(self):
        os.environ.update({
            'NODE_EXTRA_CA_CERTS': '/etc/ssl/certs/YandexInternalRootCA.pem',
            'NPM_CONFIG_REGISTRY': 'http://npm.yandex-team.ru',
            'NPM_CONFIG_USER_AGENT': 'npm/6.2.0 (verdaccio yandex canary)',
            'AB_EXPERIMENTS_TOKEN': self.vault.read('env.AB_EXPERIMENTS_TOKEN'),
        })

    def _search_experiments(self):
        """
        :rtype: list of int
        """
        tag = self.Parameters.experiment_tag
        title = self.Parameters.experiment_title

        logger.debug('Trying to search experiments for tag "{tag}" and title "{title}"'.format(
            tag=tag,
            title=title,
        ))

        command = format_args_for_run_process([
            'npx {cli} search'.format(cli=TESTID_CLI_PACKAGE),
            {
                'tag': tag,
                'title': title,
                'json': True,
            }
        ])

        p = self._run_process(command, 'testid_cli_search')

        with open(p.stdout_path, 'r') as file:
            output = file.read()

        return json.loads(output.strip())

    def _archive_experiments(self, experiments):
        """
        :type experiments: list of int
        """
        logger.debug('Trying to archive experiments: {}'.format(experiments))

        command = format_args_for_run_process([
            'npx {cli} archive'.format(cli=TESTID_CLI_PACKAGE),
            {
                'id': experiments,
                'dry-run': self.Parameters.dry_run
            }
        ])

        self._run_process(command, 'testid_cli_archive')

    def _run_process(self, command, log_prefix):
        return process.run_process(
            command,
            work_dir=str(self.working_path()),
            log_prefix=log_prefix,
            outputs_to_one_file=False,
            shell=True,
        )
