# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2

from sandbox.common.utils import singleton_property, classproperty
from sandbox.projects.sandbox_ci.utils.git import git_clone

from sandbox.projects.sandbox_ci.utils.context import Node, GitRetryWrapper
from sandbox.projects.sandbox_ci.task import BaseBuildTask


class SandboxCiBuildDockerImage(BaseBuildTask):
    class Requirements(BaseBuildTask.Requirements):
        disk_space = 10 * 1024  # выхожу за пределы 5Gb, кмк из-за использование docker vfs storage
        privileged = True

    class Parameters(BaseBuildTask.Parameters):
        with sdk2.parameters.Group('Docker registry') as docker_user_block:
            user_login = sdk2.parameters.String(
                'User login',
                description='Имя пользователя в Docker реестре',
                required=True,
            )
            registry_host = sdk2.parameters.String(
                'Host',
                default='registry.yandex.net',
                description='Адрес реестра',
                required=True,
            )

        with sdk2.parameters.Group('Docker commands') as docker_user_block:
            build_command = sdk2.parameters.String(
                'Build command',
                description='Команда сборки docker-образа',
                required=True,
                default='make docker-build'
            )
            push_command = sdk2.parameters.String(
                'Push command',
                description='Команда публикации docker-образа',
                required=True,
                default='make docker-publish'
            )

    skip_ci_scripts_checkout = True

    lifecycle_steps = {
        'start-daemon': 'sudo service docker start',
        'login': 'docker login -u {user_login} -p {registry_token} {registry_host}',
        'stop-daemon': 'sudo service docker stop',
    }

    def on_save(self):
        super(SandboxCiBuildDockerImage, self).on_save()
        # Запускаемся на контейнере с настройками для докера,
        # а остальные таски запускаются на контейнере без него, потому что
        # он ломает сеть на агенте: воспроизводится только на тех агентах
        # на которых нет сетевого интерфейса eth0
        # https://st.yandex-team.ru/SANDBOX-4592
        self.Parameters._container = 495494219

    def on_prepare(self):
        super(SandboxCiBuildDockerImage, self).on_prepare()

        self.clone_project()

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Сборка и публикация докер образа'

    @singleton_property
    def lifecycle(self):
        user_login = self.Parameters.user_login

        self.lifecycle_steps.update({
            'build': self.Parameters.build_command,
            'push': self.Parameters.push_command,
        })

        return super(SandboxCiBuildDockerImage, self).lifecycle.update_vars(
            user_login=user_login,
            registry_host=self.Parameters.registry_host,
            registry_token=self.registry_token,
        )

    @singleton_property
    def registry_token(self):
        user_login = self.Parameters.user_login
        owner = self.Parameters.owner
        vault_name = '{}_registry_oauth_token'.format(user_login)

        logging.debug('getting vault "{vault}" for "{owner}" owner'.format(
            vault=vault_name,
            owner=owner,
        ))

        return sdk2.Vault.data(owner, vault_name)

    @property
    def skip_lfs_checkout(self):
        return True

    def on_finish(self, prev_status, status):
        pass

    def clone_project(self):
        with GitRetryWrapper(), self.vault.ssh_key():
            ref = self.Parameters.project_git_base_ref
            target_dir = str(self.path(self.project_name))
            url = 'ssh://git@github.yandex-team.ru/{owner}/{repo}.git'.format(
                owner=self.Parameters.project_github_owner,
                repo=self.Parameters.project_github_repo
            )

            logging.debug('Cloning scripts from {url}#{ref} to {target_dir}'.format(url=url, ref=ref, target_dir=target_dir))
            git_clone(url, ref, target_dir)

    def start_daemon(self):
        with self.profiler.actions.start_daemon('Start Docker daemon'):
            self.lifecycle('start-daemon', work_dir=self.path(self.project_name))

    def stop_daemon(self):
        with self.profiler.actions.start_daemon('Stop Docker daemon'):
            self.lifecycle('stop-daemon', work_dir=self.path(self.project_name))

    def login(self):
        with self.profiler.actions.login('Log in to a Docker registry'):
            self.lifecycle('login', work_dir=self.path(self.project_name))

    def build_image(self):
        with self.profiler.actions.build_image('Build an image from a Dockerfile'):
            self.lifecycle('build', work_dir=self.path(self.project_name))

    def push_image(self):
        with self.profiler.actions.publish('Push an image to a registry'):
            self.lifecycle('push', work_dir=self.path(self.project_name))

    def execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            with Node(self.Parameters.node_js_version), self.vault.ssh_key():
                self.start_daemon()
                self.login()
                self.build_image()
                self.push_image()
                self.stop_daemon()
