# -*- coding: utf-8 -*-

import json

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.utils import singleton_property

from sandbox.sandboxsdk.errors import SandboxSubprocessError

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.utils.context import Debug
from sandbox.projects.sandbox_ci.task.BaseTask import BaseTask


class SandboxCiExpflags(BaseTask):
    """Синхронизирует проектные флаги из папки expflags с flag storage админки экспериментов"""

    class Requirements(BaseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        project_name = sdk2.parameters.String(
            'Project name',
            default='web4',
            required=True,
        )

        with sdk2.parameters.Group('Параметры скрипта') as script_block:
            flag_template = sdk2.parameters.JSON(
                'Общие настройки флага',
                description=u'Общие настройки для всех expflag\'ов в сторадже админки экспериментов.',
                default={},
            )
            expflags_options = sdk2.parameters.JSON(
                'Опции для expflags-cli',
                description=u'Опции для тонкой настройки expflags-cli.',
                default={},
            )
            expflags_pattern = sdk2.parameters.String(
                'Паттерн, находящий JSON-файлы',
                description=u'Паттерн должен находить только JSON-файлы. Например, `*.json`.',
                required=True
            )
            expflags_tags = sdk2.parameters.List(
                'Теги флагов',
                description=u'Будут добавляться для всех создаваемых флагов.',
            )
            upload_exp_flags = sdk2.parameters.Bool(
                'Загрузить добавленные флаги',
                description=u'Новые флаги из проектной папки expflags будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )
            update_exp_flags = sdk2.parameters.Bool(
                'Обновлять измененные флаги',
                description=u'Изменения в проектных флагах в папке expflags будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )
            deleted_exp_flags = sdk2.parameters.List(
                'Список имён удалённых флагов',
                description=u'Будут помечены как устаревшие в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
            )
            source = sdk2.parameters.String(
                'Источник флагов',
                description=u'Поле source, с которым будет происходить удаление флагов. Например, `web4`',
                required=False
            )
            deleted_exp_flags_handler = sdk2.parameters.String(
                'Хэндлер удаляемых флагов',
                description=u'Хэндлер, по которому определяются удаляемые флаги',
                required=False,
            )

        with sdk2.parameters.Group('Ресурсы') as resources_block:
            expflags_resource = sdk2.parameters.Resource(
                'Ресурс expflags',
                resource_type=SANDBOX_CI_ARTIFACT,
                required=True,
                multiple=False,
            )

        base_params = BaseTask.Parameters

    github_context = u'[Sandbox CI] Синхронизация проектных флагов с flag storage'

    @singleton_property
    def project_name(self):
        return self.Parameters.project_name

    @singleton_property
    def expflags_pattern(self):
        return self.path(self.Parameters.expflags_pattern)

    skip_ci_scripts_checkout = False

    def execute(self):
        errors = []

        self.unpack_expflags()

        try:
            self.upload_expflags()
        except SandboxSubprocessError as e:
            errors.append(e)

        try:
            self.update_expflags()
        except SandboxSubprocessError as e:
            errors.append(e)

        try:
            self.mark_expflags_as_removed_from_code()
        except SandboxSubprocessError as e:
            errors.append(e)

        if errors:
            raise TaskFailure('\n'.join([str(error) for error in errors]))

    def upload_expflags(self):
        tags = list(self.Parameters.expflags_tags)
        tags.append('expflags-cli')
        
        if self.Parameters.upload_exp_flags:
            self.run_script('upload', '"{}"'.format(self.expflags_pattern), {
                'common-flag': json.dumps(self.Parameters.flag_template),
                'options': json.dumps(self.Parameters.expflags_options),
                'tags': tags,
            })

    def update_expflags(self):
        if self.Parameters.update_exp_flags:
            self.run_script('update', '"{}"'.format(self.expflags_pattern), {
                'common-flag': json.dumps(self.Parameters.flag_template),
                'options': json.dumps(self.Parameters.expflags_options),
            })

    def mark_expflags_as_removed_from_code(self):
        args = {
            'source': self.Parameters.source,
        }

        if self.Parameters.deleted_exp_flags_handler:
            args['handler'] = self.Parameters.deleted_exp_flags_handler

        if self.Parameters.deleted_exp_flags:
            self.run_script('mark-as-removed-from-code', ' '.join(self.Parameters.deleted_exp_flags), args)

    def unpack_expflags(self):
        """
        Распаковывает архив epxflags.tag.gz на файловую систему
        """
        self.artifacts.save_resources([self.Parameters.expflags_resource], self.path())
        self.artifacts.unpack(sdk2.Path('expflags.tar.gz'))

    def run_script(self, command, *args, **kwargs):
        with Debug('si:ci:expflags:*'):
            return self.scripts.run_js(
                './script/expflags-cli.js',
                command,
                *args,
                log_prefix=command,
                **kwargs
            )
