# -*- coding: utf-8 -*-

import os
import json
import logging
import itertools

from sandbox import sdk2
from sandbox.common.types import misc as ctm

from sandbox.projects.common.arcadia.sdk import mount_arc_path, mounted_path_svnversion
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.process import format_args_for_run_process
from sandbox.projects.sandbox_ci.utils.context import Debug, Node
from sandbox.sandboxsdk import process

DEPENDENCIES = [
    '@yandex-int/si.ci.expflags-cli@9'
]

logger = logging.getLogger(__name__)

class TaskRequirements(sdk2.Requirements):
    dns = ctm.DnsType.LOCAL
    cores = 1

    class Caches(sdk2.Requirements.Caches):
        pass


class TaskParameters(parameters.CommonParameters):
    _container = parameters.environment_container()

    with sdk2.parameters.Group('Flow Control') as flow_block:
        diff_based_run = sdk2.parameters.Bool(
            'Diff based run',
            description='Архивировать только те флаги, которых нет в рабочей копии.',
            default=False,
            required=False,
        )
        dry_run = sdk2.parameters.Bool(
            'Dry run',
            description='Выводить информацию по действиям, но не выполнять их.',
            default=False,
            required=False,
        )

    with sdk2.parameters.Group('Script Parameters') as script_block:
        flag_source = sdk2.parameters.String(
            'Source of flag',
            description='Поле source для флагов, с которым будет происходить удаление. Поддерживается запуск только для одного источника. Например, "web4".',
            required=True,
        )
        flag_tag = sdk2.parameters.String(
            'Tag of flag',
            description='Тег, по которому могут быть найдены флаги для дальнейшего удаления.',
            required=True,
        )

        with diff_based_run.value[True]:
            expflags_pattern = sdk2.parameters.String(
                'Pattern for searching for local flags',
                description='Паттерн должен находить только JSON-файлы. Например, `*.json`.',
                required=True
            )

    with diff_based_run.value[True]:
        with sdk2.parameters.Group('Arc Parameters') as arc_parameters:
            path_in_arcadia = sdk2.parameters.String(
                'Arc path',
                description='Путь до проекта, в котором будут запускаться сприкты относительно корня Аркадии.',
                default='.',
                required=True,
            )
            pr_merge_ref = sdk2.parameters.String(
                'Arc ref',
                description='Ветка или коммит, на котором нужно проверить состояние флагов.',
                default='trunk',
                required=False,
            )

    node_js = parameters.NodeJsParameters


class TaskContext(sdk2.Context):
    pass

class SandboxCiExpflagsCleanup(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
    Таск для удаления неиспользуемых флагов после закрытия PR.
    """

    class Requirements(TaskRequirements):
        pass

    class Parameters(TaskParameters):
        pass

    class Context(TaskContext):
        pass

    github_context = u'[Sandbox CI] Удаление неиспользуемых флагов'

    @property
    def project_name(self):
        return ''

    @property
    def project_dir(self):
        return self.working_path()

    def working_path(self, *args):
        return self.path(*args)

    def on_execute(self):
        self._set_env_variables()

        with Debug('*'), Node(self.Parameters.node_js_version):
            self._install_dependencies()

            expflags = self._find_expflags()

            logger.debug('Found flags for the "{tag}" tag: {flags}'.format(
                tag=self.Parameters.flag_tag,
                flags=expflags,
            ))

            if self.Parameters.diff_based_run:
                expflags = self._filter_by_local_flags(expflags)

            self._archive_expflags(expflags)

    def _set_env_variables(self):
        os.environ.update({
            'NODE_EXTRA_CA_CERTS': '/etc/ssl/certs/YandexInternalRootCA.pem',
            'NPM_CONFIG_REGISTRY': 'http://npm.yandex-team.ru',
            'AB_EXPERIMENTS_TOKEN': self.vault.read('env.AB_EXPERIMENTS_TOKEN'),
        })

    def _find_expflags(self):
        command = format_args_for_run_process([
            'npx --no-install expflags search',
            {
                'source': self.Parameters.flag_source,
                'tags': self.Parameters.flag_tag,
                'outputFields': ['name', 'source', 'handler', 'tags'],
            },
        ])

        p = self._run_process(command, 'expflags_search')

        with open(p.stdout_path, 'r') as fd:
            return json.load(fd)

    def _filter_by_local_flags(self, expflags):
        with self._with_arc() as arcadia_root:
            self._show_arc_info(arcadia_root)
            project_path = os.path.join(arcadia_root, self.Parameters.path_in_arcadia)
            local_flags = self._find_local_flags_by_names(project_path, expflags)

        logger.debug('Found local flags: {flags}'.format(
            flags=local_flags,
        ))

        non_exists_local = []

        for flag in expflags:
            local_flag = list(filter(lambda x: x['name'] == flag['name'] and x['handler'] == flag['handler'], local_flags))

            if len(local_flag) == 0:
                non_exists_local.append(flag)

        logger.debug('Flags that exist in AB, but not on the FS: {flags}'.format(
            flags=non_exists_local,
        ))

        return non_exists_local

    def _find_local_flags_by_names(self, cwd, expflags):
        command = format_args_for_run_process([
            'npx --no-install expflags search-local',
            '"{}/{}"'.format(cwd, self.Parameters.expflags_pattern),
            {
                'name': map(lambda x: x['name'], expflags),
                'outputFields': ['name', 'handler'],
            },
        ])

        p = self._run_process(command, 'expflags_search_local')

        with open(p.stdout_path, 'r') as fd:
            return json.load(fd)

    def _archive_expflags(self, expflags):
        for handler, group in itertools.groupby(expflags, lambda x: x['handler']):
            flags = map(lambda x: x['name'], group)

            command = format_args_for_run_process([
                'npx --no-install expflags mark-as-removed-from-code',
                ' '.join(flags),
                {
                    'source': self.Parameters.flag_source,
                    'handler': handler,
                    'dry-run': self.Parameters.dry_run,
                },
            ])

            self._run_process(command, 'expflags_search.handler.{}'.format(handler))

    def _install_dependencies(self):
        self._run_process('npm install --no-save {}'.format(' '.join(DEPENDENCIES)), 'install_dependencies')

    def _run_process(self, command, log_prefix, **kwargs):
        return process.run_process(
            command,
            log_prefix=log_prefix,
            shell=True,
            outputs_to_one_file=False,
            **kwargs
        )

    def _with_arc(self):
        url = 'arcadia-arc:/#{ref}'.format(
            ref=self.Parameters.pr_merge_ref,
        )

        return mount_arc_path(
            url=url,
            use_arc_instead_of_aapi=True,
            arc_oauth_token=self.vault.read('env.ARC_TOKEN'),
        )

    def _show_arc_info(self, arcadia_root):
        repo_info = mounted_path_svnversion(arcadia_root, arc_vcs=True)

        logging.info("Using trunk, info {}".format(repo_info))
