# -*- coding: utf-8 -*-

import json
from sandbox import sdk2

from sandbox.common.types import misc as ctm, task as ctt

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement

PAIRS_FILENAME_TEMPLATE = '{name}.dict.json'
OBJECT_FILENAME_TEMPLATE = '{name}.object.dict.json'


class SandboxCiFetchBlockstat(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """Задача для скачивания словаря из стата и сохранением его в ресурс с таким же именем."""

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL
        cores = 1
        environments = (
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        environment = parameters.EnvironmentParameters

        dict_name = sdk2.parameters.String('Имя словаря, который нужно скачать', required=True, default='blockstat')

    @property
    def object_filename(self):
        return OBJECT_FILENAME_TEMPLATE.format(
            name=self.Parameters.dict_name,
        )

    @property
    def pairs_filename(self):
        return PAIRS_FILENAME_TEMPLATE.format(
            name=self.Parameters.dict_name,
        )

    def fetch_dict(self):
        data = self.statface.get_dict(self.Parameters.dict_name, format='json')

        return json.loads(data)

    @staticmethod
    def translate_dict(data):
        return [
            [value, key] for key, value in data.iteritems()
        ]

    @staticmethod
    def write_to_file(filename, data):
        with open(filename, 'w') as dict_file:
            dict_file.write(json.dumps(data))

    def on_save(self):
        super(SandboxCiFetchBlockstat, self).on_save()
        semaphore_name = 'fetch_blockstat'
        weight = 1
        exec_limit = 1

        self.Requirements.semaphores = ctt.Semaphores(
            [(semaphore_name, weight, exec_limit)],
            ctt.Status.Group.BREAK | ctt.Status.Group.FINISH,
        )

    def on_execute(self):
        dict_data = self.fetch_dict()

        # TODO merge resources and move Map cons converting to projects, FEI-9827, FEI-10644
        self.write_to_file(self.object_filename, dict_data)

        self.artifacts.create_artifact_resource(
            relative_path=self.object_filename,
            resource_type=SANDBOX_CI_ARTIFACT,
            dict=self.Parameters.dict_name,
            format='object',
        )

        pairs_data = self.translate_dict(dict_data)

        self.write_to_file(self.pairs_filename, pairs_data)

        self.artifacts.create_artifact_resource(
            relative_path=self.pairs_filename,
            resource_type=SANDBOX_CI_ARTIFACT,
            dict=self.Parameters.dict_name,
            format='pairs',
        )
