# -*- coding: utf-8 -*-
import os
import logging
import json
import urllib

from sandbox import common
from sandbox import sdk2

from sandbox.common.types import task as ctt
from sandbox.projects.common.nanny import nanny

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import github, git
from sandbox.projects.sandbox_ci.task import ManualRunMetaTaskMixin
from sandbox.projects.sandbox_ci.task.skip_validator import SandboxCiSkipValidator
from sandbox.projects.sandbox_ci.decorators.skip_subtask import skip_subtask
from sandbox.projects.sandbox_ci.utils.static_host import get_static_workcopy_name

from sandbox.projects.sandbox_ci.pulse import parameters as pulse_parameters
from sandbox.projects.sandbox_ci.pulse.subtasks import create_pulse_subtasks
from sandbox.projects.sandbox_ci.pulse.resource_finder import ResourceTemplatesFinder, ResourceNotFoundException

from sandbox.projects.sandbox_ci.sandbox_ci_expflags import SandboxCiExpflags
from sandbox.projects.sandbox_ci.sandbox_ci_yappy_deploy import SandboxCiYappyDeploy
from sandbox.projects.sandbox_ci.sandbox_ci_ab_experiments import SandboxCiAbExperiments
from sandbox.projects.sandbox_ci.sandbox_ci_nanny_ensure_deploy import SandboxCiNannyEnsureDeploy
from sandbox.projects.sandbox_ci.sandbox_ci_static_s3 import SandboxCiStaticS3
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_hermione import SandboxCiFijiHermione
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_hermione_e2e import SandboxCiFijiHermioneE2E
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_manual_test_run import SandboxCiFijiManualTestRun
from sandbox.projects.sandbox_ci.sandbox_ci_palmsync.synchronize import SandboxCiPalmsyncSynchronize
from sandbox.projects.sandbox_ci.sandbox_ci_palmsync.validate import SandboxCiPalmsyncValidate
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_unit_client import SandboxCiFijiUnitClient
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_unit_priv import SandboxCiFijiUnitPriv
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_unit_tmpl import SandboxCiFijiUnitTmpl
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_unit_spec import SandboxCiFijiUnitSpec
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_linters import SandboxCiFijiLinters
from sandbox.projects.sandbox_ci.sandbox_ci_blockstat_test import SandboxCiBlockstatTest
from sandbox.projects.sandbox_ci.sandbox_ci_testids_for_flags import SandboxCiTestidsForFlags
from sandbox.projects.sandbox_ci.sandbox_ci_testpalm_suite_runner import SandboxCiTestpalmSuiteRunner
from sandbox.projects.sandbox_ci.sandbox_ci_testpalm_suite_runner.utils import should_start_testpalm_suite_runner_task
from sandbox.projects.sandbox_ci.sandbox_ci_test_cost_reporter import SandboxCiTestCostReporter
from sandbox.projects.sandbox_ci.sandbox_ci_test_changes_collector import SandboxCiTestChangesCollector
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_build import SandboxCiFijiBuild
from sandbox.projects.sandbox_ci.constants import MailingLists
from sandbox.projects.sandbox_ci.managers.arc.arc_cli import get_arc_last_merge_commit_issues

PLATFORMS = ('desktop', 'touch-pad', 'touch-phone')
DEFAULT_PLATFORMS = {
    'images': [
        'desktop',
        'touch-pad',
        'touch-phone'
    ],

    'video': [
        'desktop',
        'touch-phone'
    ]
}


class SandboxCiFiji(nanny.ReleaseToNannyTask2, ManualRunMetaTaskMixin, SandboxCiFijiBuild):
    """Австосборка Картинок и Видео (mm-interfaces/fiji)"""

    project_name = 'fiji'

    class Parameters(SandboxCiFijiBuild.Parameters):
        notify_subtask_statuses = parameters.notify_subtask_statuses(default=False)
        platform_description = u'Список платформ, для которых нужно запустить проверку, в json-формате {"images": ["desktop"], "video": ["touch-phone"]}'

        with SandboxCiFijiBuild.Parameters.project_build_block() as project_build_block:
            selective_checks = parameters.selective_checks()

        with sdk2.parameters.Group('Hermione') as hermione_block:
            hermione_platforms = sdk2.parameters.List('Hermione platforms', required=True, default=PLATFORMS)
            new_hermione_platforms = sdk2.parameters.JSON(
                'Hermione platforms', required=True,
                default=DEFAULT_PLATFORMS,
                description=platform_description,
            )

        with sdk2.parameters.Group('Pulse') as pulse_block:
            pulse_should_report_to_stat = parameters.pulse_should_report_to_stat()
            fail_on_limits_exceed = pulse_parameters.fail_on_limits_exceed()
            send_email_on_limits_exceed = pulse_parameters.send_email_on_limits_exceed()
            pulse_platforms = sdk2.parameters.List(
                'Pulse platforms', required=True,
                default=('desktop', 'touch-phone', 'touch-pad'),
            )
            new_pulse_platforms = sdk2.parameters.JSON(
                'Pulse platforms', required=True,
                default=DEFAULT_PLATFORMS,
                description=platform_description,
            )

        with sdk2.parameters.Group('Blockstat') as blockstat_block:
            blockstat_platforms = sdk2.parameters.List(
                'Blockstat platforms',
                default=('desktop', 'touch-phone', 'touch-pad'),
                required=True,
                description='Список платформ, для которых нужно проверять логи blockstat',
            )

        assessors = parameters.AssessorsParameters

        manual_test_runs = parameters.ManualTestRunsParameters

        yappy_beta = parameters.DeployYappyParameters

    manual_run_task_type = SandboxCiFijiManualTestRun

    github_context = u'[Sandbox CI] Автосборка'
    output_parameters = ('is_subtasks_ready', 'is_artifacts_ready', 'static_url', 'static_origin', 'static_path',)
    report_description = 'build'

    def get_common_resource_attrs(self):
        return dict(
            super(SandboxCiFiji, self).get_common_resource_attrs(),
            coverage=self.is_coverage_enabled()
        )

    def prepare_sources(self):
        super(SandboxCiFiji, self).prepare_sources()

        if self.Parameters.project_build_context == 'pull-request':
            log_attrs = []
            for service in self.Parameters.services:
                log_attrs += [
                    dict(platform=p, project_service=s)
                    for p in self.Parameters.new_hermione_platforms[service]
                    for s in [service]
                ],

            self.selective_coverage.create_diff_lines(
                commit=self.Parameters.project_github_commit,
                log_attrs=log_attrs,
            )

    def on_prepare(self):
        super(SandboxCiFiji, self).on_prepare()

        if self.Parameters.project_build_context == 'release_test':
            self.wait_release_task()

    # Gold Crutch INFRADUTY-1919
    def wait_release_task(self):
        with self.memoize_stage.wait_tasks:
            input_parameters = dict(
                project_github_owner=self.Parameters.project_github_owner,
                project_github_repo=self.Parameters.project_github_repo,
                project_git_base_ref=self.Parameters.project_git_base_ref,
                project_git_base_commit=self.Parameters.project_git_base_commit,
                project_build_context='release',
            )

            task = sdk2.Task.find(
                type=self.type,
                input_parameters=input_parameters,
            ).order(-sdk2.Task.id).first()

            self.set_info('Will wait for task: <a href="https://sandbox.yandex-team.ru/task/{id}/view">{id}</a>'.format(id=task.id), do_escape=False)

            raise sdk2.WaitTask(task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def on_enqueue(self):
        super(SandboxCiFiji, self).on_enqueue()

        if self.is_dev:
            self.acquire_dev_build_semaphore()

        if self.is_release:
            self.create_nanny_testing_release_semaphore()

    def declare_subtasks(self):
        if self.is_coverage_enabled():
            return self.create_coverage_subtasks()

        subtasks = []

        if self.should_collect_coverage():
            subtasks.append(self.create_self_subtask_with_coverage_enabled())

        # wait for both static and dynamic packages to be ready
        deploy_wait_parameters = {self.id: 'is_static_uploaded,is_artifacts_ready'}
        deploy_yappy_subtask = self.create_deploy_yappy_subtask(wait_output_parameters=deploy_wait_parameters)

        # FIXME: FEI-14826
        if deploy_yappy_subtask:
            create_ab_wait_output_parameters = {deploy_yappy_subtask.id: 'beta_slot'}
        else:
            create_ab_wait_output_parameters = {}

        create_ab_experiment = self.create_ab_experiment_testid(wait_output_parameters=create_ab_wait_output_parameters)
        linters_subtask = self.create_linters_subtask()

        subtasks += [
            linters_subtask,
            deploy_yappy_subtask,
            create_ab_experiment,
            self.create_palmsync_validate_subtask(),
            self.create_expflags_subtask(),
        ]

        issues_info = self.startrek.get_issues_info()
        issue_keys = issues_info['keys']
        if self.config.is_enabled('tests', 'skip-validator'):
            subtasks.append(self.create_skip_validator_subtask(issue_keys=issue_keys))

        unit_tasks = self.create_unit_subtasks(wait_output_parameters=self.static_output_parameters_to_wait)
        if unit_tasks:
            subtasks += unit_tasks

        test_tasks = []
        wait_tasks = []

        beta_domain = 'https://renderer-{}-{}.hamster.yandex.ru/'.format(
            self.project_name,
            self.Parameters.beta_suffix
        )

        for service in self.Parameters.services:
            for platform in self.Parameters.new_hermione_platforms[service]:
                hermione_task = self.create_hermione_subtask(
                    service=service,
                    platform=platform,
                    wait_output_parameters=self.static_output_parameters_to_wait,
                    issue_keys=issue_keys,
                    beta_domain=beta_domain,
                )
                if hermione_task:
                    test_tasks.append(hermione_task)
                    wait_tasks.append(hermione_task)

                hermione_e2e_task = self.create_hermione_e2e_subtask(
                    service=service,
                    platform=platform,
                    wait_tasks=deploy_yappy_subtask,
                    issue_keys=issue_keys,
                    hermione_base_url=beta_domain,
                    beta_domain=beta_domain,
                )
                if hermione_e2e_task:
                    wait_tasks.append(hermione_e2e_task)
                    test_tasks.append(hermione_e2e_task)

            if test_tasks:
                test_cost_reporter = self.create_test_cost_reporter_subtask(
                    service=service,
                    wait_output_parameters=self.static_output_parameters_to_wait
                )
                subtasks.append(test_cost_reporter)

        subtasks += test_tasks

        if self.config.is_enabled('tests', 'pulse'):
            subtasks += create_pulse_subtasks(
                master_task=self,
                platforms={
                    service: platforms
                    for service, platforms in self.Parameters.new_pulse_platforms.iteritems()
                    if service in self.Parameters.services
                },
                shoot_dynamic=True,
                shoot_static=True,
                use_assets_json=True,
                use_assets_json_v2=True,
                base_static_resource_id=self._find_pulse_base_static_resource_id(),
                waitable=(not self.Parameters.is_release),
                pr_number=self.pr_number,
                notifications=self.build_release_critical_notifications(mailing_list=MailingLists.PULSE_MON),
            )

        if self.Parameters.project_build_context == 'dev':
            palmsync_task = self.create_palmsync_synchronize_subtask()
            if palmsync_task:
                subtasks.append(palmsync_task)
                subtasks.append(self.create_testids_for_flags_subtask(wait_tasks=palmsync_task))

        manual_run_task = self.create_manual_run_subtask(
            issues_info=issues_info,
            palmsync_base_url=beta_domain,
            skip_make_drafts=True,
            skip_make_runs=True,
            notifications=self.build_release_critical_notifications(),
        )

        if create_ab_experiment:
            wait_tasks.append(create_ab_experiment)

        if manual_run_task:
            subtasks.append(manual_run_task)
            subtasks.append(self.create_testids_for_flags_subtask(wait_tasks=manual_run_task))

        nanny_ensure_testing_deploy_task = self.create_nanny_ensure_testing_deploy_subtask(wait_tasks=[self])
        subtasks.append(nanny_ensure_testing_deploy_task)

        testpalm_suite_runner = self.create_testpalm_suite_runner_subtask(wait_tasks=[
            manual_run_task,
            nanny_ensure_testing_deploy_task,
        ])

        subtasks.append(testpalm_suite_runner)

        if self.Parameters.blockstat_tests:
            templates = self.get_registered_artifact_id('fiji-micropackage')

            for platform in self.Parameters.blockstat_platforms:
                blockstat_test_subtask = self.create_blockstat_test_subtask(
                    task_type=SandboxCiBlockstatTest,
                    platform=platform,
                    templates=templates,
                )
                subtasks.append(blockstat_test_subtask)

        subtasks.append(self.create_test_changes_collector_subtask())

        return subtasks

    def create_build_base_commit_subtask(self, **params):
        """
        Создаёт задачу сборки для базового коммита.

        :return: Инстанс задачи сборки
        :rtype: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
        """
        is_dev_build = self.Parameters.project_build_context == 'dev'

        if is_dev_build:
            if self.Parameters.use_arc:
                # Для выполнения в arc в базовый таск сборки нужно передавать arc_ref, потому что
                # если передать в project_git_base_commit значение project_range_base_hash
                # мы получим такой сценарий:
                # * сборка считает, что мы собираем проект в arc (что правда)
                # * сборка получает на вход project_git_base_commit, закономерно считая его гитовым
                # * на самом деле project_range_base_hash — это sha от предыдущего аркового коммита
                # * сборка не может найти базовый коммит (считает арковый гитовым)
                return self.create_build_subtask(
                    project_git_base_ref=self.Parameters.project_git_base_ref,
                    arc_ref=self.Parameters.project_range_base_hash,
                    **params
                )

            return self.create_build_subtask(
                project_git_base_ref=self.Parameters.project_git_base_ref,
                project_git_base_commit=self.Parameters.project_range_base_hash,
                **params
            )

        return super(SandboxCiFiji, self).create_build_base_commit_subtask(**params)

    @skip_subtask(SandboxCiFijiBuild, u'Не менялся код для данной задачи')
    def create_build_subtask(self, **params):
        return self.meta.declare_subtask(
            task_type=SandboxCiFijiBuild,
            project_tree_hash=None,
            **params
        )

    @skip_subtask(SandboxCiYappyDeploy, u'Не менялся код для данной задачи')
    def create_deploy_yappy_subtask(self, wait_tasks=None, wait_output_parameters={}):
        if not self.config.is_enabled('deploy', 'yappy'):
            return self.meta.skip_step(
                github_context=SandboxCiYappyDeploy.github_context,
                description=u'Деплой в Yappy отключен',
                label='dynamic_deploy',
            )

        return self.meta.declare_subtask(
            task_type=SandboxCiYappyDeploy,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            build_artifacts_resources=[self.get_registered_artifact_id('fiji')],
            templates_resources_ids=[self.get_registered_artifact_id('fiji-micropackage')],
            suffix=self.Parameters.beta_suffix,
            keep_forever=self.Parameters.keep_beta_forever,
            waitable=bool(self.config.get_deep_value(['deploy', 'yappy', 'waitable'], True)),
            webhook_urls=self.Parameters.deploy_webhook_urls,
            do_find_dev_beta_slot=self.Parameters.do_find_dev_beta_slot,
            wait_output_parameters=wait_output_parameters,
        )

    @skip_subtask(SandboxCiAbExperiments, u'Не менялся код для данной задачи')
    def create_ab_experiment_testid(self, wait_tasks=None, wait_output_parameters={}):
        domain_suffix = '{}-{}'.format(self.project_name, self.Parameters.beta_suffix)

        if self.config.is_enabled('deploy', 'ab_experiment'):
            return self.meta.declare_subtask(
                task_type=SandboxCiAbExperiments,
                wait_tasks=wait_tasks,
                description=self.Parameters.description,
                webhook_urls=self.Parameters.ab_experiments_webhook_urls,
                build_artifacts_resources=[
                    self.get_registered_artifact_id('fiji'),
                ],
                config_path=self.Parameters.ab_experiments_config_path,
                experiment_title=domain_suffix,
                experiment_domain_prefix=domain_suffix,
                wait_output_parameters=wait_output_parameters,
                use_slot_from_yappy=True,
            )

        return self.meta.skip_step(
            github_context=SandboxCiAbExperiments.github_context,
            description=u'Создание test-id эксперимента отключено в Genisys',
            label='ab_experiment_testid',
        )

    def create_hermione_subtask(self, service, platform, task_type=SandboxCiFijiHermione, wait_tasks=None, **params):
        return self.create_test_subtask(
            family='hermione',
            service=service,
            platform=platform,
            task_type=task_type,
            build_artifacts_resources=[
                self.get_registered_artifact_id('fiji'),
            ],
            wait_tasks=wait_tasks,
            reusable=True,
            waitable=bool(self.config.get_deep_value(['tests', 'hermione', 'waitable'], True)),
            selective_run=bool(self.config.get_deep_value(['tests', 'hermione', 'selective'], False)),
            html_reporter_use_sqlite=bool(self.config.get_deep_value(['tests', 'hermione', 'html_reporter_use_sqlite'], False)),
            **params
        )

    def create_hermione_e2e_subtask(self, service, platform, wait_tasks=None, **params):
        return self.create_test_subtask(
            family='hermione_e2e',
            service=service,
            platform=platform,
            wait_tasks=wait_tasks,
            task_type=SandboxCiFijiHermioneE2E,
            build_artifacts_resources=[self.get_registered_artifact_id('fiji')],
            reusable=True,
            project_name=self.project_name,
            waitable=bool(self.config.get_deep_value(['tests', 'hermione_e2e', 'waitable'], True)),
            html_reporter_use_sqlite=bool(self.config.get_deep_value(['tests', 'hermione_e2e', 'html_reporter_use_sqlite'], False)),
            **params
        )

    def create_test_subtask(self, family, task_type, service, platform, wait_tasks=None, reusable=False, **params):
        conf = self.project_conf.get('tests', {}).get(family, {})
        test_label = '{}.{}.{}'.format(family, service, platform)

        enabled = bool(conf.get('enabled', False))
        if not enabled:
            return self.meta.skip_step(
                github_context=task_type.format_github_context('{}/{}'.format(service, platform)),
                description=u'{}-тесты отключены'.format(family),
                label=test_label,
            )

        if self.need_to_skip_check(test_label):
            return self.meta.skip_step(
                github_context=task_type.format_github_context('{}/{}'.format(service, platform)),
                description=u'Не менялся код для данной платформы сервиса',
                label=test_label,
            )

        return self.meta.declare_subtask(
            task_type=task_type,
            wait_tasks=wait_tasks,
            description=self.get_test_subtask_description(platform, service),
            service=service,
            platform=platform,
            reusable=reusable,
            ref=self.ref,
            additional_tags=[platform],
            **params
        )

    @skip_subtask(SandboxCiFijiLinters, u'Не менялся код для данной проверки')
    def create_linters_subtask(self, wait_tasks=None):
        if self.config.is_enabled('tests', 'linters'):
            return self.meta.declare_subtask(
                task_type=SandboxCiFijiLinters,
                wait_tasks=wait_tasks,
                description=self.Parameters.description,
                project_git_base_ref=self.Parameters.project_git_base_ref,
                project_git_base_commit=self.Parameters.project_git_base_commit,
                project_git_merge_ref=self.Parameters.project_git_merge_ref,
                project_git_merge_commit=self.Parameters.project_git_merge_commit,
            )

    @skip_subtask(SandboxCiPalmsyncValidate, u'Не менялся код для данного инструмента')
    def create_palmsync_validate_subtask(self):
        return self.meta.declare_subtask(
            task_type=SandboxCiPalmsyncValidate,
            description=self.Parameters.description,
            build_artifacts_resources=[
                self.get_registered_artifact_id('fiji'),
            ],
            project_name=self.project_name,
            reusable=True,
            notifications=self.build_release_critical_notifications(),
        )

    def create_palmsync_synchronize_subtask(self):
        enabled = bool(self.project_conf.get('deploy', {}).get('testpalm', {}).get('enabled', False))

        if not enabled:
            return None

        return self.meta.declare_subtask(
            task_type=SandboxCiPalmsyncSynchronize,
            description=self.Parameters.description,
            build_artifacts_resources=[
                self.get_registered_artifact_id('fiji'),
            ],
            project_name=self.project_name,
        )

    def create_unit_subtasks(self, wait_tasks=None, wait_output_parameters={}):
        unit_tasks = []

        if self.config.is_enabled('tests', 'unit'):
            unit_types = {
                'unit-client': SandboxCiFijiUnitClient,
                'unit-priv': SandboxCiFijiUnitPriv,
                'unit-tmpl': SandboxCiFijiUnitTmpl,
                'unit-spec': SandboxCiFijiUnitSpec,
            }

            for task_label, task_type in unit_types.iteritems():
                if self.need_to_skip_check(task_label):
                    self.meta.skip_step(
                        github_context=task_type.format_github_context(task_type.unit_type),
                        description=u'Не менялся код для данной проверки',
                        label=task_label,
                    )
                    continue

                unit_tasks.append(self.meta.declare_subtask(
                    task_type,
                    wait_tasks,
                    description=self.Parameters.description,
                    build_artifacts_resources=[self.get_registered_artifact_id('fiji')],
                    reusable=True,
                    waitable=bool(self.config.get_deep_value(['tests', task_label, 'waitable'], True)),
                    ref=self.ref,
                    wait_output_parameters=wait_output_parameters,
                ))

        return unit_tasks

    @skip_subtask(SandboxCiSkipValidator, u'Не менялся код для данного инструмента')
    def create_skip_validator_subtask(self, wait_tasks=None, **params):
        return self.meta.declare_subtask(
            task_type=SandboxCiSkipValidator,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            build_artifacts_resources=[self.get_registered_artifact_id('fiji')],
            project_name=self.project_name,
            **params
        )

    def create_testids_for_flags_subtask(self, wait_tasks):
        if not self.Parameters.testids_for_flags:
            return None

        config_path = self.working_path('fiji/.config/testids-for-flags.json')
        config_exists = config_path.is_file()
        config_path = str(config_path)

        config = None

        if config_exists:
            logging.debug('testids-for-flags config exists, reading from {path}'.format(path=config_path))
            with open(config_path, 'r') as f:
                config = json.load(f)
                logging.debug('testids-for-flags config readed: {config}'.format(config=config))
        else:
            logging.debug('testids-for-flags config does not found by path {path}'.format(path=config_path))

        return self.meta.declare_subtask(
            task_type=SandboxCiTestidsForFlags,
            wait_tasks=wait_tasks,
            waitable=bool(self.config.get_deep_value(['tests', 'testids_for_flags', 'waitable'], False)),
            description=self.Parameters.description,
            project_name=self.project_name,
            testpalm_project_name=self.Parameters.testpalm_project_name,
            config=config,
        )

    def create_nanny_testing_release_semaphore(self):
        with self.memoize_stage.nanny_testing_semaphore:
            self.set_semaphore(
                capacity=1,
                name=self.nanny_testing_release_semaphore_name,
                release=(ctt.Status.RELEASED, ctt.Status.Group.BREAK, ctt.Status.FAILURE),
            )

    def create_nanny_ensure_testing_deploy_subtask(self, wait_tasks=None):
        if not self.is_release:
            return None

        issue_key = None
        if hasattr(self.Parameters, 'send_comment_to_issue'):
            issue_key = self.Parameters.send_comment_to_issue

        return self.meta.declare_subtask(
            task_type=SandboxCiNannyEnsureDeploy,
            release_id=self.nanny_testing_release_id,
            semaphore_name=self.nanny_testing_release_semaphore_name,
            poll_interval=300,
            attempts=12,
            failed_statuses=["DEPLOY_FAILED", "CANCELLED", "ROLLED_BACK"],
            issue_key=issue_key,
            waitable=False,
            wait_tasks=wait_tasks,
            notifications=self.build_release_critical_notifications(),
        )

    @property
    def nanny_testing_release_semaphore_name(self):
        return '{name}_{id}_testing_resource_release'.format(name=self.project_name, id=self.id)

    @property
    def nanny_testing_release_id(self):
        # Nanny генерирует id релиза в таком формате:
        # '{nanny_release_type}-{sandbox_task_id}-{sandbox_task_release_status}'
        return 'SANDBOX_RELEASE-{id}-TESTING'.format(id=self.id)

    def create_testpalm_suite_runner_subtask(self, wait_tasks):
        should_start = should_start_testpalm_suite_runner_task(
            is_release=self.is_release,
            force_run=self.Parameters.force_assessors_run,
            config_path=self.Parameters.assessors_config_path,
            testpalm_project_suffix=self.testpalm_project_suffix,
            testpalm_base_project_name=self.Parameters.testpalm_base_project_name,
            ticket_id=self.Parameters.send_comment_to_issue,
        )

        if not should_start:
            return None

        return self.meta.declare_subtask(
            task_type=SandboxCiTestpalmSuiteRunner,
            wait_tasks=wait_tasks,
            waitable=False,
            description=self.Parameters.description,
            build_artifacts_resources=[
                self.get_registered_artifact_id('fiji'),
            ],
            config_path=self.Parameters.assessors_config_path,
            testpalm_project_suffix=self.testpalm_project_suffix,
            testpalm_base_project_name=self.Parameters.testpalm_base_project_name,
            notifications=self.build_release_critical_notifications(),
            ticket_id=self.Parameters.send_comment_to_issue,
        )

    def create_test_cost_reporter_subtask(self, service, wait_tasks=None, wait_output_parameters={}):
        if not self.is_release:
            return None

        return self.meta.declare_subtask(
            task_type=SandboxCiTestCostReporter,
            waitable=False,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            target_task_id=self.id,
            release_version=self.Parameters.project_git_base_ref,
            required_task_type='SANDBOX_CI_FIJI',
            service=service,
            mode='release',
            wait_output_parameters=wait_output_parameters,
        )

    def is_coverage_enabled(self):
        return self.environ.get('COVERAGE') == 'true'

    def should_collect_coverage(self):
        return bool(self.config.get_deep_value(['tests', 'hermione', 'collect_coverage'], False))

    def create_self_subtask_with_coverage_enabled(self):
        task_parameters = dict(self.Parameters)

        # TODO: sandbox не позволяет изменять output-параметры.
        for out_param in self.output_parameters:
            task_parameters.pop(out_param, None)

        task_parameters.update(
            report_github_statuses=False,
            send_statistic=False,
            notifications=[],
            environ=dict(
                task_parameters.get('environ', dict()),
                COVERAGE='true',
            ),
        )

        return self.meta.declare_subtask(
            task_type=self.type,
            waitable=False,
            **task_parameters
        )

    def create_expflags_subtask(self, wait_tasks=None):
        is_upload = self.Parameters.upload_created_exp_flags
        is_update = self.Parameters.update_changed_exp_flags

        if not is_upload and not is_update:
            return self.meta.skip_step(
                github_context=SandboxCiExpflags.github_context,
                description=u'Синхронизация флагов не требуется',
                label='expflags.sync',
            )

        service = self.config.get_deep_value(['expflags', 'service'], [])
        component = self.config.get_deep_value(['expflags', 'component'], [])
        pattern = self.config.get_deep_value(['expflags', 'pattern'])
        source = 'fiji'
        validation = {'validateValuesType': True}

        return self.meta.declare_subtask(
            task_type=SandboxCiExpflags,
            wait_tasks=wait_tasks,
            waitable=True,
            description=self.Parameters.description,
            project_name=self.project_name,
            expflags_resource=self.get_registered_artifact_id('expflags'),
            expflags_pattern=pattern,
            expflags_options={'validation': validation},
            expflags_tags=['{}-pr-{}'.format(self.project_name, self.pr_number)],
            upload_exp_flags=is_upload,
            update_exp_flags=is_update,
            flag_template={'service': service, 'component': component, 'source': source},
            source=source,
        )

    def create_coverage_subtasks(self):
        subtasks = []

        for service in self.Parameters.services:
            # Запускаем только hermione тесты.
            for platform in self.Parameters.new_hermione_platforms[service]:
                hermione_task = self.create_hermione_subtask(
                    service=service,
                    platform=platform,
                    wait_output_parameters=self.static_output_parameters_to_wait,
                    report_github_statuses=False,
                    send_statistic=False,
                    notifications=[],
                )
                if hermione_task:
                    subtasks.append(hermione_task)

        return subtasks

    def create_test_changes_collector_subtask(self):
        if not self.config.is_enabled('tests', 'changes_collector'):
            return None

        issue_keys = get_arc_last_merge_commit_issues(self.project_dir) if self.Parameters.use_arc else git.get_git_last_merge_commit_issues(self.project_dir)

        return self.meta.create_subtask(
            task_type=SandboxCiTestChangesCollector,
            waitable=False,
            build_artifacts_resources=[self.get_registered_artifact_id('fiji')],
            test_types=['hermione', 'hermione-e2e'],
            test_read_environs=map(lambda p: { 'PLATFORM': p }, self.Parameters.hermione_platforms),
            issue_keys=issue_keys,
        )

    def _find_pulse_base_static_resource_id(self):
        try:
            return ResourceTemplatesFinder(self).base_static_resource_id(coverage=False)
        except ResourceNotFoundException:
            logging.debug('Base static resource without coverage not found, fallback to default pulse behaviour')
            return None

    def get_nanny_webhook_urls(self, additional_parameters):
        release_type = additional_parameters['release_status']

        if release_type == 'testing':
            stable_host = 'https://yandex.ru'
            beta_host = {
                'images': 'https://imgstemplates.hamster.yandex.ru',
                'video': 'https://videotemplates.hamster.yandex.ru',
            }.get(self.sub_project, stable_host)

            return [
                'https://webhook-handler.si.yandex-team.ru/v1/sandbox/release-check-static?{}'.format(
                    urllib.urlencode({
                        'genisys-section': 'search-interfaces.releases.empty',
                        'beta-host': beta_host,
                        'stable-host': stable_host,
                    })
                )
            ]

        if release_type == 'stable':
            return ['https://webhook-handler.si.yandex-team.ru/v1/nanny/release-request-tickets-status-change']

        return []

    def get_nanny_webhook_type(self, additional_parameters):
        return 'RELEASE_WITH_TICKETS'

    def get_nanny_duplicate_policy_type(self, additional_parameters):
        return 'IGNORE'

    def on_release(self, additional_parameters):
        """
        :param additional_parameters: введенные данные из формы релизов
        :type additional_parameters: dict
        """
        release_type = additional_parameters['release_status'].lower()

        tags = set(map(lambda tag: tag.lower(), self.Parameters.tags) + [release_type])

        logging.debug('Releasing task in {release_type} with parameters: {parameters} and tags: {tags}'.format(
            release_type=release_type,
            parameters=additional_parameters,
            tags=tags,
        ))

        self.Parameters.tags = list(tags)

        SandboxCiFijiBuild.on_release(self, additional_parameters)
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
